/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2012 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.capabilities.internal;

import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import org.osgi.framework.BundleContext;
import com.openexchange.capabilities.Capability;
import com.openexchange.capabilities.CapabilityChecker;
import com.openexchange.capabilities.CapabilityService;
import com.openexchange.config.ConfigurationService;
import com.openexchange.exception.OXException;
import com.openexchange.server.ServiceLookup;
import com.openexchange.tools.session.ServerSession;

/**
 * {@link CapabilityServiceImpl}
 *
 * @author <a href="mailto:francisco.laguna@open-xchange.com">Francisco Laguna</a>
 */
public class CapabilityServiceImpl implements CapabilityService {

    private static final Object PRESENT = new Object();

    private final ConcurrentMap<String, Capability> capabilities;
    private final ConcurrentMap<String, Object> declaredCapabilities;

    private final ServiceLookup services;
    private volatile Boolean autologin;

    /**
     * Initializes a new {@link CapabilityServiceImpl}.
     *
     * @param capabilitiesActivator
     * @param context
     */
    public CapabilityServiceImpl(ServiceLookup services, BundleContext context) {
        super();
        this.services = services;
        capabilities = new ConcurrentHashMap<String, Capability>();
        declaredCapabilities = new ConcurrentHashMap<String, Object>();
    }

    private boolean autologin() {
        Boolean tmp = autologin;
        if (null == tmp) {
            synchronized (this) {
                tmp = autologin;
                if (null == tmp) {
                    tmp = Boolean.valueOf(services.getService(ConfigurationService.class).getBoolProperty("com.openexchange.sessiond.autologin", false));
                    autologin = tmp;
                }
            }
        }
        return tmp.booleanValue();
    }

    @Override
    public Set<Capability> getCapabilities(ServerSession session) throws OXException {
        Set<Capability> capabilities = new HashSet<Capability>(64);
        if (!session.isAnonymous()) {
            for (String type : session.getUserConfiguration().getExtendedPermissions()) {
                if (check(type, session)) {
                    capabilities.add(getCapability(type));
                }
            }
        }

        // What about autologin?

        boolean autologin = autologin();
        if (autologin) {
            capabilities.add(new Capability("autologin", true));
        }

        for (String cap : declaredCapabilities.keySet()) {
            if (check(cap, session)) {
                capabilities.add(getCapability(cap));
            }
        }

        return capabilities;
    }

    private boolean check(String cap, ServerSession session) throws OXException {
        for (CapabilityChecker checker : getCheckers()) {
            if (!checker.isEnabled(cap, session)) {
                return false;
            }
        }
        return true;
    }

    public Set<Capability> getAllKnownCapabilities() throws OXException {
        return new HashSet<Capability>(capabilities.values());
    }

    public Capability getCapability(String id) {
        Capability capability = capabilities.get(id);

        if (capability == null) {
            Capability existingCapability = capabilities.putIfAbsent(id, capability = new Capability(id, false));
            return existingCapability != null ? existingCapability : capability;
        }

        return capability;
    }

    @Override
    public void declareCapability(String capability) {
        declaredCapabilities.put(capability, PRESENT);
    }

    public List<CapabilityChecker> getCheckers() {
        return Collections.emptyList();
    }

}
