/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2012 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.sessiond.impl;

import java.util.concurrent.atomic.AtomicLong;
import com.openexchange.session.Session;

/**
 * {@link SessionControl} - Holds a {@link Session} instance and remembers life-cycle timestamps such as last-accessed, creation-time, etc.
 *
 * @author <a href="mailto:sebastian.kauss@open-xchange.com">Sebastian Kauss</a>
 * @author <a href="mailto:thorben.betten@open-xchange.com">Thorben Betten</a>
 */
public class SessionControl {

    /**
     * Time stamp when this session control was created.
     */
    private final long creationTime;

    /**
     * The associated session.
     */
    private final SessionImpl session;

    /**
     * Last-accessed time stamp.
     */
    private final AtomicLong lastAccessed;

    /**
     * The idle time.
     */
    private final long idleTime;

    /**
     * Initializes a new {@link SessionControl}
     *
     * @param session The stored session
     * @param idleTime The session's allowed idle time or <code>-1</code> to use default setting
     */
    public SessionControl(final SessionImpl session, final long idleTime) {
        super();
        lastAccessed = new AtomicLong(System.currentTimeMillis());
        this.session = session;
        creationTime = System.currentTimeMillis();
        this.idleTime = idleTime;
    }

    /**
     * Checks if this session control is elapsed.
     *
     * @param defaultIdleTime The default idle time
     * @return <code>true</code> if elapsed; otherwise <code>false</code>
     */
    public boolean isElapsed(final long defaultIdleTime) {
        if (idleTime < 0) {
            return lastAccessed.get() < (System.currentTimeMillis() - defaultIdleTime);
        }
        return lastAccessed.get() < (System.currentTimeMillis() - idleTime);
    }

    /**
     * Initializes a new {@link SessionControl}
     *
     * @param session The stored session
     */
    public SessionControl(final SessionImpl session) {
        this(session, -1L); // Default idle time
    }

    /**
     * Gets the idle time
     *
     * @return The idle time
     */
    public long getIdleTime() {
        return idleTime;
    }

    /**
     * Gets the last-accessed time stamp.
     *
     * @return The last-accessed time stamp
     */
    public long getLastAccessed() {
        return lastAccessed.get();
    }

    /**
     * Updates last-accessed time stamp.
     *
     * @return This control with last-accessed time stamp updated
     */
    public SessionControl touch() {
        lastAccessed.set(System.currentTimeMillis());
        return this;
    }

    /**
     * Gets the stored session
     *
     * @return The stored session
     */
    public SessionImpl getSession() {
        return session;
    }

    /**
     * Gets the creation-time time stamp
     *
     * @return The creation-time time stamp
     */
    public long getCreationTime() {
        return creationTime;
    }
}
