/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2020 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package org.quartz.service.internal;

import java.util.HashMap;
import java.util.Map;
import java.util.Properties;
import org.apache.commons.logging.Log;
import org.quartz.Scheduler;
import org.quartz.SchedulerException;
import org.quartz.SchedulerFactory;
import org.quartz.impl.StdSchedulerFactory;
import org.quartz.service.QuartzService;
import com.hazelcast.core.HazelcastInstance;
import com.openexchange.config.ConfigurationService;
import com.openexchange.exception.OXException;

/**
 * {@link QuartzServiceImpl}
 *
 * @author <a href="mailto:thorben.betten@open-xchange.com">Thorben Betten</a>
 */
public final class QuartzServiceImpl implements QuartzService {

    private static final Log LOG = com.openexchange.log.Log.loggerFor(QuartzServiceImpl.class);

    private final Map<String, Scheduler> clusteredSchedulers = new HashMap<String, Scheduler>();

    private Scheduler localScheduler = null;


    /**
     * Initializes a new {@link QuartzServiceImpl}.
     */
    public QuartzServiceImpl() {
        super();
    }

    @Override
    public Scheduler getLocalScheduler() throws OXException {
        synchronized (clusteredSchedulers) {
            if (localScheduler == null) {
                ConfigurationService config = Services.getService(ConfigurationService.class);
                boolean startLocalScheduler = config.getBoolProperty(QuartzProperties.START_LOCAL_SCHEDULER, true);
                int localThreads = config.getIntProperty(QuartzProperties.LOCAL_THREADS, 3);

                Properties localProperties = new Properties();
                localProperties.put("org.quartz.scheduler.instanceName", "OX-Local-Scheduler");
                localProperties.put("org.quartz.scheduler.rmi.export", false);
                localProperties.put("org.quartz.scheduler.rmi.proxy", false);
                localProperties.put("org.quartz.scheduler.wrapJobExecutionInUserTransaction", false);
                localProperties.put("org.quartz.threadPool.class", "org.quartz.simpl.SimpleThreadPool");
                localProperties.put("org.quartz.threadPool.threadCount", String.valueOf(localThreads));
                localProperties.put("org.quartz.threadPool.threadPriority", "5");
                localProperties.put("org.quartz.threadPool.threadsInheritContextClassLoaderOfInitializingThread", true);
                localProperties.put("org.quartz.jobStore.misfireThreshold", "60000");
                localProperties.put("org.quartz.jobStore.class", "org.quartz.simpl.RAMJobStore");
                localProperties.put("org.quartz.scheduler.jmx.export", true);

                try {
                    SchedulerFactory csf = new StdSchedulerFactory(localProperties);
                    localScheduler = csf.getScheduler();
                    if (startLocalScheduler) {
                        localScheduler.start();
                    }
                } catch (SchedulerException e) {
                    throw new OXException(e);
                }
            }
            return localScheduler;
        }
    }

    @Override
    public Scheduler getClusteredScheduler(String name, boolean start, int threads) throws OXException {
        if (name == null) {
            throw new IllegalArgumentException("Parameter 'name' must not be null!");
        }

        synchronized (clusteredSchedulers) {
            Scheduler scheduler = clusteredSchedulers.get(name);
            if (scheduler == null) {
                Properties clusteredProperties = new Properties();
                clusteredProperties.put("org.quartz.scheduler.instanceName", name);
                clusteredProperties.put(
                    "org.quartz.scheduler.instanceId",
                    Services.getService(HazelcastInstance.class).getCluster().getLocalMember().getUuid());
                clusteredProperties.put("org.quartz.scheduler.rmi.export", false);
                clusteredProperties.put("org.quartz.scheduler.rmi.proxy", false);
                clusteredProperties.put("org.quartz.scheduler.wrapJobExecutionInUserTransaction", false);
                clusteredProperties.put("org.quartz.threadPool.class", "org.quartz.simpl.SimpleThreadPool");
                clusteredProperties.put("org.quartz.threadPool.threadCount", String.valueOf(threads <= 0 ? 1 : threads));
                clusteredProperties.put("org.quartz.threadPool.threadPriority", "5");
                clusteredProperties.put("org.quartz.threadPool.threadsInheritContextClassLoaderOfInitializingThread", true);
                clusteredProperties.put("org.quartz.jobStore.misfireThreshold", "60000");
                clusteredProperties.put("org.quartz.jobStore.class", "com.openexchange.quartz.hazelcast.ImprovedHazelcastJobStore");
                clusteredProperties.put("org.quartz.scheduler.jmx.export", true);
                try {
                    SchedulerFactory csf = new StdSchedulerFactory(clusteredProperties);
                    scheduler = csf.getScheduler();
                } catch (SchedulerException e) {
                    throw new OXException(e);
                }

                clusteredSchedulers.put(name, scheduler);
            }
            try {
                if (start && !scheduler.isStarted()) {
                    scheduler.start();
                }
            } catch (SchedulerException e) {
                throw new OXException(e);
            }
            return scheduler;
        }
    }

    @Override
    public void releaseClusteredScheduler(String name) {
        if (name == null) {
            return;
        }

        synchronized (clusteredSchedulers) {
            try {
                Scheduler scheduler = clusteredSchedulers.remove(name);
                if (scheduler != null && scheduler.isStarted()) {
                    scheduler.shutdown(true);
                }
            } catch (SchedulerException e) {
                LOG.warn("Could not stop clustered scheduler '" + name + "'.", e);
            }
        }
    }

    public void shutdown() {
        synchronized (clusteredSchedulers) {
            try {
                if (localScheduler != null && localScheduler.isStarted()) {
                    localScheduler.shutdown();
                }
            } catch (SchedulerException e) {
                LOG.warn("Could not stop local scheduler.", e);
            }
            for (String name : clusteredSchedulers.keySet()) {
                Scheduler scheduler = clusteredSchedulers.get(name);
                try {
                    if (scheduler.isStarted()) {
                        scheduler.shutdown();
                    }
                } catch (SchedulerException e) {
                    LOG.warn("Could not stop clustered scheduler '" + name + "'.", e);
                }
            }
            clusteredSchedulers.clear();
        }
    }
}
