/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.weakforced;

import java.util.LinkedHashMap;
import java.util.Map;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.LockSupport;
import com.openexchange.ajax.Client;
import com.openexchange.antiabuse.AllowParameters;
import com.openexchange.antiabuse.AntiAbuseService;
import com.openexchange.antiabuse.Protocol;
import com.openexchange.antiabuse.ReportParameters;
import com.openexchange.antiabuse.ReportValue;
import com.openexchange.antiabuse.Status;
import com.openexchange.authentication.LoginExceptionCodes;
import com.openexchange.config.ConfigurationService;
import com.openexchange.exception.OXException;
import com.openexchange.java.Strings;
import com.openexchange.login.LoginRequest;
import com.openexchange.login.LoginResult;
import com.openexchange.login.listener.LoginListener;
import com.openexchange.session.Session;

/**
 * {@link WeakforcedLoginListener}
 *
 * @author <a href="mailto:thorben.betten@open-xchange.com">Thorben Betten</a>
 * @since v1.0.0
 */
public class WeakforcedLoginListener implements LoginListener {

    private static final org.slf4j.Logger LOGGER = org.slf4j.LoggerFactory.getLogger(WeakforcedLoginListener.class);

    private final AntiAbuseService weakforcedClient;
    private final String[] attributeNames;
    private final ConfigurationService configurationService;

    /**
     * Initializes a new {@link WeakforcedLoginListener}.
     */
    public WeakforcedLoginListener(ConfigurationService configurationService, AntiAbuseService weakforcedClient, String[] attributeNames) {
        super();
        this.configurationService = configurationService;
        this.weakforcedClient = weakforcedClient;
        this.attributeNames = attributeNames;
    }

    private boolean isWeakforcedError(OXException e) {
        return null != e && WeakforcedExceptionCodes.prefix().equals(e.getPrefix());
    }

    /**
     * Handles specified status
     * <ul>
     * <li>Returns immediately if status signals <code>OK</code></li>
     * <li>Throws an {@link OXException} if status signals <code>BLOCKED</code></li>
     * <li>Waits for <code>X</code> seconds if status signals a positive integer</li>
     * </ul>
     *
     * @param status The status to handle
     * @param login The login string in order to report a possible blocked authentication attempt
     * @param password The password in order to report a possible blocked authentication attempt
     * @param remoteAddress The remote address in order to report a possible blocked authentication attempt
     * @param userAgent The User-Agent string
     * @param protocol The protocol
     * @throws OXException If status signals <code>BLOCKED</code>
     */
    private void handleStatus(Status status, String login, String password, String remoteAddress, String userAgent, Protocol protocol) throws OXException {
        if (null == status || status.isOk()) {
            return;
        }
        if (status.isBlocked()) {
            // Do the report
            performReportSafe(ReportValue.DENIED_BY_SERVICE, login, password, remoteAddress, userAgent, protocol);

            // Throw exception since Weakforced denied login
            boolean showErrorMessage = this.configurationService.getBoolProperty(WeakforcedProperties.INCLUDE_ERROR, true);
            if (showErrorMessage) {
                String message = status.getMessage();
                String messageSuffix = this.configurationService.getProperty(WeakforcedProperties.ERROR_SUFFIX);
                if (Strings.isNotEmpty(messageSuffix)) {
                    message = Strings.isNotEmpty(message) ? (message + " " + messageSuffix) : messageSuffix;
                }
                if (Strings.isNotEmpty(message)) {
                    throw LoginExceptionCodes.LOGIN_DENIED_WITH_MESSAGE.create(message);
                }
            }
            throw LoginExceptionCodes.LOGIN_DENIED.create();
        }
        int waitSeconds = status.getWaitSeconds();
        if (waitSeconds > 0) {
            LockSupport.parkNanos(TimeUnit.SECONDS.toNanos(waitSeconds));
        }
    }

    private boolean isAnonymous(String login) {
        return Strings.isEmpty(login);
    }

    private Protocol protocolFor(LoginRequest request) {
        String client = request.getClient();
        if (client != null && Strings.asciiLowerCase(client).startsWith(Client.MOBILE_API_FACADE.getClientId())) {
            return Protocol.newInstance("mobileapi", request.isSecure());
        }

        // Otherwise assume "http"
        return request.isSecure() ? Protocol.HTTPS : Protocol.HTTP;
    }

    private Status performAllowSafe(String login, String password, String remoteAddress, Map<String, String> attributes, String userAgent, Protocol protocol) {
        try {
            AllowParameters.Builder parameters = AllowParameters.builder()
                .login(login)
                .password(password)
                .remoteAddress(remoteAddress)
                .attributes(attributes)
                .userAgent(userAgent)
                .protocol(protocol);
            return weakforcedClient.allow(parameters.build());
        } catch (OXException e) {
            // Request failed...
            LOGGER.warn("'allow' request to Weakforced failed", e);
            return null;
        }
    }

    private void performReportSafe(ReportValue value, String login, String password, String remoteAddress, String userAgent, Protocol protocol) {
        try {
            ReportParameters.Builder parameters = ReportParameters.builder()
                .reportValue(value)
                .login(login)
                .password(password)
                .remoteAddress(remoteAddress)
                .userAgent(userAgent)
                .protocol(protocol);
            weakforcedClient.report(parameters.build());
        } catch (OXException e) {
            // Request failed...
            LOGGER.warn("'report' request to Weakforced failed", e);
        }
    }

    private String getPassword(LoginRequest request) {
        String password = request.getPassword();
        if (Strings.isEmpty(password)) {
            password = "0000";
        }
        return password;
    }

    @Override
    public void onBeforeAuthentication(LoginRequest request, Map<String, Object> properties) throws OXException {
        String login = request.getLogin();
        if (isAnonymous(login)) {
            LOGGER.debug("Denying 'allow' request to Weakforced end-point since login is absent or empty");
            return;
        }

        String password = getPassword(request);
        String userAgent = request.getUserAgent();
        Protocol protocol =  protocolFor(request);
        String remoteAddress = request.getClientIP();

        Map<String, String> attributes = null;
        String client = request.getClient();
        if (!Strings.isEmpty(client)) {
            attributes = new LinkedHashMap<String, String>(2);
            attributes.put("clientID", client);
        }

        Status status = performAllowSafe(login, password, remoteAddress, attributes, userAgent, protocol);
        handleStatus(status, login, password, remoteAddress, userAgent, protocol);
    }

    @Override
    public void onSucceededAuthentication(LoginResult result) throws OXException {
        LoginRequest request = result.getRequest();
        String login = request.getLogin();
        if (isAnonymous(login)) {
            LOGGER.debug("Denying 'report' request to Weakforced end-point on succeeded authentication since login is absent or empty");
            return;
        }

        String password = getPassword(request);
        String remoteAddress = request.getClientIP();
        String userAgent = request.getUserAgent();
        Protocol protocol = protocolFor(request);

        // The post-authentication 'allow' hook is to enable the client to send data to weakforced from the provisioning system (e.g. LDAP)
        {
            String[] attributeNames = this.attributeNames;
            if (null != attributeNames) {
                Session session = result.getSession();
                Map<String, String> attributes = null;
                for (String attributeName : attributeNames) {
                    Object attribute = session.getParameter(attributeName);
                    if (attribute instanceof String) {
                        if (null == attributes) {
                            attributes = new LinkedHashMap<String, String>();
                        }
                        attributes.put(attributeName, attribute.toString());
                    }
                }

                if (null != attributes) {
                    // There are some attribute; hence do the 'allow' to propagate them
                    String client = request.getClient();
                    if (!Strings.isEmpty(client)) {
                        attributes.put("clientID", client);
                    }

                    // Do the allow
                    Status status = performAllowSafe(login, password, remoteAddress, attributes, userAgent, protocol);
                    handleStatus(status, login, password, remoteAddress, userAgent, protocol);
                }
            }
        }

        // Do the report
        performReportSafe(ReportValue.SUCCESS, login, password, remoteAddress, userAgent, protocol);
    }

    @Override
    public void onFailedAuthentication(LoginRequest request, Map<String, Object> properties, OXException e) throws OXException {
        if (isWeakforcedError(e)) {
            return;
        }

        String login = request.getLogin();
        if (isAnonymous(login)) {
            LOGGER.debug("Denying 'report' request to Weakforced end-point on failed authentication since login is absent or empty");
            return;
        }

        String password = getPassword(request);
        String remoteAddress = request.getClientIP();
        String userAgent = request.getUserAgent();
        Protocol protocol = protocolFor(request);

        // Do the report
        performReportSafe(ReportValue.FAILURE, login, password, remoteAddress, userAgent, protocol);
    }

    @Override
    public void onRedirectedAuthentication(LoginRequest request, Map<String, Object> properties, OXException e) throws OXException {
        // Nothing to do
    }

}
