/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.weakforced;

import java.io.IOException;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.EnumMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicReference;
import org.apache.http.HttpEntity;
import org.apache.http.HttpResponse;
import org.apache.http.client.HttpClient;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpRequestBase;
import org.apache.http.util.EntityUtils;
import com.openexchange.config.ConfigurationService;
import com.openexchange.exception.OXException;
import com.openexchange.java.Strings;
import com.openexchange.rest.client.endpointpool.Endpoint;
import com.openexchange.rest.client.endpointpool.EndpointAvailableStrategy;
import com.openexchange.rest.client.endpointpool.EndpointManager;
import com.openexchange.rest.client.endpointpool.EndpointManagerFactory;
import com.openexchange.rest.client.httpclient.HttpClientService;
import com.openexchange.rest.client.httpclient.SpecificHttpClientConfigProvider;
import com.openexchange.server.ServiceExceptionCode;
import com.openexchange.server.ServiceLookup;

/**
 * {@link WeakforcedEndpointManager}
 *
 * @author <a href="mailto:thorben.betten@open-xchange.com">Thorben Betten</a>
 * @since v1.0.0
 */
public class WeakforcedEndpointManager {

    private static final org.slf4j.Logger LOG = org.slf4j.LoggerFactory.getLogger(WeakforcedEndpointManager.class);

    private static final EndpointAvailableStrategy AVAILABILITY_STRATEGY = new EndpointAvailableStrategy() {

        @Override
        public AvailableResult isEndpointAvailable(Endpoint endpoint, HttpClient httpClient) throws OXException {
            HttpGet get = null;
            HttpResponse response = null;
            try {
                get = new HttpGet(WeakforcedClient.buildUri(new URI(endpoint.getBaseUri()), WeakforcedClient.toQueryString(mapFor("command", "ping")), null));
                response = httpClient.execute(get);
                int status = response.getStatusLine().getStatusCode();
                if (200 == status) {
                    return AvailableResult.AVAILABLE;
                }
                if (401 == status) {
                    return AvailableResult.NONE;
                }
            } catch (URISyntaxException e) {
                // ignore
                return AvailableResult.NONE;
            } catch (IOException e) {
                // ignore
            } finally {
                close(get, response);
            }

            return AvailableResult.UNAVAILABLE;
        }
    };

    /**
     * Gets a (parameters) map for specified arguments.
     *
     * @param args The arguments
     * @return The resulting map
     */
    static Map<String, String> mapFor(String... args) {
        if (null == args) {
            return null;
        }

        int length = args.length;
        if (0 == length || (length % 2) != 0) {
            return null;
        }

        Map<String, String> map = new LinkedHashMap<String, String>(length >> 1);
        for (int i = 0; i < length; i+=2) {
            map.put(args[i], args[i+1]);
        }
        return map;
    }

    /**
     * Closes the supplied HTTP request & response resources silently.
     *
     * @param request The HTTP request to reset
     * @param response The HTTP response to consume and close
     */
    static void close(HttpRequestBase request, HttpResponse response) {
        if (null != response) {
            HttpEntity entity = response.getEntity();
            if (null != entity) {
                try {
                    EntityUtils.consume(entity);
                } catch (Exception e) {
                    LOG.debug("Error consuming HTTP response entity", e);
                }
            }
        }

        if (null != request) {
            try {
                request.reset();
            } catch (Exception e) {
                LOG.debug("Error resetting HTTP request", e);
            }
        }
    }

    // -------------------------------------------------------------------------------------------------------------------------------

    private final AtomicReference<EnumMap<WeakforcedCall, ClientIdAndEndpointManager>> endpointsReference;
    private final ServiceLookup services;

    /**
     * Initializes a new {@link WeakforcedEndpointManager}.
     */
    public WeakforcedEndpointManager(ServiceLookup services) {
        super();
        this.services = services;
        endpointsReference = new AtomicReference<EnumMap<WeakforcedCall, ClientIdAndEndpointManager>>(null);
    }

    /**
     * Initializes this instance.
     *
     * @param factory The end-point factory to use
     * @param services The service look-up
     * @return The HTTP client configuration providers to register
     * @throws OXException If initialization fails
     */
    public List<SpecificHttpClientConfigProvider> init(EndpointManagerFactory factory, ServiceLookup services) throws OXException {
        org.slf4j.Logger logger = org.slf4j.LoggerFactory.getLogger(WeakforcedEndpointManager.class);

        // Get needed service
        ConfigurationService configService = services.getServiceSafe(ConfigurationService.class);

        // Check credentials
        String authLogin = configService.getProperty(WeakforcedProperties.AUTH_LOGIN);
        String authPassword = configService.getProperty(WeakforcedProperties.AUTH_PASSWORD);
        if (Strings.isEmpty(authLogin) || Strings.isEmpty(authPassword)) {
            throw OXException.general("Denied initialization due to unset Basic-Auth configuration. Please set properties 'com.openexchange.weakforced.basic-auth.login' and 'com.openexchange.weakforced.basic-auth.password' appropriately.");
        }
        authLogin = authLogin.trim();
        authPassword = authPassword.trim();

        // Initialize collections
        WeakforcedCall[] weakforcedCalls = WeakforcedCall.values();
        EnumMap<WeakforcedCall, ClientIdAndEndpointManager> endpoints = new EnumMap<WeakforcedCall, ClientIdAndEndpointManager>(WeakforcedCall.class);
        List<SpecificHttpClientConfigProvider> configProviders = new ArrayList<>(weakforcedCalls.length);

        // Parse configuration
        String fallBackName = WeakforcedProperties.ENDPOINTS;
        EndpointManager fallBackEntry = null;
        for (WeakforcedCall call : weakforcedCalls) {
            String propName = WeakforcedProperties.ENDPOINTS + "." + call.toString();
            String endPoints = configService.getProperty(propName);
            if (Strings.isEmpty(endPoints)) {
                String httpClientId = "weakforced";
                if (null == fallBackEntry) {
                    endPoints = configService.getProperty(fallBackName);
                    if (Strings.isEmpty(endPoints)) {
                        // No end-point
                        logger.info("No Weakforced end-points defined for call {} via property {}", call.getName(), propName);
                        return Collections.emptyList();
                    }

                    // Parse end-point list
                    List<String> l = Arrays.asList(Strings.splitByComma(endPoints.trim()));

                    configProviders.add(initConfigProviderFor(httpClientId, l.size(), authLogin, authPassword, new StringBuilder(fallBackName).append('.'), configService, services));
                    fallBackEntry = endpointManagerFor(l, httpClientId, factory);
                }
                endpoints.put(call, new ClientIdAndEndpointManager(httpClientId, fallBackEntry));
            } else {
                String httpClientId = WeakforcedHttpClientConfig.generateClientIdFor(call);

                // Parse end-point list
                List<String> l = Arrays.asList(Strings.splitByComma(endPoints.trim()));

                configProviders.add(initConfigProviderFor(httpClientId, l.size(), authLogin, authPassword, new StringBuilder(propName).append('.'), configService, services));
                endpoints.put(call, new ClientIdAndEndpointManager(httpClientId, endpointManagerFor(l, httpClientId, factory)));
            }
        }

        endpointsReference.set(endpoints);
        return configProviders;
    }

    private static EndpointManager endpointManagerFor(List<String> endPoints, String httpClientId, EndpointManagerFactory factory) throws OXException {
        // Setup & return end-point manager
        return factory.createEndpointManager(endPoints, httpClientId, AVAILABILITY_STRATEGY, 60, TimeUnit.SECONDS);
    }

    private static SpecificHttpClientConfigProvider initConfigProviderFor(String httpClientId,int numberOfEndpoints, String authLogin, String authPassword, StringBuilder propPrefix, ConfigurationService configService, ServiceLookup services) {
        // Read properties for HTTP connections/pooling
        int resetLen = propPrefix.length();
        int totalConnections = configService.getIntProperty(propPrefix.append("totalConnections").toString(), 100);
        propPrefix.setLength(resetLen);
        int maxConnectionsPerRoute = configService.getIntProperty(propPrefix.append("maxConnectionsPerRoute").toString(), 0);
        if (maxConnectionsPerRoute <= 0) {
            maxConnectionsPerRoute = totalConnections / numberOfEndpoints;
        }
        propPrefix.setLength(resetLen);
        int readTimeout = configService.getIntProperty(propPrefix.append("readTimeout").toString(), 2500);
        propPrefix.setLength(resetLen);
        int connectTimeout = configService.getIntProperty(propPrefix.append("connectTimeout").toString(), 1500);

        ClientConfig clientConfig = new ClientConfig(authLogin, authPassword, totalConnections, maxConnectionsPerRoute, readTimeout, connectTimeout);
        return new WeakforcedHttpClientConfig(httpClientId, clientConfig, services);
    }

    /**
     * Shuts-down this instance.
     */
    public void shutDown() {
        EnumMap<WeakforcedCall, ClientIdAndEndpointManager> endpoints = endpointsReference.getAndSet(null);
        if (null != endpoints) {
            HttpClientService clientService = services.getOptionalService(HttpClientService.class);
            if (clientService != null) {
                for (ClientIdAndEndpointManager c : endpoints.values()) {
                    clientService.destroyHttpClient(c.httpClientId);
                }
            }
        }
    }

    /**
     * Black-lists specified end-point for given call.
     *
     * @param call The call
     * @param endpoint The end-point to black-list
     */
    public void blacklist(WeakforcedCall call, Endpoint endpoint) {
        if (null == call) {
            return;
        }

        EnumMap<WeakforcedCall, ClientIdAndEndpointManager> endpoints = endpointsReference.get();
        if (null == endpoints) {
            return;
        }

        ClientIdAndEndpointManager c = endpoints.get(call);
        if (null == c) {
            return;
        }

        c.endpointManager.blacklist(endpoint);
    }

    /**
     * Gets the HTTP client and base URI for specified Weakforced call.
     *
     * @param call The Weakforced call that is about to be invoked
     * @return The HTTP client and base URI
     * @throws OXException If HTTP client and base URI cannot be returned
     */
    public HttpClientAndEndpoint getHttpClientAndUri(WeakforcedCall call) throws OXException {
        if (null == call) {
            throw WeakforcedExceptionCodes.UNKNOWN_CALL.create("null");
        }

        EnumMap<WeakforcedCall, ClientIdAndEndpointManager> endpoints = endpointsReference.get();
        if (null == endpoints) {
            throw OXException.general("Weakforced connector not initialized.");
        }

        ClientIdAndEndpointManager c = endpoints.get(call);
        if (null == c) {
            throw WeakforcedExceptionCodes.UNKNOWN_CALL.create(call.toString());
        }

        Endpoint endpoint = c.endpointManager.get();
        if (null == endpoint) {
            throw WeakforcedExceptionCodes.WEAKFORCED_NOT_REACHABLE.create(call.toString());
        }

        HttpClientService clientService = services.getOptionalService(HttpClientService.class);
        if (clientService == null) {
            throw ServiceExceptionCode.absentService(HttpClientService.class);
        }

        return new HttpClientAndEndpoint(clientService.getHttpClient(c.httpClientId), endpoint);
    }

    private static class ClientIdAndEndpointManager {

        final EndpointManager endpointManager;
        final String httpClientId;

        ClientIdAndEndpointManager(String httpClientId, EndpointManager endpointManager) {
            super();
            this.httpClientId = httpClientId;
            this.endpointManager = endpointManager;
        }
    }

}
