/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.weakforced;

import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStreamWriter;
import java.io.UnsupportedEncodingException;
import java.net.URI;
import java.net.URISyntaxException;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import org.apache.http.HttpHost;
import org.apache.http.HttpResponse;
import org.apache.http.NameValuePair;
import org.apache.http.StatusLine;
import org.apache.http.auth.AuthenticationException;
import org.apache.http.client.ClientProtocolException;
import org.apache.http.client.HttpClient;
import org.apache.http.client.HttpResponseException;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpRequestBase;
import org.apache.http.client.utils.URIBuilder;
import org.apache.http.client.utils.URLEncodedUtils;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.InputStreamEntity;
import org.apache.http.impl.auth.BasicScheme;
import org.apache.http.message.BasicNameValuePair;
import org.apache.http.protocol.BasicHttpContext;
import org.json.JSONException;
import org.json.JSONInputStream;
import org.json.JSONObject;
import org.json.JSONValue;
import com.openexchange.ajax.container.ThresholdFileHolder;
import com.openexchange.antiabuse.AllowParameters;
import com.openexchange.antiabuse.AntiAbuseService;
import com.openexchange.antiabuse.Protocol;
import com.openexchange.antiabuse.ReportParameters;
import com.openexchange.antiabuse.Status;
import com.openexchange.exception.OXException;
import com.openexchange.java.Charsets;
import com.openexchange.java.Streams;
import com.openexchange.java.Strings;
import com.openexchange.rest.client.endpointpool.Endpoint;

/**
 * {@link WeakforcedClient} - The Weakforced REST client.
 * <p>
 * The goal of 'weakforced' is to detect brute forcing of passwords across many
 * servers, services and instances. In order to support the real world, brute
 * force detection policy can be tailored to deal with "bulk, but legitimate"
 * users of your service, as well as botnet-wide slowscans of passwords.</p>
 *
 * @author <a href="mailto:thorben.betten@open-xchange.com">Thorben Betten</a>
 * @since v1.0.0
 */
public class WeakforcedClient implements AntiAbuseService {

    private static final org.slf4j.Logger LOGGER = org.slf4j.LoggerFactory.getLogger(WeakforcedClient.class);

    private static interface ResultType<R> {

        /**
         * Gets the class of the result type
         *
         * @return The class
         */
        Class<? extends R> getType();
    }

    private static final ResultType<Void> VOID = new ResultType<Void>() {

        @Override
        public Class<? extends Void> getType() {
            return null;
        }
    };

    private static final ResultType<InputStream> INPUT_STREAM = new ResultType<InputStream>() {

        @Override
        public Class<? extends InputStream> getType() {
            return InputStream.class;
        }
    };

    private static final ResultType<JSONObject> JSON = new ResultType<JSONObject>() {

        @Override
        public Class<? extends JSONObject> getType() {
            return JSONObject.class;
        }
    };

    private static final String JSON_FIELD_STATUS = "status";

    private static final String JSON_FIELD_MSG = "msg";

    private static final String JSON_FIELD_ATTRIBUTES = "r_attrs";

    // -------------------------------------------------------------------------------------------------------------- //

    /** The status code policy to obey */
    public static interface StatusCodePolicy {

        /**
         * Examines given status line
         *
         * @param httpResponse The HTTP response
         * @throws OXException If an Open-Xchange error is yielded from status
         * @throws HttpResponseException If status is interpreted as an error
         */
        void handleStatusCode(HttpResponse httpResponse) throws OXException, HttpResponseException;
    }

    /** The default status code policy; accepting greater than/equal to <code>200</code> and lower than <code>300</code> */
    public static final StatusCodePolicy STATUS_CODE_POLICY_DEFAULT = new StatusCodePolicy() {

        @Override
        public void handleStatusCode(HttpResponse httpResponse) throws OXException, HttpResponseException {
            final StatusLine statusLine = httpResponse.getStatusLine();
            final int statusCode = statusLine.getStatusCode();
            if (statusCode < 200 || statusCode >= 300) {
                if (404 == statusCode) {
                    throw WeakforcedExceptionCodes.NOT_FOUND_SIMPLE.create();
                }
                String reason;
                try {
                    InputStreamReader reader = new InputStreamReader(httpResponse.getEntity().getContent(), Charsets.UTF_8);
                    String sResponse = Streams.reader2string(reader);
                    JSONObject jsonObject = new JSONObject(sResponse);
                    reason = jsonObject.getString("reason");
                } catch (final Exception e) {
                    reason = statusLine.getReasonPhrase();
                }
                throw new HttpResponseException(statusCode, reason);
            }
        }
    };

    /** The status code policy; accepting greater than/equal to <code>200</code> and lower than <code>300</code> while ignoring <code>404</code> */
    public static final StatusCodePolicy STATUS_CODE_POLICY_IGNORE_NOT_FOUND = new StatusCodePolicy() {

        @Override
        public void handleStatusCode(HttpResponse httpResponse) throws HttpResponseException {
            final StatusLine statusLine = httpResponse.getStatusLine();
            final int statusCode = statusLine.getStatusCode();
            if ((statusCode < 200 || statusCode >= 300) && statusCode != 404) {
                String reason;
                try {
                    final JSONObject jsonObject = new JSONObject(new InputStreamReader(httpResponse.getEntity().getContent(), Charsets.UTF_8));
                    reason = jsonObject.getJSONObject("error").getString("message");
                } catch (final Exception e) {
                    reason = statusLine.getReasonPhrase();
                }
                throw new HttpResponseException(statusCode, reason);
            }
        }
    };

    // -------------------------------------------------------------------------------------------------------------- //

    private final String algorithm;
    private final boolean truncate;
    private final String secret;
    private final WeakforcedEndpointManager endpoints;
    private final BasicHttpContext localcontext;

    /**
     * Initializes a new {@link WeakforcedClient}.
     */
    public WeakforcedClient(WeakforcedEndpointManager endpoints, String secret, String algorithm, boolean truncate) {
        super();
        this.endpoints = endpoints;
        this.secret = secret;
        this.algorithm = Strings.isEmpty(algorithm) ? "SHA-256" : algorithm.trim();
        this.truncate = truncate;

        // Generate BASIC scheme object and stick it to the local execution context
        final BasicHttpContext context = new BasicHttpContext();
        final BasicScheme basicAuth = new BasicScheme();
        context.setAttribute("preemptive-auth", basicAuth);
        this.localcontext = context;
    }

    private CallProperties getCallProperties(WeakforcedCall call) throws OXException {
        HttpClientAndEndpoint clientAndUri = endpoints.getHttpClientAndUri(call);
        Endpoint endpoint = clientAndUri.endpoint;
        String sUrl = endpoint.getBaseUri();
        try {
            URI uri = new URI(sUrl);
            HttpHost targetHost = new HttpHost(uri.getHost(), uri.getPort(), uri.getScheme());
            return new CallProperties(uri, clientAndUri.httpClient, targetHost, endpoint);
        } catch (final URISyntaxException e) {
            throw WeakforcedExceptionCodes.INVALID_WEAKFORCED_URL.create(null == sUrl ? "<empty>" : sUrl);
        }
    }

    /**
     * Shuts-down this instance.
     */
    public void shutDown() {
        endpoints.shutDown();
    }

    @Override
    public Status allow(AllowParameters parameters) throws OXException {
        try {
            Map<String, String> requestParameters = new LinkedHashMap<String, String>(2);
            requestParameters.put("command", "allow");

            JSONObject jBody = new JSONObject(10);
            jBody.put("login", parameters.getLogin());
            jBody.put("remote", parameters.getRemoteAddress());
            jBody.put("pwhash", createPasswordHash(parameters.getLogin(), parameters.getPassword(), secret, algorithm, truncate));
            String userAgent = parameters.getUserAgent();
            if (!Strings.isEmpty(userAgent)) {
                jBody.put("device_id", userAgent);
            }
            Protocol protocol = parameters.getProtocol();
            if (null != protocol) {
                jBody.put("protocol", protocol.getName());
                jBody.put("tls", protocol.isSecure());
            }
            Map<String, String> attributes = parameters.getAttributes();
            if (null != attributes && !attributes.isEmpty()) {
                jBody.put("attrs", new JSONObject(attributes));
            }

            JSONObject jStatus = executePost(WeakforcedCall.ALLOW, requestParameters, jBody, JSON);

            int status = jStatus.getInt(JSON_FIELD_STATUS);
            String message = null;
            Map<String, Object> rAttributes = null;
            Map<String, String> properties = null;
            {
                int length = jStatus.length();
                if (length > 1) {
                    // Has more than one JSON field; check others...
                    properties = new LinkedHashMap<String, String>(length);
                    for (Map.Entry<String, Object> entry : jStatus.entrySet()) {
                        String jName = entry.getKey();
                        if (JSON_FIELD_STATUS.equals(jName)) {
                            // Ignore
                        } else if (JSON_FIELD_ATTRIBUTES.equals(jName)) {
                            rAttributes = ((JSONObject) entry.getValue()).asMap();
                        } else if (JSON_FIELD_MSG.equals(jName)) {
                            message = entry.getValue().toString();
                        } else {
                            properties.put(entry.getKey(), entry.getValue().toString());
                        }
                    }
                }
            }

            return new Status(status, message, rAttributes, properties);
        } catch (JSONException e) {
            throw WeakforcedExceptionCodes.JSON_ERROR.create(e, e.getMessage());
        } catch (RuntimeException e) {
            throw WeakforcedExceptionCodes.UNEXPECTED_ERROR.create(e, e.getMessage());
        }
    }

    @Override
    public void report(ReportParameters parameters) throws OXException {
        try {
            Map<String, String> requestParameters = new LinkedHashMap<String, String>(2);
            requestParameters.put("command", "report");

            JSONObject jBody = new JSONObject(10);
            switch (parameters.getReportValue()) {
                case DENIED_BY_SERVICE:
                    jBody.put("success",  false);
                    jBody.put("policy_reject", true);
                    break;
                case FAILURE:
                    jBody.put("success", false);
                    break;
                case SUCCESS:
                    jBody.put("success", true);
                    break;
                default:
                    jBody.put("success", true);
                    break;
            }
            jBody.put("login", parameters.getLogin());
            jBody.put("remote", parameters.getRemoteAddress());
            jBody.put("pwhash", createPasswordHash(parameters.getLogin(), parameters.getPassword(), secret, algorithm, truncate));
            String userAgent = parameters.getUserAgent();
            if (!Strings.isEmpty(userAgent)) {
                jBody.put("device_id", userAgent);
            }
            Protocol protocol = parameters.getProtocol();
            if (null != protocol) {
                jBody.put("protocol", protocol.getName());
                jBody.put("tls", protocol.isSecure());
            }

            executePost(WeakforcedCall.REPORT, requestParameters, jBody, VOID);
        } catch (JSONException e) {
            throw WeakforcedExceptionCodes.JSON_ERROR.create(e, e.getMessage());
        } catch (RuntimeException e) {
            throw WeakforcedExceptionCodes.UNEXPECTED_ERROR.create(e, e.getMessage());
        }
    }

    private <R> R executePost(WeakforcedCall call, Map<String, String> parameters, JSONObject jBody, ResultType<R> resultType) throws OXException {
        CallProperties callProperties = getCallProperties(call);

        HttpPost post = null;
        try {
            URI uri = buildUri(callProperties.uri, toQueryString(parameters), null);
            post = new HttpPost(uri);
            post.setEntity(new InputStreamEntity(new JSONInputStream(jBody, "UTF-8"), -1L, ContentType.APPLICATION_JSON));
            post.setHeader("Content-Type", "application/json");

            return handleHttpResponse(execute(post, callProperties.targetHost, callProperties.httpClient), resultType);
        } catch (final HttpResponseException e) {
            if (400 == e.getStatusCode() || 401 == e.getStatusCode()) {
                // Authentication failed
                throw WeakforcedExceptionCodes.AUTH_ERROR.create(e, e.getMessage());
            }
            throw handleHttpResponseError(null, e);
        } catch (final IOException e) {
            throw handleIOError(e, callProperties.endpoint, call);
        } catch (final RuntimeException e) {
            throw WeakforcedExceptionCodes.UNEXPECTED_ERROR.create(e, e.getMessage());
        } finally {
            reset(post);
        }
    }

    // --------------------------------------------------------------------------------------------------------------------------------

    /**
     * Builds the URI from given arguments
     *
     * @param baseUri The base URI
     * @param queryString The query string parameters
     * @return The built URI string
     * @throws IllegalArgumentException If the given string violates RFC 2396
     */
    protected static URI buildUri(URI baseUri, List<NameValuePair> queryString, String optPath) {
        try {
            URIBuilder builder = new URIBuilder();
            String path = null == optPath ? baseUri.getPath() : optPath;
            builder.setScheme(baseUri.getScheme()).setHost(baseUri.getHost()).setPort(baseUri.getPort()).setPath(Strings.isEmpty(path) ? "/" : path).setQuery(null == queryString ? null : URLEncodedUtils.format(queryString, "UTF-8"));
            return builder.build();
        } catch (final URISyntaxException x) {
            throw new IllegalArgumentException("Failed to build URI", x);
        }
    }

    /**
     * Turns specified JSON value into an appropriate HTTP entity.
     *
     * @param jValue The JSON value
     * @return The HTTP entity
     * @throws JSONException If a JSON error occurs
     * @throws IOException If an I/O error occurs
     */
    protected InputStreamEntity asHttpEntity(JSONValue jValue) throws JSONException, IOException {
        if (null == jValue) {
            return null;
        }

        ThresholdFileHolder sink = new ThresholdFileHolder();
        boolean error = true;
        try {
            final OutputStreamWriter osw = new OutputStreamWriter(sink.asOutputStream(), Charsets.UTF_8);
            jValue.write(osw);
            osw.flush();
            final InputStreamEntity entity = new InputStreamEntity(sink.getStream(), sink.getLength(), ContentType.APPLICATION_JSON);
            error = false;
            return entity;
        } catch (final OXException e) {
            final Throwable cause = e.getCause();
            if (cause instanceof IOException) {
                throw (IOException) cause;
            }
            throw new IOException(null == cause ? e : cause);
        } finally {
            if (error) {
                Streams.close(sink);
            }
        }
    }

    /**
     * Gets the appropriate query string for given parameters
     *
     * @param parameters The parameters
     * @return The query string
     */
    protected static List<NameValuePair> toQueryString(Map<String, String> parameters) {
        if (null == parameters || parameters.isEmpty()) {
            return null;
        }
        final List<NameValuePair> l = new LinkedList<NameValuePair>();
        for (final Map.Entry<String, String> e : parameters.entrySet()) {
            l.add(new BasicNameValuePair(e.getKey(), e.getValue()));
        }
        return l;
    }

    /**
     * Executes specified HTTP method/request using given HTTP client instance.
     *
     * @param method The method/request to execute
     * @param targetHost The target host
     * @param httpClient The HTTP client to use
     * @return The HTTP response
     * @throws ClientProtocolException If client protocol error occurs
     * @throws IOException If an I/O error occurs
     */
    protected HttpResponse execute(HttpRequestBase method, HttpHost targetHost, HttpClient httpClient) throws ClientProtocolException, IOException {
        return execute(method, targetHost, httpClient, localcontext);
    }

    /**
     * Executes specified HTTP method/request using given HTTP client instance.
     *
     * @param method The method/request to execute
     * @param targetHost The target host
     * @param httpClient The HTTP client to use
     * @param context The context
     * @return The HTTP response
     * @throws ClientProtocolException If client protocol error occurs
     * @throws IOException If an I/O error occurs
     */
    protected HttpResponse execute(HttpRequestBase method, HttpHost targetHost, HttpClient httpClient, BasicHttpContext context) throws ClientProtocolException, IOException {
        return httpClient.execute(targetHost, method, context);
    }

    /**
     * Resets given HTTP request
     *
     * @param request The HTTP request
     */
    protected static void reset(HttpRequestBase request) {
        if (null != request) {
            try {
                request.reset();
            } catch (final Exception e) {
                // Ignore
            }
        }
    }

    /**
     * Handles given HTTP response while expecting <code>200 (Ok)</code> status code.
     *
     * @param httpResponse The HTTP response
     * @param type The type of the result object
     * @return The result object
     * @throws OXException If an Open-Xchange error occurs
     * @throws ClientProtocolException If a client protocol error occurs
     * @throws IOException If an I/O error occurs
     */
    protected <R> R handleHttpResponse(HttpResponse httpResponse, ResultType<R> type) throws OXException, ClientProtocolException, IOException {
        return handleHttpResponse(httpResponse, STATUS_CODE_POLICY_DEFAULT, type);
    }

    /**
     * Handles given HTTP response while expecting given status code.
     *
     * @param httpResponse The HTTP response
     * @param policy The status code policy to obey
     * @param type The type of the result object
     * @return The result object
     * @throws OXException If an Open-Xchange error occurs
     * @throws ClientProtocolException If a client protocol error occurs
     * @throws IOException If an I/O error occurs
     * @throws IllegalStateException If content stream cannot be created
     */
    protected <R> R handleHttpResponse(HttpResponse httpResponse, StatusCodePolicy policy, ResultType<R> type) throws OXException, ClientProtocolException, IOException {
        policy.handleStatusCode(httpResponse);

        // OK, continue
        if (JSON == type) {
            try {
                return (R) new JSONObject(new InputStreamReader(httpResponse.getEntity().getContent(), Charsets.UTF_8));
            } catch (final JSONException e) {
                throw WeakforcedExceptionCodes.JSON_ERROR.create(e, e.getMessage());
            }
        }
        if (VOID == type) {
            return null;
        }
        if (INPUT_STREAM == type) {
            return (R) httpResponse.getEntity().getContent();
        }
        return null;
    }

    /**
     * Handles given I/O error.
     *
     * @param e The I/O error
     * @param endpoint The end-point for which an I/O error occurred
     * @param call The associated call
     * @return The resulting exception
     */
    protected OXException handleIOError(IOException e, Endpoint endpoint, WeakforcedCall call) {
        final Throwable cause = e.getCause();
        if (cause instanceof AuthenticationException) {
            return WeakforcedExceptionCodes.AUTH_ERROR.create(cause, cause.getMessage());
        }
        endpoints.blacklist(call, endpoint);
        return WeakforcedExceptionCodes.IO_ERROR.create(e, e.getMessage());
    }

    /** Status code (401) indicating that the request requires HTTP authentication. */
    private static final int SC_UNAUTHORIZED = 401;

    /** Status code (404) indicating that the requested resource is not available. */
    private static final int SC_NOT_FOUND = 404;

    /**
     * Handles given HTTP response error.
     *
     * @param identifier The optional identifier for associated Microsoft OneDrive resource
     * @param e The HTTP error
     * @return The resulting exception
     */
    protected OXException handleHttpResponseError(String identifier, HttpResponseException e) {
        if (null != identifier && SC_NOT_FOUND == e.getStatusCode()) {
            return WeakforcedExceptionCodes.NOT_FOUND.create(e, identifier);
        }
        if (SC_UNAUTHORIZED == e.getStatusCode()) {
            return WeakforcedExceptionCodes.AUTH_ERROR.create();
        }
        return WeakforcedExceptionCodes.WEAKFORCED_SERVER_ERROR.create(e, Integer.valueOf(e.getStatusCode()), e.getMessage());
    }

    private static String createPasswordHash(String login, String password, String secret, String algorithm, boolean truncate) throws OXException {
        // According to Weakforced proposal: TRUNCATE(SHA256(SECRET + LOGIN + '\x00' + PASSWORD), 12) with 4 0 bits prepended
        byte[] digest = getDigest(new StringBuilder(32).append(secret).append(login).append('\0').append(password).toString(), algorithm);

        // Do the bit shifting/extracting stuff & return as HEX with padding of 4
        if (truncate) {
            return toHexString(((digest[0] & 0xFF) << 4) | ((digest[1] & 0xFF) >> 4), 4);
        }

        return asHex(digest);
    }

    /**
     * All possible chars for representing a number as a String
     */
    private static final char[] digits = {
        '0' , '1' , '2' , '3' , '4' , '5' ,
        '6' , '7' , '8' , '9' , 'a' , 'b' ,
        'c' , 'd' , 'e' , 'f' , 'g' , 'h' ,
        'i' , 'j' , 'k' , 'l' , 'm' , 'n' ,
        'o' , 'p' , 'q' , 'r' , 's' , 't' ,
        'u' , 'v' , 'w' , 'x' , 'y' , 'z'
    };

    /**
     * Converts the two-byte integer to a HEX string.
     */
    private static String toHexString(int i, int padding) {
        char[] buf = new char[16];
        int charPos = 16;
        int mask = (1 << 4) - 1;
        do {
            buf[--charPos] = digits[i & mask];
            i >>>= 4;
        } while (i != 0);

        while ((16 - charPos) < padding) {
            buf[--charPos] = '0';
        }

        return new String(buf, charPos, (16 - charPos));
    }

    private static final char[] HEX_CHARS = { '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', 'a', 'b', 'c', 'd', 'e', 'f' };

    /**
     * Turns array of bytes into string representing each byte as unsigned hex number.
     *
     * @param hash Array of bytes to convert to hex-string
     * @return Generated hex string
     */
    private static String asHex(final byte[] hash) {
        final int length = hash.length;
        final char[] buf = new char[length * 2];
        for (int i = 0, x = 0; i < length; i++) {
            buf[x++] = HEX_CHARS[(hash[i] >>> 4) & 0xf];
            buf[x++] = HEX_CHARS[hash[i] & 0xf];
        }
        return new String(buf);
    }

    /**
     * Gets the SHA256 digest of specified string.
     *
     * @param string The string to hash
     * @param algorithm The name of the algorithm
     * @return The digest
     */
    private static byte[] getDigest(String string, String algorithm) throws OXException {
        try {
            MessageDigest md = MessageDigest.getInstance(algorithm);
            md.update(string.getBytes("UTF-8"));
            return md.digest();
        } catch (final NoSuchAlgorithmException e) {
            throw WeakforcedExceptionCodes.INVALID_ALGORITHM_NAME.create(algorithm);
        } catch (final UnsupportedEncodingException e) {
            LOGGER.error("", e);
        }
        return null;
    }

    // --------------------------------------------------------------------------------------------------------------------------------

    private static final class CallProperties {

        final URI uri;
        final HttpHost targetHost;
        final HttpClient httpClient;
        final Endpoint endpoint;

        CallProperties(URI uri, HttpClient httpClient, HttpHost targetHost, Endpoint endpoint) {
            super();
            this.uri = uri;
            this.httpClient = httpClient;
            this.targetHost = targetHost;
            this.endpoint = endpoint;
        }
    }

}
