/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.weakforced;

import java.io.IOException;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.Arrays;
import java.util.EnumMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicReference;
import org.apache.http.HttpEntity;
import org.apache.http.HttpResponse;
import org.apache.http.auth.AuthScope;
import org.apache.http.auth.Credentials;
import org.apache.http.auth.UsernamePasswordCredentials;
import org.apache.http.client.HttpClient;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpRequestBase;
import org.apache.http.impl.client.DefaultHttpClient;
import org.apache.http.util.EntityUtils;
import com.openexchange.config.ConfigurationService;
import com.openexchange.exception.OXException;
import com.openexchange.java.Strings;
import com.openexchange.rest.client.endpointpool.Endpoint;
import com.openexchange.rest.client.endpointpool.EndpointAvailableStrategy;
import com.openexchange.rest.client.endpointpool.EndpointManager;
import com.openexchange.rest.client.endpointpool.EndpointManagerFactory;
import com.openexchange.rest.client.httpclient.HttpClients;
import com.openexchange.rest.client.httpclient.HttpClients.ClientConfig;
import com.openexchange.version.Version;
import com.openexchange.weakforced.WeakforcedClient.PreemptiveAuth;

/**
 * {@link WeakforcedEndpointManager}
 *
 * @author <a href="mailto:thorben.betten@open-xchange.com">Thorben Betten</a>
 * @since v1.0.0
 */
public class WeakforcedEndpointManager {

    private static final org.slf4j.Logger LOG = org.slf4j.LoggerFactory.getLogger(WeakforcedEndpointManager.class);

    private static class Entry {
        final DefaultHttpClient httpClient;
        final EndpointManager endpointManager;

        Entry(DefaultHttpClient httpClient, EndpointManager endpointManager) {
            super();
            this.httpClient = httpClient;
            this.endpointManager = endpointManager;
        }
    }

    private static DefaultHttpClient newHttpClient(int totalConnections, int maxConnectionsPerRoute, int readTimeout, int connectTimeout, String authLogin, String authPassword) {
        ClientConfig clientConfig = ClientConfig.newInstance()
            .setUserAgent("OX Weakforced Http Client v" + Version.getInstance().getVersionString())
            .setMaxTotalConnections(totalConnections)
            .setMaxConnectionsPerRoute(maxConnectionsPerRoute)
            .setConnectionTimeout(connectTimeout)
            .setSocketReadTimeout(readTimeout);

        DefaultHttpClient tmp = HttpClients.getHttpClient(clientConfig);

        Credentials credentials = new UsernamePasswordCredentials(authLogin, authPassword);
        tmp.getCredentialsProvider().setCredentials(AuthScope.ANY, credentials);

        // Add "preemptive-auth" as the first request interceptor
        tmp.addRequestInterceptor(new PreemptiveAuth(), 0);
        return tmp;
    }

    private static Entry entryFor(String endPoints, EndpointManagerFactory factory, String authLogin, String authPassword, StringBuilder propPrefix, ConfigurationService configService) throws OXException {
        // Parse end-point list
        List<String> l = Arrays.asList(Strings.splitByComma(endPoints.trim()));

        // Read properties for HTTP connections/pooling
        int totalConnections = configService.getIntProperty(propPrefix.append("totalConnections").toString(), 100);
        int maxConnectionsPerRoute = configService.getIntProperty(propPrefix.append("maxConnectionsPerRoute").toString(), 0);
        if (maxConnectionsPerRoute <= 0) {
            maxConnectionsPerRoute = totalConnections / l.size();
        }
        int readTimeout = configService.getIntProperty(propPrefix.append("readTimeout").toString(), 2500);
        int connectTimeout = configService.getIntProperty(propPrefix.append("connectTimeout").toString(), 1500);

        // Initialize HTTP client
        DefaultHttpClient httpClient = newHttpClient(totalConnections, maxConnectionsPerRoute, readTimeout, connectTimeout, authLogin, authPassword);

        // Setup end-point manager
        EndpointManager endpointManager = factory.createEndpointManager(l, httpClient, AVAILABILITY_STRATEGY, 60, TimeUnit.SECONDS);

        // Return entry for bundled HTTP client & end-point manager
        return new Entry(httpClient, endpointManager);
    }

    private static final EndpointAvailableStrategy AVAILABILITY_STRATEGY = new EndpointAvailableStrategy() {

        @Override
        public AvailableResult isEndpointAvailable(Endpoint endpoint, HttpClient httpClient) throws OXException {
            HttpGet get = null;
            HttpResponse response = null;
            try {
                get = new HttpGet(WeakforcedClient.buildUri(new URI(endpoint.getBaseUri()), WeakforcedClient.toQueryString(mapFor("command", "ping")), null));
                response = httpClient.execute(get);
                int status = response.getStatusLine().getStatusCode();
                if (200 == status) {
                    return AvailableResult.AVAILABLE;
                }
                if (401 == status) {
                    return AvailableResult.NONE;
                }
            } catch (URISyntaxException e) {
                // ignore
                return AvailableResult.NONE;
            } catch (IOException e) {
                // ignore
            } finally {
                close(get, response);
            }

            return AvailableResult.UNAVAILABLE;
        }
    };

    /**
     * Gets a (parameters) map for specified arguments.
     *
     * @param args The arguments
     * @return The resulting map
     */
    static Map<String, String> mapFor(String... args) {
        if (null == args) {
            return null;
        }

        int length = args.length;
        if (0 == length || (length % 2) != 0) {
            return null;
        }

        Map<String, String> map = new LinkedHashMap<String, String>(length >> 1);
        for (int i = 0; i < length; i+=2) {
            map.put(args[i], args[i+1]);
        }
        return map;
    }

    /**
     * Closes the supplied HTTP request & response resources silently.
     *
     * @param request The HTTP request to reset
     * @param response The HTTP response to consume and close
     */
    static void close(HttpRequestBase request, HttpResponse response) {
        if (null != response) {
            HttpEntity entity = response.getEntity();
            if (null != entity) {
                try {
                    EntityUtils.consume(entity);
                } catch (Exception e) {
                    LOG.debug("Error consuming HTTP response entity", e);
                }
            }
        }

        if (null != request) {
            try {
                request.reset();
            } catch (Exception e) {
                LOG.debug("Error resetting HTTP request", e);
            }
        }
    }

    // -------------------------------------------------------------------------------------------------------------------------------

    private final AtomicReference<EnumMap<WeakforcedCall, Entry>> endpointsReference;

    /**
     * Initializes a new {@link WeakforcedEndpointManager}.
     */
    public WeakforcedEndpointManager() {
        super();
        endpointsReference = new AtomicReference<EnumMap<WeakforcedCall,Entry>>(null);
    }

    /**
     * Initializes this instance.
     *
     * @param factory The end-point factory to use
     * @param configService The configuration service to read properties from
     * @throws OXException If initialization fails
     */
    public void init(EndpointManagerFactory factory, ConfigurationService configService) throws OXException {
        org.slf4j.Logger logger = org.slf4j.LoggerFactory.getLogger(WeakforcedEndpointManager.class);

        String authLogin = configService.getProperty("com.openexchange.weakforced.basic-auth.login");
        String authPassword = configService.getProperty("com.openexchange.weakforced.basic-auth.password");
        if (Strings.isEmpty(authLogin) || Strings.isEmpty(authPassword)) {
            throw OXException.general("Denied initialization due to unset Basic-Auth configuration. Please set properties 'com.openexchange.weakforced.basic-auth.login' and 'com.openexchange.weakforced.basic-auth.password' appropriately.");
        }
        authLogin = authLogin.trim();
        authPassword = authPassword.trim();

        EnumMap<WeakforcedCall, Entry> endpoints = new EnumMap<WeakforcedCall, Entry>(WeakforcedCall.class);

        String fallBackName = "com.openexchange.weakforced.endpoints";
        Entry fallBackEntry = null;
        for (WeakforcedCall call : WeakforcedCall.values()) {
            String propName = "com.openexchange.weakforced.endpoints." + call.toString();
            String endPoints = configService.getProperty(propName);
            if (Strings.isEmpty(endPoints)) {
                if (null == fallBackEntry) {
                    endPoints = configService.getProperty(fallBackName);
                    if (Strings.isEmpty(endPoints)) {
                        // No end-point
                        logger.info("No Weakforced end-points defined for call {} via property {}", call.toString(), propName);
                        return;
                    }

                    fallBackEntry = entryFor(endPoints, factory, authLogin, authPassword, new StringBuilder(fallBackName.length() + 1).append(fallBackName).append('.'), configService);
                }
                endpoints.put(call, fallBackEntry);
            } else {
                endpoints.put(call, entryFor(endPoints, factory, authLogin, authPassword, new StringBuilder(propName.length() + 1).append(propName).append('.'), configService));
            }
        }

        endpointsReference.set(endpoints);
    }

    /**
     * Shuts-down this instance.
     */
    public void shutDown() {
        EnumMap<WeakforcedCall, Entry> endpoints = endpointsReference.getAndSet(null);
        if (null != endpoints) {
            for (Entry entry : endpoints.values()) {
                HttpClients.shutDown(entry.httpClient);
            }
        }
    }

    /**
     * Black-lists specified end-point for given call.
     *
     * @param call The call
     * @param endpoint The end-point to black-list
     */
    public void blacklist(WeakforcedCall call, Endpoint endpoint) {
        if (null == call) {
            return;
        }

        EnumMap<WeakforcedCall, Entry> endpoints = endpointsReference.get();
        if (null == endpoints) {
            return;
        }

        Entry entry = endpoints.get(call);
        if (null == entry) {
            return;
        }

        entry.endpointManager.blacklist(endpoint);
    }

    /**
     * Gets the HTTP client and base URI for specified Weakforced call.
     *
     * @param call The Weakforced call that is about to be invoked
     * @return The HTTP client and base URI
     * @throws OXException If HTTP client and base URI cannot be returned
     */
    public HttpClientAndEndpoint getHttpClientAndUri(WeakforcedCall call) throws OXException {
        if (null == call) {
            throw WeakforcedExceptionCodes.UNKNOWN_CALL.create("null");
        }

        EnumMap<WeakforcedCall, Entry> endpoints = endpointsReference.get();
        if (null == endpoints) {
            throw OXException.general("Weakforced connector not initialized.");
        }

        Entry entry = endpoints.get(call);
        if (null == entry) {
            throw WeakforcedExceptionCodes.UNKNOWN_CALL.create(call.toString());
        }

        Endpoint endpoint = entry.endpointManager.get();
        if (null == endpoint) {
            throw WeakforcedExceptionCodes.WEAKFORCED_NOT_REACHABLE.create(call.toString());
        }

        return new HttpClientAndEndpoint(entry.httpClient, endpoint);
    }

}
