/**
 * fs-utils <https://github.com/assemble/fs-utils>
 *
 * Copyright (c) 2014 Jon Schlinkert, Brian Woodward, contributors.
 * Licensed under the MIT license.
 */

const expect = require('chai').expect;
const file   = require('../');
const path   = require('path');
const cwd    = process.cwd();

// Normalize slashes in some test results
var normalize = file.normalizeSlash;

describe('Normalize slashes', function() {
  it('should normalize slash', function() {
    var expected = 'foo/bar/baz';
    var actual = file.normalizeSlash('foo\\bar/baz');
    expect(actual).to.eql(expected);

    expected = '/foo/bar/baz/';
    actual = file.normalizeSlash('\\foo\\bar\\baz\\');
    expect(actual).to.eql(expected);
  });
});

describe('Trailing slashes', function() {
  describe('Add trailing slashes', function() {
    it('should add a trailing slash when it appears to be a directory', function() {
      var expected = 'foo/bar/baz/';
      var actual = file.addTrailingSlash('foo/bar/baz');
      expect(normalize(actual)).to.eql(expected);

      expected = '/foo/bar/baz/';
      actual = file.addTrailingSlash('/foo/bar/baz');
      expect(normalize(actual)).to.eql(expected);

      expected = 'foo/bar.baz/quux/';
      actual = file.addTrailingSlash('./foo/bar.baz/quux');
      expect(normalize(actual)).to.eql(expected);

      expected = 'foo/bar/baz/';
      actual = file.addTrailingSlash('./foo/bar/baz');
      expect(normalize(actual)).to.eql(expected);

      expected = '/foo/bar/baz/';
      actual = file.addTrailingSlash('\\foo\\bar\\baz');
      expect(normalize(actual)).to.eql(expected);

      expected = 'foo/bar/baz/';
      actual = file.addTrailingSlash('foo\\bar\\baz');
      expect(normalize(actual)).to.eql(expected);
    });

    it('should not add a trailing slash when it already has one', function() {
      var expected = 'foo/bar/baz/';
      var actual = file.addTrailingSlash('foo/bar/baz/');
      expect(normalize(actual)).to.eql(expected);

      expected = '/foo/bar/baz/';
      actual = file.addTrailingSlash('/foo/bar/baz/');
      expect(normalize(actual)).to.eql(expected);
    });

    it('should not add a trailing slash when it appears to be a file', function() {
      var expected = 'foo/bar/baz.md';
      var actual = file.addTrailingSlash('./foo/bar/baz.md');
      expect(normalize(actual)).to.eql(expected);

      expected = '/foo/bar/baz.md';
      actual = file.addTrailingSlash('/foo/bar/baz.md');
      expect(normalize(actual)).to.eql(expected);

      expected = '/foo/bar/baz.md';
      actual = file.addTrailingSlash('\\foo\\bar\\baz.md');
      expect(normalize(actual)).to.eql(expected);
    });
  });

  describe('Remove trailing slashes', function() {
    it('should remove a trailing slash from the given file path', function() {
      var expected = 'one/two';
      var actual = file.removeTrailingSlash('./one/two/');
      expect(normalize(actual)).to.eql(expected);

      expected = '/three/four/five';
      actual = file.removeTrailingSlash('/three/four/five/');
      expect(normalize(actual)).to.eql(expected);

      expected = '/six/seven/eight';
      actual = file.removeTrailingSlash('\\six\\seven\\eight\\');
      expect(normalize(actual)).to.eql(expected);
    });
  });
});


describe('endsWith', function() {
  it('should return true if the path ends with the given string', function() {
    var expected = true;
    var actual = file.endsWith('foo\\bar\\baz\\', '/');
    expect(actual).to.eql(expected);

    expected = true;
    actual = file.endsWith('foo\\bar\\baz\\', '\\');
    expect(actual).to.eql(expected);

    expected = true;
    actual = file.endsWith('foo/bar/baz/', '/');
    expect(actual).to.eql(expected);

    expected = true;
    actual = file.endsWith('foo\\bar\\baz.md', 'baz.md');
    expect(actual).to.eql(expected);

    expected = true;
    actual = file.endsWith('foo\\bar\\baz.md', '.md');
    expect(actual).to.eql(expected);
  });

  it('should return false if the path does not end with the given string', function() {
    var expected = false;
    var actual = file.endsWith('foo\\bar\\baz.md', '/');
    expect(actual).to.eql(expected);

    expected = false;
    actual = file.endsWith('foo\\bar\\baz.md', 'baz');
    expect(actual).to.eql(expected);
  });
});

describe('lastExt', function() {
  it('should return the last file extension', function() {
    var expected = 'md';
    var actual = file.lastExt('foo/bar/baz/quux.bar/file.tmpl.md');
    expect(actual).to.eql(expected);

    expected = 'md';
    actual = file.lastExt('./foo/bar/baz/quux.bar/file.tmpl.md');
    expect(actual).to.eql(expected);

    expected = '';
    actual = file.lastExt('foo/bar/baz/quux.bar/CHANGELOG');
    expect(actual).to.eql(expected);

    expected = 'gitignore';
    actual = file.lastExt('/foo/bar/baz/quux.bar/.gitignore');
    expect(actual).to.eql(expected);

    expected = 'html';
    actual = file.lastExt('./foo/bar/baz/quux/index.html');
    expect(actual).to.eql(expected);
  });
});


describe('withExt', function() {
  it('should return files from specified directory that end with the given file extension.', function() {
    var expected = ['test.txt'];
    var actual = file.withExt('./test/fixtures', 'txt');
    expect(actual).to.eql(expected);

    expected = ['fs-tests.js', 'path-tests.js'];
    actual = file.withExt('./test', 'js');
    expect(actual).to.eql(expected);
  });
});


describe('firstSegment', function() {
  it('should return the first segment in the given file path', function() {
    var expected = 'apple';
    var actual = file.firstSegment('apple/orange/file.ext');
    expect(actual).to.eql(expected);

    expected = 'grape';
    actual = file.firstSegment('/grape/watermelon/quux');
    expect(actual).to.eql(expected);

    expected = 'banana';
    actual = file.firstSegment('./banana/strawberry/quux/');
    expect(actual).to.eql(expected);
  });
});

describe('lastSegment', function() {
  it('should return the last segment in the given file path', function() {
    var expected = 'file.ext';
    var actual = file.lastSegment('square/rectangle/file.ext');
    expect(actual).to.eql(expected);

    expected = 'four';
    actual = file.lastSegment('one/two/three/four');
    expect(actual).to.eql(expected);

    expected = 'grape';
    actual = file.lastSegment('apple/orange/grape/');
    expect(actual).to.eql(expected);
  });
});

describe('removeFilename', function() {
  it('should remove the filename from the given file path', function() {
    var expected = 'square/rectangle/';
    var actual = file.removeFilename('square/rectangle/file.ext');
    expect(normalize(actual)).to.eql(expected);

    expected = 'one/two/three/four';
    actual = file.removeFilename('one/two/three/four');
    expect(normalize(actual)).to.eql(expected);

    expected = 'apple/orange/grape/';
    actual = file.removeFilename('apple/orange/grape/');
    expect(normalize(actual)).to.eql(expected);
  });
});

describe('dirname', function() {

  it('should return the dirname in the given file path', function() {
    var expected = 'square/rectangle/';
    var actual = file.dirname('square/rectangle/file.ext');
    expect(normalize(actual)).to.eql(expected);

    expected = 'one/two/three/';
    actual = file.dirname('one/two/three/four');
    expect(normalize(actual)).to.eql(expected);

    expected = 'apple/orange/grape/';
    actual = file.dirname('apple/orange/grape/');
    expect(normalize(actual)).to.eql(expected);
  });

  it('should return the directory in the given file path, including filenames that do not have an extensions.', function() {
    var expected = 'square/rectangle/';
    var actual = file.dir('square/rectangle/file.ext');
    expect(normalize(actual)).to.eql(expected);

    expected = 'one/two/three/four';
    actual = file.dir('one/two/three/four');
    expect(normalize(actual)).to.eql(expected);

    expected = 'one/two/three/CHANGELOG';
    actual = file.dir('one/two/three/CHANGELOG');
    expect(normalize(actual)).to.eql(expected);

    expected = 'apple/orange/grape/';
    actual = file.dir('./apple/orange/grape/');
    expect(normalize(actual)).to.eql(expected);
  });
});


describe('firstDir', function() {
  it('should return the first directory in the given file path (alias for firstSegment)', function() {
    var expected = 'apple';
    var actual = file.firstDir('apple/orange/file.ext');
    expect(actual).to.eql(expected);

    expected = 'grape';
    actual = file.firstDir('/grape/watermelon/quux');
    expect(actual).to.eql(expected);

    expected = 'banana';
    actual = file.firstDir('./banana/strawberry/quux/');
    expect(actual).to.eql(expected);
  });
});


describe('lastDir', function() {
  it('should return the last dirname in the given file path', function() {
    var expected = 'rectangle';
    var actual = file.lastDir('square/rectangle/file.ext');
    expect(normalize(actual)).to.eql(expected);

    expected = 'four';
    actual = file.lastDir('one/two/three/four');
    expect(normalize(actual)).to.eql(expected);

    expected = 'grape';
    actual = file.lastDir('apple/orange/grape/');
    expect(normalize(actual)).to.eql(expected);
  });
});


describe('lastChar:', function() {
  it('should return the last character in the given file path', function() {
    var expected = 't';
    var actual = file.lastChar('foo/bar/baz/quux/file.ext');
    expect(actual).to.eql(expected);

    expected = 'x';
    actual = file.lastChar('foo/bar/baz/quux');
    expect(actual).to.eql(expected);

    expected = '/';
    actual = file.lastChar('foo/bar/baz/quux/');
    expect(actual).to.eql(expected);
  });
});


describe('path.basename:', function () {
  it('should get the basename using the Node.js path module', function () {
    var expected = 'file.json';
    var actual = path.basename('path/to/file.json');
    expect(actual).to.eql(expected);

    expected = 'file.tmpl.md';
    actual = path.basename('path/to/file.tmpl.md');
    expect(actual).to.eql(expected);

    expected = 'file';
    actual = path.basename('path/to/file');
    expect(actual).to.eql(expected);

    expected = 'baz.quux';
    actual = path.basename('.foo.bar/baz.quux');
    expect(actual).to.eql(expected);

    expected = 'baz.quux.';
    actual = path.basename('.foo.bar/baz.quux.');
    expect(actual).to.eql(expected);

    expected = '.html';
    actual = path.basename('.html');
    expect(actual).to.eql(expected);

    expected = 'foo.bar.baz.quux';
    actual = path.basename('/foo.bar.baz.quux');
    expect(actual).to.eql(expected);

    expected = 'quux';
    actual = path.basename('/foo/bar/baz/asdf/quux');
    expect(actual).to.eql(expected);

    expected = 'quux.html';
    actual = path.basename('/foo/bar/baz/asdf/quux.html');
    expect(actual).to.eql(expected);

    expected = 'quux';
    actual = path.basename('/foo/bar/baz/quux');
    expect(actual).to.eql(expected);

    expected = 'quux';
    actual = path.basename('/foo/bar/baz/quux/');
    expect(actual).to.eql(expected);

    expected = 'quux';
    actual = path.basename('/quux');
    expect(actual).to.eql(expected);

    expected = 'quux';
    actual = path.basename('/quux/');
    expect(actual).to.eql(expected);

    expected = 'foo.bar.baz.quux';
    actual = path.basename('foo.bar.baz.quux');
    expect(actual).to.eql(expected);

    expected = 'baz.quux';
    actual = path.basename('foo.bar/baz.quux');
    expect(actual).to.eql(expected);

    expected = 'bar.baz.quux';
    actual = path.basename('foo/bar.baz.quux');
    expect(actual).to.eql(expected);

    expected = 'bar.baz.quux';
    actual = path.basename('foo/bar.baz.quux/');
    expect(actual).to.eql(expected);

    expected = 'quux';
    actual = path.basename('foo/bar.baz/quux');
    expect(actual).to.eql(expected);

    expected = 'baz.quux';
    actual = path.basename('foo/bar/baz.quux');
    expect(actual).to.eql(expected);

    expected = 'baz.quux.';
    actual = path.basename('foo/bar/baz.quux.');
    expect(actual).to.eql(expected);

    expected = 'quux';
    actual = path.basename('foo/bar/baz/quux');
    expect(actual).to.eql(expected);

    expected = 'quux';
    actual = path.basename('foo/bar/baz/quux/');
    expect(actual).to.eql(expected);

    expected = 'quux';
    actual = path.basename('foo\\bar\\baz\\quux\\');
    expect(actual).to.eql(expected);

    expected = 'quux';
    actual = path.basename('quux/');
    expect(actual).to.eql(expected);
  });


  it('should return the name of a file, exluding directories from the result', function () {
    var expected = 'file.json';
    var actual = file.filename('path/to/file.json');
    expect(actual).to.eql(expected);

    expected = 'file.tmpl.md';
    actual = file.filename('path/to/file.tmpl.md');
    expect(actual).to.eql(expected);

    expected = 'file';
    actual = file.filename('path/to/file');
    expect(actual).to.eql(expected);

    expected = 'baz.quux';
    actual = file.filename('.foo.bar/baz.quux');
    expect(actual).to.eql(expected);

    expected = 'baz.quux.';
    actual = file.filename('.foo.bar/baz.quux.');
    expect(actual).to.eql(expected);

    expected = '.html';
    actual = file.filename('.html');
    expect(actual).to.eql(expected);

    expected = 'foo.bar.baz.quux';
    actual = file.filename('/foo.bar.baz.quux');
    expect(actual).to.eql(expected);

    expected = 'quux';
    actual = file.filename('/foo/bar/baz/asdf/quux');
    expect(actual).to.eql(expected);

    expected = 'quux.html';
    actual = file.filename('/foo/bar/baz/asdf/quux.html');
    expect(actual).to.eql(expected);

    expected = 'quux';
    actual = file.filename('/foo/bar/baz/quux');
    expect(actual).to.eql(expected);

    // Different result than path.basename
    expected = '';
    actual = file.filename('/foo/bar/baz/quux/');
    expect(actual).to.eql(expected);

    expected = 'quux';
    actual = file.filename('/quux');
    expect(actual).to.eql(expected);

    // Different result than path.basename
    expected = '';
    actual = file.filename('/quux/');
    expect(actual).to.eql(expected);

    expected = 'foo.bar.baz.quux';
    actual = file.filename('foo.bar.baz.quux');
    expect(actual).to.eql(expected);

    expected = 'baz.quux';
    actual = file.filename('foo.bar/baz.quux');
    expect(actual).to.eql(expected);

    expected = 'bar.baz.quux';
    actual = file.filename('foo/bar.baz.quux');
    expect(actual).to.eql(expected);

    // Different result than path.basename
    expected = '';
    actual = file.filename('foo/bar.baz.quux/');
    expect(actual).to.eql(expected);

    expected = 'quux';
    actual = file.filename('foo/bar.baz/quux');
    expect(actual).to.eql(expected);

    expected = 'baz.quux';
    actual = file.filename('foo/bar/baz.quux');
    expect(actual).to.eql(expected);

    expected = 'baz.quux.';
    actual = file.filename('foo/bar/baz.quux.');
    expect(actual).to.eql(expected);

    expected = 'quux';
    actual = file.filename('foo/bar/baz/quux');
    expect(actual).to.eql(expected);

    // Different result than path.basename
    expected = '';
    actual = file.filename('foo/bar/baz/quux/');
    expect(actual).to.eql(expected);

    // Different result than path.basename
    expected = '';
    actual = file.filename('foo\\bar\\baz\\quux\\');
    expect(actual).to.eql(expected);

    // Different result than path.basename
    expected = '';
    actual = file.filename('quux/');
    expect(actual).to.eql(expected);
  });
});

describe('File name:', function() {
  it('should get the extension', function() {
    var expected = 'file.json';
    var actual = file.filename('path/to/file.json');
    expect(actual).to.eql(expected);
  });
});

describe('file extension:', function() {
  it('should get the extension', function() {
    var expected = 'json';
    var actual = file.ext('path/to/file.json');
    expect(actual).to.eql(expected);
  });

  it('should get the basename', function() {
    var expected = 'file';
    var actual = file.basename('path/to/file.json');
    expect(actual).to.eql(expected);
  });

  it('should get the base without extension', function() {
    var expected = 'file';
    var actual = file.base('path/to/file.json');
    expect(actual).to.eql(expected);
  });
});

describe('cwd:', function() {
  it('should change the cwd', function() {
    var expected = file.normalizeSlash(path.join(cwd, 'test', 'fixtures'));
    file.setCWD('test', 'fixtures');
    var actual = file.normalizeSlash(process.cwd());
    expect(actual).to.eql(expected);

    file.setCWD('..', '..');
  });
});
