/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Richard Petersen <richard.petersen@open-xchange.com>
 *
 */

(function (tinymce) {
    tinymce.PluginManager.add('oxdrop', function (editor) {

        editor.on('PreInit', function (e) {
            function onDrop(e) {
                var dataTransfer = e.originalEvent.dataTransfer;
                if (dataTransfer && _(dataTransfer.types).contains('text/uri-list')) {
                    var text = decodeURIComponent(dataTransfer.getData('text/uri-list')),
                        list = text.match(/([^,]("[^"]*")?)+/g);

                    _(list).each(function (item) {
                        // only use image tags
                        if (!/^<img /.test(item)) return;
                        var $elem = $(item);
                        if ($elem.attr('data-pending') !== 'true') return;
                        var id = $elem.data('id'),
                            img = $(editor.getElement()).find('img[data-id="' + id + '"]');
                        if (!img.length) return;
                        // remove data from dataTransfer instead of stop propagation
                        // therefore, tinymce executes drag and drop without content (especially sets the cursor)
                        dataTransfer.setData('text/html', '');
                        var html = img.prop('outerHTML');
                        _.delay(function () {
                            editor.insertContent(html);
                        }, 100);
                    });
                }
            };

            // capture drop event to prevent bug in tinymce paste plugin when a html image is dragged
            $(editor.contentDocument)
                .on('dragover', false)
                .on('drop', onDrop);

            require(['io.ox/core/dropzone', 'io.ox/core/notifications', 'gettext!io.ox/core'], function (dropzone, notifications, gt) {
                var target = $(editor.getContainer()),
                    zone = new dropzone.Inplace({
                        // any changes in this gt-call should also be changed in contenteditable.js, because create_pot collects the strings there
                        caption: gt('Drop inline images here'),
                        eventTarget: $(editor.contentDocument)
                    });

                zone.on({
                    'show': function () {
                        var container = $(editor.getContentAreaContainer()),
                            parentOffset = target.offset().top - target.parent().offset().top,
                            containerOffset = container.offset().top - target.offset().top;
                        zone.$el.css({ top: parentOffset + containerOffset, margin: 0 });
                    },
                    'drop': function (files) {
                        var hasNonImages = _(files).reduce(function (memo, file) {
                            return memo || !/^image/.test(file.type);
                        }, false);

                        if (hasNonImages) {
                            // any changes in this gt-call should also be changed in contenteditable.js, because create_pot collects the strings there
                            notifications.yell('error', gt('Please only drop images here. If you want to send other files, you can send them as attachments.'));
                        } else {
                            _(files).each(editor.plugins.oximage.uploadBlob);
                        }
                    }
                });

                target.after(zone.render().$el.addClass('abs'));
            });
        });

    });
}(window.tinymce));
