/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2013 Open-Xchange Inc., Tarrytown, NY, USA. info@open-xchange.com
 *
 * @author Matthias Biggeleben <matthias.biggeleben@open-xchange.com>
 */

define('io.ox/backbone/mini-views/date',
    ['io.ox/backbone/mini-views/abstract',
     'io.ox/core/date',
     'gettext!io.ox/core'
    ], function (AbstractView, date, gt) {

    'use strict';

    var titles = {
        year: gt('Year'),
        month: gt('Month'),
        date: gt('Day')
    };

    //
    // <div><select class="date"><select class="month"><select class="year"></div>
    //

    // core/date.js would be a better place but who cares
    function localize(name, text) {
        if (ox.language === 'ja_JP') {
            if (name === 'year') return text + '年';
            if (name === 'date') return text + '日';
        }
        return text;
    }

    // helper
    function createSelect(name, from, to, setter, format) {

        var node = $('<select tabindex="1">').attr({ name: name, title: titles[name] }),
            i = Math.min(from, to),
            $i = Math.max(from, to),
            d = new date.Local(0),
            options = [],
            empty, text;

        for (; i <= $i; i++) {
            setter.call(d, i);
            text = d.format(format);
            text = localize(name, text);
            options.push($('<option>').val(i).text(text));
        }

        // revert?
        if (from > to) {
            options.reverse();
        }

        // add empty option - do that after revert
        empty = $('<option>').text('');
        if (name === 'year') { empty.val('0001'); }
        options.unshift(empty);

        // append
        return node.append(options);
    }

    var DateView = AbstractView.extend({

        className: 'native-date-picker',
        events: { 'change select': 'onChange' },

        onChange: function () {
            var year = this.$el.find('.year').val(),
                month = this.$el.find('.month').val(),
                day = this.$el.find('.date').val();
            if (month !== '' && day !== '') {//year doesn't matter, it's always set
                //check if date is invalid(like feb 30) and prevent month jump
                var str = year + '-' + _.pad(parseInt(month, 10) + 1, 2) + '-' + _.pad(day, 2),
                    tempDate = date.UTC.parse(str, 'yyyy-M-d'),
                    tempMonth = tempDate.getMonth();
                if (tempMonth.toString() !== month) {
                    tempDate.setDate(0);//last valid day of previous month
                    //set dayfield to right day (needs to be done or an invalid date can be selected if model already has the corrected date)
                    this.$el.find('.date').val(tempDate.getDate());
                }
                this.model.set(this.name, tempDate.getTime());
            } else {
                this.model.set(this.name, null);
                this.$el.find('.date').children().prop('disabled', false);//enable all
            }
        },

        update: function () {
            var value = this.model.get(this.name), d, year, text;
            // change boxes only for valid dates
            if (_.isNumber(value)) {
                d = new Date(value);
                year = String(d.getFullYear());
                if (year !== '1') {
                    // if the year is not our dropdown we add it
                    var yearValues = [];
                    this.$el.find('.year option').each(function () {
                        yearValues.push($(this).val());
                    });
                    if (!_.contains(yearValues, year)) {
                        text = localize('year', year);
                        this.$el.find('.year').append(
                            $('<option>').val(year).text(text)
                        );
                    }
                }
                this.$el.find('.year').val(year);
                this.$el.find('.month').val(d.getMonth());
                this.$el.find('.date').val(d.getDate());
                // disable invalid dayfields
                d.setDate(1);
                d.setMonth(d.getMonth() + 1);
                d.setDate(0);
                var validDays = d.getDate(),
                    options = this.$el.find('.date').children().prop('disabled', false);
                options = options.slice(validDays + 1, options.length);
                options.prop('disabled', true);
            }
        },

        setup: function (options) {
            this.name = options.name;
            this.listenTo(this.model, 'change:' + this.name, this.update);
        },

        render: function () {

            var self = this;

            date.getFormat(date.locale.formats.yMMMd).replace(
                /(Y+|y+|u+)|(M+|L+)|(d+)|(?:''|'(?:[^']|'')*'|[^A-Za-z'])+/g,
                function (match, y, m, d) {
                    var proto = date.Local.prototype, node, year;
                    if (y) {
                        year = (new date.Local()).getYear();
                        node = createSelect('year', year, year - 150, proto.setYear, y).addClass('year');
                    } else if (m) {
                        node = createSelect('month', 0, 11, proto.setMonth, 'MMMM').addClass('month');
                    } else if (d) {
                        node = createSelect('date', 1, 31, proto.setDate, match).addClass('date');
                    }
                    self.$el.append(node);
                }
            );

            this.update();
            return this;
        }
    });

    return {
        DateView: DateView
    };
});
