/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Tobias Prinz <tobias.prinz@open-xchange.com>
 * @author Markus Bode <markus.bode@open-xchange.com>
 */

define('plugins/portal/twitter/register',
    ['io.ox/core/extensions',
     'io.ox/oauth/proxy',
     'io.ox/core/strings',
     'io.ox/keychain/api',
     'gettext!plugins/portal',
     'io.ox/core/notifications',
     'io.ox/core/date',
     'less!plugins/portal/twitter/style.less'], function (ext, proxy, strings, keychain, gt, notifications, date) {

    'use strict';

    var extensionId = 'twitter';
    var loadEntriesPerPage = 20;
    var offset = 0;
    var $tweets = $('<div>').addClass('twitter');
    var $busyIndicator = $('<div>').html('&nbsp;');

    var parseTweet = function (text, entities) {
        var offsets = {},
            linkMatches;

        _(entities.hashtags).each(function (hashtag) {
            var elem = $('<a>', {href: 'https://twitter.com/#!/search/%23' + hashtag.text, target: '_blank'}).text('#' + hashtag.text);
            offsets[hashtag.indices[0]] = {
                elem: elem,
                indices: hashtag.indices
            };
        });
        _(entities.urls).each(function (url) {
            var elem = $('<a>', {href: url.expanded_url, target: '_blank'}).text(url.display_url);
            offsets[url.indices[0]] = {
                elem: elem,
                indices: url.indices
            };
        });
        _(entities.user_mentions).each(function (user_mention) {
            var elem = $('<a>', {href: 'https://twitter.com/#!/' + user_mention.screen_name, target: '_blank'}).text('@' + user_mention.screen_name);
            offsets[user_mention.indices[0]] = {
                elem: elem,
                indices: user_mention.indices
            };
        });

        //hack to highligh some t.co-URLs that Twitter does not identify as such:
        linkMatches = text.match(/http:\/\/t\.co\/\w+/gi);
        _(linkMatches).each(function (myLink) {
            var index = text.indexOf(myLink),
                length = myLink.length;
            if (_(offsets).keys().indexOf(index.toString()) === -1) { //make sure there is nothing planned for this part already
                offsets[index] = {
                    elem: $('<a>', {href: myLink}).text(myLink),
                    indices: [index, index + length]
                };
            }
        });

        var keySet = _(offsets).keys().sort(function (a, b) {return a - b; });
        var bob = $('<span>');
        var cursor = 0;
        _(keySet).each(function (key) {
            var element = offsets[key];
            bob.append(text.substring(cursor, element.indices[0])).append(element.elem);
            cursor = element.indices[1];
        });

        if (cursor < text.length) {
            bob.append(text.substr(cursor, text.length));
        }

        return bob;
    };

    var loadFromTwitter = function (params) {
        var def = proxy.request({api: 'twitter', url: 'https://api.twitter.com/1/statuses/home_timeline.json', params: params});
        return def.pipe(function (response) {
            if (response) {
                var jsonResponse = JSON.parse(response);

                if (!jsonResponse.error) {
                    return jsonResponse;
                } else {
                    return {};
                }
            } else {
                return {};
            }
        });
    };

    var loadTweets = function (count, offset, newerThanId) {
        var params = {'include_entities': true};
        if (offset) {
            params.max_id = offset;
            // increment because max_id is inclusive and we're going to ignore the first tweet in our result
            params.count = count + 1;
        }

        if (newerThanId) {
            params.since_id = newerThanId;
        } else {
            params.count = count;
        }

        return loadFromTwitter(params);
    };

    var showTweet = function (tweet) {
        if (tweet.retweeted_status) {
            var $temp = renderTweet(tweet.retweeted_status);
            $temp.find('.text').append(
                $('<div class="io-ox-twitter-retweet-source">').append(
                    $('<i class="icon-retweet">'),
                    " ",
                    $('<span>').text(gt('Retweeted by %s', tweet.user.screen_name))
                )
            );
            return $temp;
        }
        return renderTweet(tweet);
    };

    function followButton(tweet) {
        var button_config = "show_count=false&align=right&show_screen_name=false&dnt=true";
        // add lang parameter (use the first 2 letters as language indicator for twitter
        button_config += "&lang=" + ox.language.split('_')[0];
        button_config += "&screen_name=" + tweet.user.screen_name;
        return $('<iframe>')
            .attr("src", "//platform.twitter.com/widgets/follow_button.html?" + button_config)
            .attr("allowtransparency", "true")
            .attr("frameborder", "0")
            .attr("scrolling", "no")
            .addClass("io-ox-twitter-follow");
    }

    function parseDate(str) {
        var v = str.split(' ');
        return new Date(Date.parse(v[1] + ' ' + v[2] + ', ' + v[5] + ' ' + v[3] + ' UTC'));
        // thx for having exactly the same problem:
        // http://stackoverflow.com/questions/3243546/problem-with-javascript-date-function-in-ie-7-returns-nan
    }

    var renderTweet = function (tweet) {
        var tweetLink = 'https://twitter.com/' + tweet.user.screen_name + '/status/' + tweet.id_str;
        var profileLink = 'https://twitter.com/' + tweet.user.screen_name;
        var tweeted = new date.Local(parseDate(tweet.created_at)).format(date.DATE_TIME);
        var $myTweet = $('<div class="tweet">').data('entry', tweet).append(
            followButton(tweet),
            $('<a>').attr({href: profileLink}).append(
                $('<img>', {src: tweet.user.profile_image_url_https, 'class': 'profilePicture', alt: tweet.user.description})
            ),
            $('<div class="text">').append(
                $('<strong class="io-ox-twitter-name">').text(tweet.user.name),
                '<br />',
                $('<a>', {'class': 'name', href: profileLink, target: '_blank'}).text('@' + tweet.user.screen_name),
                '<br />',
                parseTweet(tweet.text, tweet.entities)
            ),
            $('<div class="io-ox-twitter-details">').append(
                $('<a>').attr({'class': 'io-ox-twitter-date', 'href': tweetLink, 'target': '_blank'}).text(tweeted),
                $('<a>').attr({'class': 'io-ox-twitter-reply', 'href': 'https://twitter.com/intent/tweet?in_reply_to=' + tweet.id_str}).text(gt('Reply')),
                $('<a>').attr({'class': 'io-ox-twitter-retweet', 'href': "https://twitter.com/intent/retweet?tweet_id=" + tweet.id_str}).text(gt('Retweet')),
                $('<a>').attr({'class': 'io-ox-twitter-favorite', 'href': "https://twitter.com/intent/favorite?tweet_id=" + tweet.id_str}).text(gt('Favorite'))
            )
        );
        if (tweet.favorited) {
            $myTweet.find('.io-ox-twitter-favorite').addClass('favorited').text(gt('Favorited'));
        }
        if (tweet.retweeted) {
            $myTweet.find('.io-ox-twitter-retweet').addClass('retweeted').text(gt('Retweeted'));
        }
        return $myTweet;
    };

    var onPullToRefresh = function () {
        offset = 0;
        var $first = $('div.tweet:first');
        var newestId = $first.data('entry').id_str;
        $tweets.addClass('pulltorefresh-refreshing');

        $.when(loadTweets(0, 0, newestId), _.wait(500))
            .done(function (j) {
                j.reverse();
                _(j).each(function (tweet) {
                    showTweet(tweet).prependTo($tweets);
                });

                var $o = $('div.io-ox-sidepopup-pane');
                var top = $o.scrollTop() - $o.offset().top + $first.offset().top;
                $o.animate({scrollTop: top}, 250, 'swing');
                $tweets.removeClass('pulltorefresh-refreshing');
            })
            .fail(function () {
                $tweets.removeClass('pulltorefresh-refreshing');
                notifications.yell('error', gt('Could not load new tweets.'));
            });
    };

    var drawPreview = function (baton) {
        var content = baton.contentNode;
        if (baton.data.length === 0) {
            content.append(
                $('<div class="paragraph">').text(gt('No tweets yet.'))
            );

        } else if (baton.data.errors && baton.data.errors.length > 0) {
            content.removeClass('pointer');
            $('<div class="paragraph">').text(gt('Twitter reported the following errors:')).appendTo(content);
            _(baton.data.errors).each(function (myError) {
                $('<div class="error">').text("(" + myError.code + ") " + myError.message).appendTo(content);
                handleError(myError.code, this, baton).appendTo(content);
            });

        } else {
            content.addClass('pointer');
            _(baton.data).each(function (tweet) {
                var message = String(tweet.text).replace(/((#|@)[\wäöüß]+)/ig, '<span class="accent">$1</span>');
                content.append(
                    $('<div class="paragraph">').append(
                        $('<span class="bold">').text('@' + tweet.user.name + ': '),
                        $('<span class="normal">').html(message)
                    )
                );
            });
        }
    };

    var handleError = function (errorCode, baton) {
        if (errorCode === 32 || errorCode === 89 || errorCode === 135) {
            var account = keychain.getStandardAccount('twitter');

            return $('<a class="solution">').text(gt('Click to authorize your account again')).on('click', function () {
                keychain.submodules.twitter.reauthorize(account).done(function () {
                    console.log(gt("You have reauthorized this %s account.", 'Twitter'));
                }).fail(function () {
                    console.error(gt("Something went wrong reauthorizing the %s account.", 'Twitter'));
                });
            });
        } else if (errorCode === 88 || errorCode === 130) {
            return $('<a class="solution">').text(gt('Click to retry later.')).on('click', function () { keychain.submodules.twitter.trigger('update'); });
        } else {
            return $('<a class="solution">').text(gt('Click to retry')).on('click', function () { keychain.submodules.twitter.trigger('update'); });
        }
    };

    ext.point('io.ox/portal/widget/twitter').extend({

        title: gt('Twitter'),

        initialize: function (baton) {
            keychain.submodules.twitter.on('update create delete', function () {
                loadFromTwitter({ count: loadEntriesPerPage, include_entities: true }).done(function (data) {
                    baton.data = data;
                    if (baton.contentNode) {
                        baton.contentNode.empty();
                        drawPreview(baton);
                    }
                });
            });
        },

        action: function (baton) {
            window.open('https://twitter.com/', 'twitter');
        },

        isEnabled: function () {
            return keychain.isEnabled('twitter');
        },

        requiresSetUp: function () {
            return keychain.isEnabled('twitter') && ! keychain.hasStandardAccount('twitter');
        },

        performSetUp: function () {
            var win = window.open(ox.base + "/busy.html", "_blank", "height=400, width=600");
            return keychain.createInteractively('twitter', win);
        },

        load: function (baton) {
            return loadFromTwitter({ count: loadEntriesPerPage, include_entities: true }).done(function (data) {
                baton.data = data;
            });
        },

        preview: function (baton) {
            if (!baton.data) { return; }
            var content = $('<div class="content pointer">');
            baton.contentNode = content;
            drawPreview(baton);
            this.append(content);
        },


        draw: function (baton) {

            var timeline = baton.data;

            // Pull to refresh
            $(this).on('onResume', function () {
                $(this).on('onPullToRefresh', onPullToRefresh);
            }).on('onPause', function () {
                $(this).off('onPullToRefresh', onPullToRefresh);
                //ptr.detachEvents();
            }).on('onPullToRefreshDown', function () {
                $('div.tweet > div.text').addClass('pulltorefresh-unselectable');
                $('div.tweet > div.text > span').addClass('pulltorefresh-unselectable');
            }).on('onPullToRefreshUp', function () {
                $('div.tweet > div.text').removeClass('pulltorefresh-unselectable');
                $('div.tweet > div.text > span').removeClass('pulltorefresh-unselectable');
            }).on('onAppended', function () {
                //ptr.attachEvents($('div.io-ox-sidepopup-pane'), $(this), $tweets);
            });

            if (!timeline) {
                this.remove();
                return;
            }

            var script = document.createElement('script');
            script.type = 'text/javascript';
            script.src = 'https://platform.twitter.com/widgets.js'; //TODO must be stored locally, even if the Twitter guys hate us

            this.empty().append(
                $('<a>').text(gt('Tweet')).attr({
                    href: 'https://twitter.com/intent/tweet',
                    target: '_blank',
                    'class': 'twitter-share-button io-ox-twitter-action-tweet',
                    'data-count': 'none',
                    'data-size': 'large',
                    'data-lang': ox.language.split('_')[0],
                    'data-url': ' ', //need to provide %20 here, so twitter detects, we set some data
                    'data-text': ' ' //there must be text in the text attribute, may be we could add an input field to the sidebar
                                      //to pre-fill the popup, leaving the default message empty for now
                }),
                $('<div>').addClass('clear-title').text('Twitter'),
                script
            );

            this.append($tweets.empty(), $busyIndicator);

            _(timeline).each(function (tweet) {
                offset = tweet.id_str;
                showTweet(tweet).appendTo($tweets);
            });
        },

        loadMoreResults: function (finishFn) {
            $busyIndicator.addClass('io-ox-busy');

            $.when(loadTweets(loadEntriesPerPage, offset), _.wait(500))
                .done(function (j) {
                    j = j.slice(1);
                    _(j).each(function (tweet) {
                        offset = tweet.id_str;
                        showTweet(tweet).appendTo($tweets);
                    });
                    finishFn($busyIndicator);
                })
                .fail(function () {
                    finishFn($busyIndicator);
                });
        },

        drawCreationDialog: function () {
            var $node = $(this);
            $node.append(
                $('<h1>').text('Twitter'),
                $('<div class="io-ox-portal-preview centered">').append(
                    $('<div>').text(gt('Add your account'))
                )
            );
        },

        error: function (error) {
            if (error.code !== "OAUTH-0006")
                return; //let the default handling do the job

            $(this).empty().append(
                $('<h2>').text(gt('Twitter')),
                $('<div class="content">').text(gt('Click here to add your account'))
                .on('click', {}, function () {
                    ext.point('io.ox/portal/widget/twitter').invoke('performSetUp');
                })
            );
            console.log("DEBUG", error);
        }
    });

    ext.point('io.ox/portal/widget/twitter/settings').extend({
        title: gt('Twitter'),
        type: 'twitter',
        editable: false,
        unique: true
    });
});
