###################################################################
#
# Basic LDAP class
#
###################################################################


package OX::LDAPConnector::ldap;

use strict;
use warnings;
use Net::LDAP;
use Net::LDAP::Control::Paged;
use Data::Dumper;
use OX::Config::Reader;
use POSIX qw(strftime);
use Log::Log4perl qw(:easy);

sub new()
{
    my $class = shift;
    my $ldapuri = shift;
    my $config = shift;
    my $mesg;

    $config->{ldap} = Net::LDAP->new("$ldapuri") or die("Connection to LDAP server failed!\n");

    if( !defined($config->{ldapuserdn}) || length($config->{ldapuserdn}) == 0 ) {
	$mesg = $config->{ldap}->bind(noauth => 1);
    } else {
	$mesg = $config->{ldap}->bind($config->{ldapuserdn}, password => $config->{ldapuserpassword});
    }

    $mesg->code && die  "Error on bind: $@ : ".$mesg->error;

    bless ($config,$class);

    return $config;
}

#######################################
#
# Returns current Groups in grouphash
#
#######################################

sub getGroups()
{
    my $self = shift;
    return $self->{grouphash};
}
#######################################
#
# Starts user search
#
#######################################
sub searchusers()
{
    my($self) = @_;

    my $ldap = $self->{ldap};
    my $basedn = $self->{userbasedn};

    my $userfilter = $self->{userfilter};

    $self->{usersearchtime} = strftime("%Y%m%d%H%M%SZ", gmtime);


    my $mesg = $self->subsearch($ldap, $basedn, "$userfilter");

    $mesg->code && die  "Error on search: $@ : ".$mesg->error;

    $self->{UserSearchObj} = $mesg;

    if (lc($self->{updategroups}) eq "yes")
    {
	$self->makegrouphash();
    }
}

######################################
#
# Basic search func
#
######################################
sub subsearch()
{	
    my $class = shift;
    my $ldap = shift;
    my $basedn = shift;
    my $filter = shift;

    my $scope = $class->{'ldapsearchscope'} || 'sub';

    my $mesg = $ldap->search(
                base => $basedn,
                filter => $filter,
		attrs => ['*', 'modifyTimestamp'],
                scope => $scope,
                );
    return $mesg;
}
#################################################
#
# pops one user entry with special grouphandling
#
#################################################

sub user_pop_entry()
{
    my $self = shift;
    my $basedn = $self->{userbasedn};
    my $page = $self->{page};
    my $ldap = $self->{ldap};

    my $searchobj = $self->{UserSearchObj};

    my %grouphash = ();
    my $entry;

    if ( $entry = $searchobj->pop_entry()) {

        if (lc($self->{updategroups}) eq "yes")
	{
	    if ($self->{grouphash})
	    {
	        %grouphash = %{$self->{grouphash}};
	    }
	    if (my $groupname = $self->groupName_for_gid($entry->get_value($self->{userprimarygroupattribute})))
	    {
		push ( @{$grouphash{$groupname}{member} }, $entry->get_value($self->{uidattribute})) if (! (grep {$_ eq $entry->get_value($self->{uidattribute})} @{$grouphash{$groupname}{member} }));	    
	    }

	    %{$self->{grouphash}} = %grouphash;	
	}

	return $entry;
    } else {
	return undef;
    }

}

##################################################
#
# Starts groupsearch
#
##################################################

sub searchgroups()
{
    my ($self) = @_;

    my $ldap = $self->{ldap};
    my $basedn = $self->{groupbasedn};

    my $mesg = $self->subsearch($ldap, $basedn, $self->{groupfilter});
    $mesg->code && die "Error on search $@ : " . $mesg->error;

    $self->{GroupSearchObj} = $mesg;
}

##################################################
#
# searches a group name by gidnumber
#
##################################################
sub groupName_for_gid()
{
    my ($self) = @_;
    
    my $class = shift;
    my $gid = shift;

    return undef if ! $gid;

    my %grouphash = %{$self->{grouphash}};
    
    foreach my $key (keys%grouphash)
    {
	return undef if (! defined($grouphash{$key}{'number'}));
	if ($grouphash{$key}{'number'} == $gid)
	{
	    return $key;
	}
    }
    return undef;
}

################################################
#
# builds the grouphash with all found groups
#
################################################
sub makegrouphash()
{
    my ($self) = @_;
    
    my $class = shift;

    my %grouphash = ();

    if ($self->{grouphash})
    {
        %grouphash = $self->{grouphash};
    }

    $self->searchgroups();

    while (defined(my $entry = $self->group_pop_entry()))
    {
	my $members = $entry->get_value($self->{groupmemberattribute}, asref => 1);
	$grouphash{$entry->get_value($self->{groupnameattribute})}{'number'} = $self->makeGid($entry->get_value($self->{groupnumberattribute}));
	$grouphash{$entry->get_value($self->{groupnameattribute})}{'displayname'} = $entry->get_value($self->{groupdisplaynameattribute});

	foreach my $member (@$members)
	{	
	    if (lc($self->{memberattributeisdn}) eq "yes")
	    {
		$member = $self->getUid($member);
	    }
	    if (defined ($member))
	    {
    		push @{ $grouphash{$entry->get_value($self->{groupnameattribute})}{member} }, $member;
	    }
	}
    }
    %{$self->{grouphash}} = %grouphash;
#    print(Dumper($self->{grouphash}));die("\n");
}

###################################################
#
# Returns the group id in human readable form
#
###################################################

sub makeGid()
{
    my $self = shift;
    my $gid = shift;

    return $gid;
}

####################################################
#
# returns the uid from a distinguished name
#
####################################################
sub getUid()
{
    my $self = shift;
    my $dn = shift;

    my $ldap = $self->{ldap};
    my $basedn = $self->{userbasedn};

    my $filter = "dn=$dn";
    my $mesg = $self->subsearch($ldap, $basedn, $filter);

    if (my $entry = $mesg->pop_entry())
    {
#	print($entry->get_value($self->{uidattribute})."\n");
	return $entry->get_value($self->{uidattribute});
    }
}

#    my $self = shift;
#    my $dn = shift;
#
#    my @chars = split(//,$dn);
#    
#    my @uid;
#
#    my $firstchar = 0;
#    my $ignore = 0;
#
#    foreach my $char (@chars)
#    {
#	if (! $firstchar )
#	{
#	    if ($char eq "=")
#	    {
#		$firstchar = 1;
#	    }
#	    next;
#	} elsif ($char eq "\\")
#	{
#	    $ignore = 1;
#	    next;
#	} elsif ($char eq "," && ! $ignore)
#	{
#	    last;
#	} elsif ($ignore) {
#	    $ignore = 0;
#	}
#	push @uid , $char;
#    }
#
#    return join("",@uid);
#}

############################################
#
# pops one group entry
#
############################################
sub group_pop_entry()
{
    my($self) = @_;
    my $basedn = $self->{userbasedn};
    my $ldap = $self->{ldap};

    my $searchobj = $self->{GroupSearchObj};

    my $adentry;

    if ( $adentry = $searchobj->pop_entry()) {
	return $adentry;
    } else {
	return undef;
    }

}

sub saveLastUsertime()
{
    my $self = shift;
    my $vardir = $self->{vardirectory};
    LOGDIE("Var directory not defined! Please check configuration!\n") if (! defined($vardir));
    if (! -e "$vardir/timestate") 
    {
	LOGDIE ("Can not write to $vardir\n") if (! -w $vardir);

	open (my $FILE, "> $vardir/timedate");
	print ($FILE "usertime = ".$self->{usersearchtime});
	close($FILE);
    }
}

sub getLastUsertime()
{
    my $self = shift;
    my $vardir = $self->{vardirectory};
    LOGDIE ("Var directory not defined! Please check configuration!\n") if (! defined($vardir));
    if (! -e "$vardir/timestate") 
    {
	LOGDIR ("Can not write to $vardir\n") if (! -w $vardir);
    }

    my $timedata = OX::Config::Reader::readConfig($vardir."/timedate");
    $timedata->{usertime} = undef if (!$timedata->{usertime});

    return $timedata->{usertime};
}


1;
