###########################################################
#
#
# Class for Active Directory specific handling
#
#
###########################################################


package OX::LDAPConnector::ads;

use strict;
use warnings;
use Net::LDAP;
use Net::LDAP::Control::Paged;
use Net::LDAP::Constant qw( LDAP_CONTROL_PAGED );
use Data::Dumper;
use OX::LDAPConnector::ldap;
use POSIX qw(strftime);
use Log::Log4perl qw(:easy);

our @ISA = qw(OX::LDAPConnector::ldap);

my $page ;
my $cookie;
my $count = 0;


sub new()
{
    my ($class, %args) = @_;

    my $this = $class->SUPER::new(%args);

    if (! $this->{userfilter})
    {
        $this->{userfilter} = "(objectClass=user)";
    }
    
    if (! $this->{groupfilter})
    {
	$this->{groupfilter} = "(objectClass=group)";
    }

    bless ($this,$class);
    return $this;
}

#########################################################
#
# Overwritten searchuser function
#
########################################################

sub searchusers()
{
    my $self = shift;

    my $ldap = $self->{ldap};
    my $page = $self->{userpage};
    my $basedn = $self->{userbasedn};

    $self->{usersearchtime} = strftime("%Y%m%d%H%M%SZ", gmtime);

    $page = Net::LDAP::Control::Paged->new( size => 2 );
    my $mesg;
    ($mesg, $page) = $self->subsearch($ldap, $basedn, $self->{userfilter}, $page);

    $self->{userpage} = $page;
    $mesg->code && LOGDIE( "Error on search: $@ : ".$mesg->error);

    $self->{UserSearchObj} = $mesg;

    # Get all groups now to minimize ldap queries and primarygroupid to group mapping
    if (lc($self->{updategroups}) eq "yes")
    {
        $self->makegrouphash();
    }
}

##################################################################
#
# Search all users, overwritte because of page control
#
##################################################################

sub subsearch()
{	
    my $self = shift;
    my $ldap = shift;
    my $basedn = shift;
    my $filter = shift;
    my $page = shift;

    my $scope = $self->{'ldapsearchscope'} || 'sub';

    my $mesg = $ldap->search(
                base => $basedn,
                filter => $filter,
                control => [ $page ],
		scope => $scope,
		attrs => ['*', 'modifyTimestamp' ]
                );

    $mesg->code && LOGDIE( "Error on search: $@ : ".$mesg->error);


    return ($mesg, $page);
}

################################################################
#
# pops one user entry, overwritten because of page control
#
################################################################
sub user_pop_entry()
{
    my $self = shift;
    my $basedn = $self->{userbasedn};
    my $page = $self->{userpage};
    my $ldap = $self->{ldap};
    my $cookie = $self->{cookie};
    my $searchobj = $self->{UserSearchObj};
    my $adentry;
    my $retpage;
    my %grouphash = ();

    while (! defined($adentry = $searchobj->pop_entry())) {
	my ($resp) = $searchobj->control( LDAP_CONTROL_PAGED ) or return undef;
	$cookie    = $resp->cookie or return undef;

	$page->cookie($cookie);
	$self->{cookie} = $cookie;

	($searchobj, $retpage) = $self->subsearch($ldap, $basedn, $self->{userfilter}, $page);
	$self->{userpage} = $retpage;
	$self->{UserSearchObj} = $searchobj;
    }
    if ($adentry)
    {
        if (lc($self->{updategroups}) eq "yes")
        {
	    # Build the objectSid for primary group of user
	    my (@unpack) = unpack( "H2 H2 n N V*", $adentry->get_value("objectSid"));
	    my ( $sid_rev, $num_auths, $id1, $id2, @ids ) = (@unpack);
	    my $user_sid_string = join( "-", "S", $sid_rev, ( $id1 << 32 ) + $id2, @ids );
	    my $gid = $adentry->get_value("primaryGroupID");
	    ( my $group_sid_string = $user_sid_string ) =~ s/\-[^\-]+$/-$gid/;

            if ($self->{grouphash})
            {
                %grouphash = %{$self->{grouphash}};
            }
            if (my $groupname = $self->groupName_for_gid($group_sid_string))
            {
                push @{$grouphash{$groupname}{member} }, $adentry->get_value($self->{uidattribute});
            }

            %{$self->{grouphash}} = %grouphash;
        }
        return $adentry;
    } else {
	return undef;
    }
}


############################################################
#
# Searches groups, overwritten because of page control
#
############################################################
sub searchgroups()
{
    my $self = shift;

    my $ldap = $self->{ldap};
    my $basedn = $self->{groupbasedn};
    my $page = $self->{grouppage};

    $page = Net::LDAP::Control::Paged->new( size => 2 );
    my $mesg;
    ($mesg, $page) = $self->subsearch($ldap, $basedn, $self->{groupfilter}, $page);
    $mesg->code && die "Error on search $@ : " . $mesg->error;

    $self->{GroupSearchObj} = $mesg;
    $self->{grouppage} = $page;
}

#############################################################
#
# Pops one group entry, overwritten because of page control
#
#############################################################

sub group_pop_entry()
{
    my $self = shift;
    my $basedn = $self->{userbasedn};
    my $page = $self->{grouppage};
    my $ldap = $self->{ldap};
    my $cookie = $self->{cookie};
    my $searchobj = $self->{GroupSearchObj};
    my $adentry;
    my $retpage;


    while (! defined($adentry = $searchobj->pop_entry())) {
	my ($resp) = $searchobj->control( LDAP_CONTROL_PAGED ) or return undef;
	$cookie    = $resp->cookie or return undef;

	$page->cookie($cookie);
	$self->{cookie} = $cookie;

	($searchobj, $retpage) = $self->subsearch($ldap, $basedn, $self->{groupfilter}, $page);
	$self->{grouppage} = $retpage;
	$self->{GroupSearchObj} = $searchobj;
    }
    if ($adentry)
    {
        return $adentry;
    } else {
	return undef;
    }
}


###############################################################################
#
# Forms the objectsid of group into human readable format
#
###############################################################################
sub makeGid()
{
    my $self = shift;
    my $gid = shift;
    
    my (@unpack) = unpack( "H2 H2 n N V*", $gid);
#    print Dumper(@unpack);
    my ( $sid_rev, $num_auths, $id1, $id2, @ids ) = (@unpack);
    my $group_sid_string = join( "-", "S", $sid_rev, ( $id1 << 32 ) + $id2, @ids );

    return $group_sid_string;
}

########################################################################
#
# Gets the groupname for gid, overwritten because of string comparison
#
########################################################################

sub groupName_for_gid()
{
    my ($self) = @_;

    my $class = shift;
    my $gid = shift;

    return undef if ! $gid;

    my %grouphash = %{$self->{grouphash}};

    foreach my $key (keys%grouphash)
    {
        if ($grouphash{$key}{'number'} eq $gid)
        {
            return $key;
        }
    }
    return undef;
}

sub getUid()
{
    my $self = shift;
    my $dn = shift;

    if (! defined($self->{dnusermap}{$dn}))
    {
        my $ldap = $self->{ldap};
	my $basedn = $self->{userbasedn};
	my $page = Net::LDAP::Control::Paged->new( size => 2 );
	my $mesg;
        ($mesg, $page) = $self->subsearch($ldap, $basedn, $self->{userfilter}, $page);

	$dn =~ s/\*/\\2a/g;
	$dn =~ s/\(/\\28/g;
	$dn =~ s/\)/\\29/g;
	$dn =~ s/\\/\\5c/g;
	$dn =~ s/\//\\2f/g;

        my $filter = "distinguishedName=".$dn;
        ($mesg, $page) = $self->subsearch($ldap, $basedn, $filter, $page);
        if (my $entry = $mesg->pop_entry())
	{
	    $self->{dnusermap}{$dn} = $entry->get_value($self->{uidattribute});
	}
    }
    return $self->{dnusermap}{$dn};

}


1;
