/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2010 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */
package com.openexchange.usm.syncml.sync;

import java.util.*;

import org.apache.commons.logging.Log;

import com.openexchange.usm.api.contenttypes.ContentType;
import com.openexchange.usm.api.exceptions.USMException;
import com.openexchange.usm.api.session.*;
import com.openexchange.usm.syncml.*;
import com.openexchange.usm.syncml.commands.*;
import com.openexchange.usm.syncml.data.SyncMLProtocolInfo;
import com.openexchange.usm.syncml.elements.*;
import com.openexchange.usm.syncml.elements.xml.SimpleXMLPart;
import com.openexchange.usm.syncml.mapping.ContentTypeMapping;
import com.openexchange.usm.syncml.mapping.MappingFailedException;
import com.openexchange.usm.syncml.servlet.SyncMLRequest;
import com.openexchange.usm.syncml.util.XmlSizeSupport;

/**
 * This class contains all accumulated information that needs to be accessed during a sync session. Since
 * a SyncML sync session is executed over several http requests, this data is stored in the HttpSession.
 * 
 * @author afe
 *
 */
public class SyncInformation {
	// Global information
	private final SyncType _syncType;

	// OX specific information
	private final String _oxFolderID;
	private final ContentType _contentType;
	private final long _lastServerTimestamp;
	private long _nextServerTimestamp;

	private ContentTypeMapping _mapping;
	private DataObject[] _newSyncState;
	private DataObject[] _clientResults;
	private int _clientResultsSent;

	// Used to store list of requests that are already transmitted by the client in case the client splits its sync into multiple requests
	private DataObject[] _clientRequests;

	private boolean _syncStateChanged;

	// SyncML specific information
	private final String _clientLocURI;
	private final String _serverLocURI;
	private final String _lastClientAnchor;
	private final String _nextClientAnchor;
	private final String _lastServerAnchor;
	private final String _nextServerAnchor;

	// Used to create response Sync command(s)
	private Target _target;
	private Source _source;

	/**
	 * Initially store all fixed information about this Sync
	 * 
	 * @param syncType
	 * @param clientLocURI
	 * @param serverLocURI
	 * @param oxFolder
	 * @param lastClientAnchor
	 * @param nextClientAnchor
	 * @param lastServerAnchor
	 * @param nextServerAnchor
	 * @param lastServerTimestamp
	 */
	public SyncInformation(SyncType syncType, String clientLocURI, String serverLocURI, Folder oxFolder,
			String lastClientAnchor, String nextClientAnchor, String lastServerAnchor, String nextServerAnchor,
			long lastServerTimestamp) {
		_syncType = syncType;
		_clientLocURI = clientLocURI;
		_serverLocURI = serverLocURI;
		_oxFolderID = oxFolder.getID();
		_contentType = oxFolder.getElementsContentType();
		_lastClientAnchor = lastClientAnchor;
		_nextClientAnchor = nextClientAnchor;
		_lastServerAnchor = lastServerAnchor;
		_nextServerAnchor = nextServerAnchor;
		_lastServerTimestamp = lastServerTimestamp;
	}

	public long getNextServerTimestamp() {
		return _nextServerTimestamp;
	}

	public String getClientLocURI() {
		return _clientLocURI;
	}

	public String getServerLocURI() {
		return _serverLocURI;
	}

	public String getOxFolderID() {
		return _oxFolderID;
	}

	public String getLastClientAnchor() {
		return _lastClientAnchor;
	}

	public String getNextClientAnchor() {
		return _nextClientAnchor;
	}

	public String getLastServerAnchor() {
		return _lastServerAnchor;
	}

	public String getNextServerAnchor() {
		return _nextServerAnchor;
	}

	public long getLastServerTimestamp() {
		return _lastServerTimestamp;
	}

	public SyncType getSyncType() {
		return _syncType;
	}

	public ContentType getContentType() {
		return _contentType;
	}

	public DataObject[] getNewSyncState() {
		return _newSyncState;
	}

	public void setSyncStateChanged() {
		_syncStateChanged = true;
	}

	public void updateFromSyncResult(SyncResult result, ContentTypeMapping mapping, Target target, Source source) {
		_nextServerTimestamp = result.getTimestamp();
		_newSyncState = result.getNewState();
		_clientResults = result.getChanges();
		_clientResultsSent = -1; // Initialize with -1, we want to send at least 1 Sync command for each client Sync command
		_mapping = mapping;
		_target = target;
		_source = source;
		_clientRequests = null;
		// Sort array of client changes: first deletions, then modifications, then creations
		Arrays.sort(_clientResults, ClientResultComparator.getInstance());
	}

	public boolean updateUSMStorage(Session usmSession, Log journal) throws USMException {
		if (_newSyncState == null) {
			if (journal.isDebugEnabled())
				journal.debug(usmSession + ": No new SyncState (?), removing " + this);
			return true;
		}
		for (DataObject o : _newSyncState) {
			if (!(o.getProtocolInformation() instanceof SyncMLProtocolInfo)) {
				if (journal.isDebugEnabled())
					journal.debug(usmSession + ": " + this + " not finished, no Client ID for " + o);
				return false;
			}
		}
		if (journal.isDebugEnabled())
			journal.debug(usmSession + ": " + this + " finished");
		if (_syncStateChanged) {
			_nextServerTimestamp = usmSession.storeSyncState(_lastServerTimestamp, _nextServerTimestamp, _oxFolderID,
					_newSyncState);
			if (journal.isDebugEnabled())
				journal.debug(usmSession + ": Modified SyncState saved for " + this);
		}
		return true;
	}

	public boolean isNew() {
		return _clientResults == null;
	}

	public boolean isIncomplete() {
		return isNew() || _clientResultsSent < _clientResults.length;
	}

	public int addSyncCommands(SyncMLRequest syncMLRequest, int currentMessageSize, int maxMessageSize) {
		if (_clientResults == null)
			return currentMessageSize;
		if (_clientResultsSent < 0)
			_clientResultsSent = 0;
		List<SyncMLCommand> subCommands = new ArrayList<SyncMLCommand>();
		// hadCommand is used to ensure that at least 1 object is sent even if that object is too large for MaxMsgSize
		for (boolean hadCommand = false; _clientResultsSent < _clientResults.length; _clientResultsSent++) {
			SyncMLCommand command = createResponse(syncMLRequest, _clientResults[_clientResultsSent]);
			if (command != null) {
				command.setCmdID("9999"); // Set command id to some random large value to allow computation of command size, will be overwritten in SyncMLRequest with final correct command id
				int commandSize = XmlSizeSupport.computeSize(command, syncMLRequest.isWbxml(), false);
				if (hadCommand && currentMessageSize + commandSize >= maxMessageSize)
					break;
				subCommands.add(command);
				hadCommand = true;
				currentMessageSize += commandSize;
			}
		}
		syncMLRequest.addCommandToResponse(new Sync(false, null, _target, _source, null, subCommands));
		return currentMessageSize;
	}

	private SyncMLCommand createResponse(SyncMLRequest syncMLRequest, DataObject dataObject) {
		try {
			ChangeState changeState = dataObject.getChangeState();
			if (changeState == ChangeState.DELETED)
				return new Delete(computeTarget(dataObject));
			if (changeState != ChangeState.CREATED && changeState != ChangeState.MODIFIED)
				return null;
			Meta meta = new Meta(new SimpleXMLPart(SyncMLConstants.TYPE, SyncMLConstants.NS_SYNCML_METINF, _mapping
					.getMimeType()));
			String data = _mapping.mapToText(dataObject);
			return (changeState == ChangeState.CREATED) ? new Add(meta, data, dataObject.getID()) : new Replace(meta,
					data, computeTarget(dataObject));
		} catch (MappingFailedException e) {
			// TODO Other log level ?
			syncMLRequest.getJournal().info(
					syncMLRequest.getUSMSession() + ": Mapping failed for object ID " + dataObject.getID(), e);
			return null;
		}
	}

	private Target computeTarget(DataObject o) throws MappingFailedException {
		if (o != null) {
			ProtocolInformation pi = o.getProtocolInformation();
			if (pi instanceof SyncMLProtocolInfo) {
				SyncMLProtocolInfo info = (SyncMLProtocolInfo) pi;
				return new Target(info.getDeviceObjectID(), null);
			}
		}
		throw new MappingFailedException(-1, SyncMLStatusCode.COMMAND_FAILED,
				"Couldn't determine client id for DataObject");
	}

	@Override
	public String toString() {
		return "SyncInfo(Timestamp: " + _oxFolderID + ", " + _lastServerTimestamp + "->" + _nextServerTimestamp
				+ ", Client: " + _lastClientAnchor + "->" + _nextClientAnchor + ", Server: " + _lastServerAnchor + "->"
				+ _nextServerAnchor + ", " + _clientResultsSent + " of "
				+ ((_clientResults == null) ? -1 : _clientResults.length) + " sent, internal data changed: "
				+ _syncStateChanged + ")";
	}

	public void setClientRequests(DataObject[] clientRequests) {
		_clientRequests = clientRequests;
	}

	public DataObject[] getClientRequests() {
		return _clientRequests;
	}
}
