/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2010 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */
package com.openexchange.usm.syncml.parser;

import java.io.IOException;

import org.apache.commons.logging.Log;
import org.kxml2.io.KXmlSerializer;
import org.kxml2.wap.Wbxml;
import org.kxml2.wap.WbxmlParser;
import org.xmlpull.v1.*;

import com.openexchange.usm.syncml.SyncMLConstants;

public class XmlPullParserUtil {

	public static int skipWhitespace(XmlPullParser parser) throws XmlPullParserException, IOException {
		int eventType = parser.getEventType();
		while (eventType == XmlPullParser.TEXT && parser.isWhitespace()) { // skip whitespace
			if (parser.getText().trim().length() > 0)
				break;
			eventType = parser.next();
		}
		return eventType;
	}

	public static boolean isAtStartTag(XmlPullParser parser, String tag) throws XmlPullParserException, IOException {
		skipWhitespace(parser);
		return XmlPullParser.START_TAG == parser.getEventType() && tag.equals(parser.getName());
	}

	public static boolean isAtEndTag(XmlPullParser parser, String tag) throws XmlPullParserException, IOException {
		skipWhitespace(parser);
		return XmlPullParser.END_TAG == parser.getEventType() && tag.equals(parser.getName());
	}

	public static void parseEndTag(XmlPullParser parser, String string) throws XmlPullParserException, IOException {
		skipWhitespace(parser);
		parser.require(XmlPullParser.END_TAG, null, string);
		parser.next();
	}

	public static String getNextTextValue(XmlPullParser parser, String tag, String expectedValue, Log journal)
			throws XmlPullParserException, IOException {
		parseStartTag(parser, tag);
		String string = parser.getText();
		if (!expectedValue.equals(string))
			journal.warn("Unexpected value <" + string + "> (expected <" + expectedValue + ">) in tag " + tag);
		parser.next();
		parseEndTag(parser, tag);
		return string;
	}

	public static String readText(XmlPullParser parser, String tag) throws XmlPullParserException, IOException {
		String string = readOptionalText(parser, tag);
		if (string == null)
			throw new XmlPullParserException("Required field missing: " + tag);
		return string;
	}

	public static int readInteger(XmlPullParser parser, String tag, int minVal, int maxVal)
			throws XmlPullParserException, IOException {
		String text = readText(parser, tag);
		try {
			int i = Integer.parseInt(text);
			if (i >= minVal && i <= maxVal)
				return i;
		} catch (NumberFormatException ignored) {
		}
		throw new XmlPullParserException("Unexpected value <" + text + "> (expected between " + minVal + " and "
				+ maxVal + ") for tag " + tag);
	}

	public static String readOptionalText(XmlPullParser parser, String tag) throws XmlPullParserException, IOException {
		String string = null;
		if (isAtStartTag(parser, tag)) {
			requireStartTag(parser, tag);
			if ((parser.nextToken() == Wbxml.EXT_I_0) && (parser instanceof WbxmlParser)) {
				WbxmlParser p2 = (WbxmlParser) parser;
				if (p2.getWapCode() == Wbxml.OPAQUE) {
					Object o = p2.getWapExtensionData();
					if (o instanceof byte[])
						string = new String((byte[]) o, "UTF-8");
				}
			} else {
				string = parser.getText();
			}
			parser.next();
			parseEndTag(parser, tag);
		}
		return string;
	}

	public static boolean readOptionalFlag(XmlPullParser parser, String tag) throws XmlPullParserException, IOException {
		if (!isAtStartTag(parser, tag))
			return false;
		parseStartTag(parser, tag);
		parseEndTag(parser, tag);
		return true;
	}

	public static void parseStartTag(XmlPullParser parser, String string) throws XmlPullParserException, IOException {
		requireStartTag(parser, string);
		parser.next();
	}

	private static void requireStartTag(XmlPullParser parser, String string) throws XmlPullParserException, IOException {
		skipWhitespace(parser);
		parser.require(XmlPullParser.START_TAG, null, string);
	}

	public static void writeOptionalFlag(XmlSerializer serializer, String tag, boolean set) throws IOException {
		if (set) {
			serializer.startTag(SyncMLConstants.SYNCML_NS, tag);
			serializer.endTag(SyncMLConstants.SYNCML_NS, tag);
		}
	}

	public static void writeOptionalText(XmlSerializer serializer, String tag, String value)
			throws IllegalArgumentException, IllegalStateException, IOException {
		if (value != null)
			writeText(serializer, tag, value);
	}

	public static void writeText(XmlSerializer serializer, String tag, String value) throws IllegalArgumentException,
			IllegalStateException, IOException {
		serializer.startTag(SyncMLConstants.SYNCML_NS, tag);
		serializer.text(value);
		serializer.endTag(SyncMLConstants.SYNCML_NS, tag);
	}

	public static void writeText(XmlSerializer serializer, String tag, String value, String ns)
			throws IllegalArgumentException, IllegalStateException, IOException {
		if (serializer instanceof KXmlSerializer) {
			serializer.setPrefix("", ns);
		}
		serializer.startTag(ns, tag);
		serializer.text(value);
		serializer.endTag(ns, tag);
	}
}
