package com.openexchange.usm.syncml.mapping.vcard;

import java.io.ByteArrayOutputStream;
import java.io.UnsupportedEncodingException;
import java.util.*;

import com.openexchange.usm.syncml.SyncMLConstants;

/*
*
*    OPEN-XCHANGE legal information
*
*    All intellectual property rights in the Software are protected by
*    international copyright laws.
*
*
*    In some countries OX, OX Open-Xchange, open xchange and OXtender
*    as well as the corresponding Logos OX Open-Xchange and OX are registered
*    trademarks of the Open-Xchange, Inc. group of companies.
*    The use of the Logos is not covered by the GNU General Public License.
*    Instead, you are allowed to use these Logos according to the terms and
*    conditions of the Creative Commons License, Version 2.5, Attribution,
*    Non-commercial, ShareAlike, and the interpretation of the term
*    Non-commercial applicable to the aforementioned license is published
*    on the web site http://www.open-xchange.com/EN/legal/index.html.
*
*    Please make sure that third-party modules and libraries are used
*    according to their respective licenses.
*
*    Any modifications to this package must retain all copyright notices
*    of the original copyright holder(s) for the original code used.
*
*    After any such modifications, the original and derivative code shall remain
*    under the copyright of the copyright holder(s) and/or original author(s)per
*    the Attribution and Assignment Agreement that can be located at
*    http://www.open-xchange.com/EN/developer/. The contributing author shall be
*    given Attribution for the derivative code and a license granting use.
*
*     Copyright (C) 2004-2010 Open-Xchange, Inc.
*     Mail: info@open-xchange.com
*
*
*     This program is free software; you can redistribute it and/or modify it
*     under the terms of the GNU General Public License, Version 2 as published
*     by the Free Software Foundation.
*
*     This program is distributed in the hope that it will be useful, but
*     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
*     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
*     for more details.
*
*     You should have received a copy of the GNU General Public License along
*     with this program; if not, write to the Free Software Foundation, Inc., 59
*     Temple Place, Suite 330, Boston, MA 02111-1307 USA
*
*/
public class VCardEntry {
	private static final String HEX_CODES = "0123456789ABCDEF";
	private static final String[] EMPTY_ARRAY = new String[0];

	private final String _key;
	private final Map<String, String[]> _options = new HashMap<String, String[]>();
	private String[] _data;

	public VCardEntry(String line) {
		String[] parts = line.split(":", 2);
		_key = extractKeyAndOptions(parts[0]);
		_data = (parts.length > 1) ? parseDataLine(parts[1]) : EMPTY_ARRAY;
	}

	public VCardEntry(String key, String... data) {
		_key = key;
		_data = data;
	}

	public VCardEntry(String key, String[] options, String[] extraOptions, String... data) {
		_key = key;
		_data = data;
		for (String option : options)
			setOption(option);
		for (String option : extraOptions)
			setOption(option);
	}

	public void setOption(String key) {
		setOption(key.toUpperCase(), EMPTY_ARRAY);
	}

	public void setOption(String key, String[] values) {
		_options.put(key.toUpperCase(), values);
	}

	private String extractKeyAndOptions(String key) {
		int pos = key.indexOf('.');
		if (pos >= 0) // ignore initial group element, if present
			key = key.substring(pos + 1);
		pos = key.indexOf(';');
		if (pos < 0)
			return key;
		extractOptions(key.substring(pos + 1));
		return key.substring(0, pos);
	}

	private void extractOptions(String options) {
		for (int pos = options.indexOf(';'); pos >= 0; pos = options.indexOf(';')) {
			String param = options.substring(0, pos);
			extractOption(param);
			options = options.substring(pos + 1);
		}
		extractOption(options);
	}

	private void extractOption(String option) {
		if (option.trim().length() > 0) {
			int delimiterPos = option.indexOf('=');
			if (delimiterPos < 0)
				setOption(option);
			else
				setOption(option.substring(0, delimiterPos), option.substring(delimiterPos + 1).split(","));
		}
	}

	public String getKey() {
		return _key;
	}

	public String[] getData() {
		return _data;
	}

	public boolean isVCardBegin() {
		return VCardConstants.BEGIN.equalsIgnoreCase(_key) && isDataVCardName();
	}

	public boolean isVCardEnd() {
		return VCardConstants.END.equalsIgnoreCase(_key) && isDataVCardName();
	}

	private boolean isDataVCardName() {
		return _data.length == 1 && VCardConstants.VCARD.equalsIgnoreCase(_data[0]);
	}

	public void addToLastDataEntry(String line) {
		_data[_data.length - 1] += line;
	}

	public String[] getOption(String name) {
		return _options.get(name);
	}

	@Override
	public String toString() {
		StringBuilder sb = new StringBuilder(_key);
		for (Map.Entry<String, String[]> e : _options.entrySet()) {
			sb.append(';').append(e.getKey());
			char delim = '=';
			for (String s : e.getValue()) {
				sb.append(delim).append(s);
				delim = ',';
			}
		}
		boolean needsSemiColon = false;
		if (isAsciiData()) {
			char delim = ':';
			for (String s : _data) {
				sb.append(delim).append(s);
				delim = ';';
			}
			return sb.toString();
		}
		sb.append(";ENCODING=QUOTED-PRINTABLE;CHARSET=UTF-8:");
		try {
			for (String s : _data) {
				if (needsSemiColon)
					sb.append(';');
				needsSemiColon = true;
				for (byte b : s.getBytes(SyncMLConstants.UTF_8)) {
					int v = ((int) b) & 0xFF;
					if (needsEscaping(v)) {
						sb.append('=').append(HEX_CODES.charAt(v / 0x10)).append(HEX_CODES.charAt(v & 0xF));
					} else {
						sb.append((char) v);
					}
				}
			}
			return sb.toString();
		} catch (UnsupportedEncodingException e1) {
			throw new IllegalStateException("Character encoding UTF-8 not supported");
		}
	}

	private boolean isAsciiData() {
		for (String s : _data) {
			int len = s.length();
			for (int i = 0; i < len; i++) {
				char c = s.charAt(i);
				if (c != 32 && needsEscaping(c))
					return false;
			}
		}
		return true;
	}

	private boolean needsEscaping(int c) {
		// escape outside of ASCII and '=', ';', ',', '"', '\\'
		return c < 33 || c > 126 || c == 61 || c == 59 || c == 44 || c == 34 || c == 92;
	}

	private String[] parseDataLine(String dataLine) {
		boolean isQuotedPrintable = hasQuotedPrintableEncoding();
		List<String> list = new ArrayList<String>();
		int len = dataLine.length();
		StringBuilder element = new StringBuilder(128);
		for (int i = 0; i < len; i++) {
			char c = dataLine.charAt(i);
			if (c == ';') {
				addToDataList(list, element, isQuotedPrintable);
				element = new StringBuilder(128);
			} else if (c == '\\') {
				if (++i < len) {
					char c2 = dataLine.charAt(i);
					if (c2 == 'n' || c2 == 'N')
						element.append('\n');
					else
						element.append(c2);
				}
				// Ignore invalid '\' at end of data
			} else {
				element.append(c);
			}
		}
		if (element.length() > 0)
			addToDataList(list, element, isQuotedPrintable);
		return list.toArray(new String[list.size()]);
	}

	private void addToDataList(List<String> list, Object element, boolean isQuotedPrintable) {
		list.add(isQuotedPrintable ? decodeQuotedPrintable(element.toString()) : element.toString());
	}

	private boolean hasQuotedPrintableEncoding() {
		String[] values = _options.get(VCardConstants.ENCODING);
		if (values != null) {
			for (String s : values) {
				if (VCardConstants.QUOTED_PRINTABLE.equalsIgnoreCase(s))
					return true;
			}
		}
		return false;
	}

	private String getCharset() {
		String[] values = _options.get(VCardConstants.CHARSET);
		return (values != null && values.length > 0) ? values[0] : "cp1252";
	}

	private String decodeQuotedPrintable(String input) {
		int len = input.length();
		ByteArrayOutputStream baos = new ByteArrayOutputStream();
		for (int i = 0; i < len; i++) {
			char c = input.charAt(i);
			switch (c) {
				case '=':
					if (i < len - 1) {
						char c1 = input.charAt(i + 1);
						if (c1 == '\n') {
							i++;
							break;
						}
						if (i < len - 2) {
							char c2 = input.charAt(i + 2);
							if (c1 == '\r' && c2 == '\n') {
								i += 2;
								break;
							}
							int d1 = getHexCode(c1);
							int d2 = getHexCode(c2);
							if (d1 >= 0 && d2 >= 0) {
								baos.write((d1 << 4) + d2);
								i += 2;
								break;
							}
						}
					}
					// should not be reached, just in case we received bad input
					baos.write(c);
					break;
				case '\t':
				case ' ':
					char c2 = 0;
					for (int j = i + 1; j < len; j++) {
						c2 = input.charAt(j);
						if (c2 != ' ' && c2 != '\t')
							break;
					}
					if (c2 != '\r' && c2 != '\n' && c2 != 0)
						baos.write(c);
					break;
				default:
					if (c >= 33 && c <= 126)
						baos.write(c);
					break;
			}
		}
		try {
			return baos.toString(getCharset());
		} catch (UnsupportedEncodingException e) {
			return baos.toString();
		}
	}

	private static int getHexCode(char c1) {
		if (c1 >= '0' && c1 <= '9')
			return c1 - '0';
		if (c1 >= 'A' && c1 <= 'F')
			return c1 + (10 - 'A');
		if (c1 >= 'a' && c1 <= 'f')
			return c1 + (10 - 'a');
		return -1;
	}
}
