/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2010 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */
package com.openexchange.usm.syncml.commands;

import java.io.IOException;
import java.util.*;
import java.util.Map;

import org.xmlpull.v1.*;

import com.openexchange.usm.api.contenttypes.ContentType;
import com.openexchange.usm.api.exceptions.USMException;
import com.openexchange.usm.api.session.*;
import com.openexchange.usm.session.dataobject.DataObjectSet;
import com.openexchange.usm.syncml.*;
import com.openexchange.usm.syncml.data.SyncMLProtocolInfo;
import com.openexchange.usm.syncml.elements.*;
import com.openexchange.usm.syncml.exceptions.SyncMLException;
import com.openexchange.usm.syncml.mapping.ContentTypeMapping;
import com.openexchange.usm.syncml.mapping.ContentTypeMappings;
import com.openexchange.usm.syncml.parser.XmlPullParserUtil;
import com.openexchange.usm.syncml.servlet.SyncMLRequest;
import com.openexchange.usm.syncml.sync.SyncInformation;

public class Sync extends SyncContainerCommand {
	public static final String ELEMENT_NAME = "Sync";

	private final boolean _noResp;
	private final Cred _cred;
	private final Target _target;
	private final Source _source;
	private final Meta _meta;
	private final String _numberOfChanges;

	private final List<SyncMLCommand> _commands = new ArrayList<SyncMLCommand>();

	private SyncResult _result = null;
	private SyncMLStatusCode _status = SyncMLStatusCode.OK;

	public Sync(XmlPullParser parser, Meta meta, String cmdID) throws XmlPullParserException, IOException {
		setCmdID(cmdID);
		_noResp = readNoResp(parser);
		_cred = readOptionalCred(parser, meta);
		_target = readOptionalTarget(parser);
		_source = readOptionalSource(parser);
		_meta = readOptionalMeta(parser, meta);
		_numberOfChanges = XmlPullParserUtil.readOptionalText(parser, SimpleElements.NUMBER_OF_CHANGES);
		readCommands(parser, _meta, _commands, 0);
		readElementEnd(parser);
	}

	public Sync(boolean noResp, Cred cred, Target target, Source source, Meta meta, List<SyncMLCommand> subCommands) {
		_noResp = noResp;
		_cred = cred;
		_target = target;
		_source = source;
		_meta = meta;
		_numberOfChanges = null;
		_commands.addAll(subCommands);
	}

	public String getElementName() {
		return ELEMENT_NAME;
	}

	@Override
	public boolean isSyncAction() {
		return true;
	}

	public boolean isNoResp() {
		return _noResp;
	}

	@Override
	public List<SyncMLCommand> getSubCommands() {
		return _commands;
	}

	public Meta getMeta() {
		return _meta;
	}

	public void addStatusForCommand(SyncResult result, SyncMLRequest syncMLRequest) {
		syncMLRequest.addStatusForCommand(this, _target.getLocURI(), _source.getLocURI(), _status);
		addStatusForSubCommands(result, syncMLRequest);
	}

	public void execute(SyncMLRequest syncMLRequest) throws SyncMLException {
		_status = performSync(syncMLRequest);
		addStatusForCommand(_result, syncMLRequest);
	}

	@Override
	protected void writeContent(XmlSerializer serializer, Meta meta) throws IOException {
		super.writeContent(serializer, meta);
		writeNoResp(serializer, _noResp);
		if (_cred != null)
			_cred.write(serializer, meta);
		if (_target != null)
			_target.write(serializer, meta);
		if (_source != null)
			_source.write(serializer, meta);
		if (_meta != null) {
			_meta.write(serializer, meta);
			meta = _meta;
		}
		if (_numberOfChanges != null)
			XmlPullParserUtil.writeOptionalText(serializer, SimpleElements.NUMBER_OF_CHANGES, _numberOfChanges);
		for (SyncMLCommand command : _commands)
			command.write(serializer, meta);
	}

	private SyncMLStatusCode performSync(SyncMLRequest syncMLRequest) {
		// Determine Sync type from HttpSession, 
		SyncInformation syncInfo = syncMLRequest.getSyncInformation(_source.getLocURI(), _target.getLocURI());
		if (syncInfo == null)
			return SyncMLStatusCode.BAD_REQUEST;
		SyncType syncType = syncInfo.getSyncType();
		Session usmSession = syncMLRequest.getUSMSession();
		ContentType contentType = syncInfo.getContentType();
		List<ContentTypeMapping> mappings = ContentTypeMappings.getMappings(contentType);
		if (mappings == null || mappings.size() == 0)
			return SyncMLStatusCode.OPTIONAL_FEATURE_NOT_SUPPORTED;
		String folderID = syncInfo.getOxFolderID();
		try {
			if (syncType == SyncType.SLOW_SYNC) {
				DataObject[] clientRequests = buildClientRequests(syncMLRequest, syncInfo, null);
				if (!syncMLRequest.isRequestFinal()) {
					syncInfo.setClientRequests(clientRequests);
					return SyncMLStatusCode.OK;
				}
				_result = usmSession.syncWithServer(folderID, Session.NO_LIMIT, null, true,
						ConflictResolution.USE_SERVER_DELETE_OVER_CHANGE, clientRequests);
			} else if (syncType == SyncType.TWO_WAY_SYNC) {
				//collect all data object changes from all sub-commands
				long lastServerTimestamp = syncInfo.getLastServerTimestamp();
				DataObject[] oldSyncState = usmSession.getCachedFolderElements(folderID, contentType,
						lastServerTimestamp);
				DataObject[] clientRequests = buildClientRequests(syncMLRequest, syncInfo, oldSyncState);
				if (!syncMLRequest.isRequestFinal()) {
					syncInfo.setClientRequests(clientRequests);
					return SyncMLStatusCode.OK;
				}
				//Perform sync with server,
				_result = usmSession.syncChangesWithServer(folderID, lastServerTimestamp, Session.NO_LIMIT, null, true,
						ConflictResolution.USE_SERVER_DELETE_OVER_CHANGE, clientRequests);
			} // LATER add other sync types in future
			if (SyncMLConstants.PERFORM_INTEGRITY_CHECKS)
				validateIntegrity(syncMLRequest);
			// Store information for Sync response in SyncInformation
			syncInfo.updateFromSyncResult(_result, mappings.get(0), new Target(_source), new Source(_target));
			return SyncMLStatusCode.OK;
		} catch (USMException e) {
			syncMLRequest.getJournal().error(
					usmSession + ": Internal error syncing folder " + folderID + "(" + contentType.getID() + ")", e);
			_result = null;
			return SyncMLStatusCode.SERVER_FAILURE;
		}
	}

	private void validateIntegrity(SyncMLRequest syncMLRequest) {
		DataObjectSet changes = new DataObjectSet(_result.getChanges());
		for (DataObject o : _result.getNewState()) {
			if (!(o.getProtocolInformation() instanceof SyncMLProtocolInfo)) {
				DataObject co = changes.get(o.getID());
				if (co == null || co.getChangeState() != ChangeState.CREATED)
					syncMLRequest.getJournal().error(
							syncMLRequest.getUSMSession()
									+ ": DataObject without ProtocolInfo that's not newly created: " + o);
			}
		}
	}

	private DataObject[] buildClientRequests(SyncMLRequest syncMLRequest, SyncInformation syncInfo,
			DataObject[] oldSyncState) {
		List<DataObject> requests = new ArrayList<DataObject>();
		DataObject[] previousRequests = syncInfo.getClientRequests();
		if (previousRequests != null) {
			for (DataObject o : previousRequests)
				requests.add(o);
		}
		addDataObjectsForCommand(syncMLRequest, requests, syncInfo.getContentType(), buildClientIDMap(oldSyncState));
		for (DataObject o : requests)
			o.setParentFolderID(syncInfo.getOxFolderID());
		return requests.toArray(new DataObject[requests.size()]);
	}

	private Map<String, DataObject> buildClientIDMap(DataObject[] oldSyncState) {
		if (oldSyncState == null)
			return Collections.emptyMap();
		Map<String, DataObject> map = new HashMap<String, DataObject>();
		for (DataObject o : oldSyncState) {
			ProtocolInformation i = o.getProtocolInformation();
			if (i instanceof SyncMLProtocolInfo)
				map.put(((SyncMLProtocolInfo) i).getDeviceObjectID(), o);
		}
		return map;
	}
}
