/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2010 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */
package com.openexchange.usm.syncml.commands;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;

import org.xmlpull.v1.*;

import com.openexchange.usm.api.exceptions.USMException;
import com.openexchange.usm.api.session.Folder;
import com.openexchange.usm.syncml.*;
import com.openexchange.usm.syncml.elements.*;
import com.openexchange.usm.syncml.elements.xml.ComposedXMLPart;
import com.openexchange.usm.syncml.exceptions.SyncMLException;
import com.openexchange.usm.syncml.servlet.SyncMLRequest;
import com.openexchange.usm.syncml.sync.SyncInformation;

public class Alert extends BaseCommand {
	public static final String ELEMENT_NAME = "Alert";

	private final boolean _noResp;
	private final Cred _cred;
	private final Data _data;
	private final List<Item> _items = new ArrayList<Item>();

	private final List<Item> _statusItems = new ArrayList<Item>();
	private final List<Item> _responseItems = new ArrayList<Item>();
	private AlertType _responseAlertType;
	private String _targetRef;
	private String _sourceRef;
	private SyncMLStatusCode _status = SyncMLStatusCode.OK;

	public Alert(boolean noResp, Cred cred, Data data, List<Item> items) {
		_noResp = noResp;
		_cred = cred;
		_data = data;
		_items.addAll(items);
	}

	public Alert(XmlPullParser parser, Meta meta, String cmdID) throws XmlPullParserException, IOException {
		setCmdID(cmdID);
		_noResp = readNoResp(parser);
		_cred = readOptionalCred(parser, meta);
		_data = readOptionalData(parser, null);
		readItems(parser, meta, _items, 1);
		readElementEnd(parser);
	}

	public String getElementName() {
		return ELEMENT_NAME;
	}

	@Override
	protected void writeContent(XmlSerializer serializer, Meta meta) throws IOException {
		super.writeContent(serializer, meta);
		writeNoResp(serializer, _noResp);
		if (_cred != null)
			_cred.write(serializer, meta);
		if (_data != null)
			_data.write(serializer, null);
		writeItems(serializer, meta, _items);
	}

	public boolean isNoResp() {
		return _noResp;
	}

	@Override
	public boolean isSyncAction() {
		return true;
	}

	public void execute(SyncMLRequest syncMLRequest) throws SyncMLException {
		SyncMLStatusCode status = performAlert(syncMLRequest);
		syncMLRequest.addStatusForCommand(this, _targetRef, _sourceRef, (status == null) ? _status : status,
				_statusItems.toArray(new Item[_statusItems.size()]));
		if (status == null && _responseAlertType != null)
			syncMLRequest.addCommand(new Alert(false, null, new Data(String.valueOf(_responseAlertType.getCode())),
					_responseItems));
	}

	private SyncMLStatusCode performAlert(SyncMLRequest syncMLRequest) throws SyncMLException {
		extractAlertType(_data);
		if (_responseAlertType == null)
			return SyncMLStatusCode.INCOMPLETE_COMMAND;
		switch (_responseAlertType) {
			case NEXT_MESSAGE:
				return SyncMLStatusCode.OK;
			case SLOW_SYNC:
			case TWO_WAY:
				for (Item i : _items) {
					try {
						SyncMLStatusCode status = performAlert(syncMLRequest, i);
						if (status != null)
							return status;
					} catch (USMException e) {
						throw new SyncMLException(SyncMLBundleErrorCodes.ALERT_FAILED_FOR_FOLDER,
								SyncMLStatusCode.SERVER_FAILURE, "Error while executing Alert", e);
					}
				}
				return null;
			default:
				return SyncMLStatusCode.OPTIONAL_FEATURE_NOT_SUPPORTED;
		}
	}

	private SyncMLStatusCode performAlert(SyncMLRequest syncMLRequest, Item i) throws USMException {
		Target t = i.getTarget();
		Source s = i.getSource();
		Meta m = i.getMeta();
		if (t == null || s == null || m == null)
			return SyncMLStatusCode.INCOMPLETE_COMMAND;
		_targetRef = t.getLocURI();
		_sourceRef = s.getLocURI();
		Folder folder = syncMLRequest.getFolder(_targetRef);
		if (folder == null)
			return SyncMLStatusCode.NOT_FOUND;
		String lastClientAnchor = m.getXMLData(SyncMLConstants.ANCHOR, SyncMLConstants.LAST);
		String nextClientAnchor = m.getXMLData(SyncMLConstants.ANCHOR, SyncMLConstants.NEXT);
		String lastServerAnchor = null;
		String nextServerAnchor = null;
		long lastSyncTimestamp = 0L;
		if (_responseAlertType == AlertType.TWO_WAY) {
			if (syncMLRequest.isClientAnchorValid(_sourceRef, lastClientAnchor, folder)) {
				lastServerAnchor = syncMLRequest.getLastServerAnchor(_sourceRef);
				lastSyncTimestamp = syncMLRequest.getLastSyncTimestamp(_sourceRef);
			} else {
				_responseAlertType = AlertType.SLOW_SYNC;
				_status = SyncMLStatusCode.REFRESH_REQUIRED;
				lastClientAnchor = null;
			}
		}
		nextServerAnchor = syncMLRequest.getNextServerAnchor(lastSyncTimestamp);
		if (_responseAlertType == AlertType.SLOW_SYNC) {
			lastServerAnchor = nextServerAnchor;
			syncMLRequest.setSlowSyncRequired();
		}
		syncMLRequest.storeSyncInformation(new SyncInformation(_responseAlertType.getSyncType(), _sourceRef,
				_targetRef, folder, lastClientAnchor, nextClientAnchor, lastServerAnchor, nextServerAnchor,
				lastSyncTimestamp));
		ComposedXMLPart statusData = new ComposedXMLPart(SyncMLConstants.ANCHOR, SyncMLConstants.NS_SYNCML_METINF);
		statusData.addXMLData(nextClientAnchor, SyncMLConstants.NEXT);
		_statusItems.add(new Item(null, null, null, new Data(statusData)));
		ComposedXMLPart responseData = new ComposedXMLPart(SyncMLConstants.ANCHOR, SyncMLConstants.NS_SYNCML_METINF);
		if (lastServerAnchor != null)
			responseData.addXMLData(lastServerAnchor, SyncMLConstants.LAST);
		responseData.addXMLData(nextServerAnchor, SyncMLConstants.NEXT);
		_responseItems.add(new Item(new Target(s), new Source(t), new Meta(responseData), null));
		return null;
	}

	private void extractAlertType(Data data) {
		if (data != null && data.isStringContent()) {
			try {
				_responseAlertType = AlertType.fromCode(Integer.parseInt(data.getStringContent()));
			} catch (NumberFormatException ignored) {
			}
		}
	}
}
