/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2010 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */
/**
 * 
 */
package com.openexchange.usm.session.sync.test;

import java.util.ArrayList;
import java.util.List;
import java.util.UUID;

import junit.framework.TestCase;

import com.openexchange.usm.api.exceptions.ConflictingChange;
import com.openexchange.usm.api.exceptions.USMException;
import com.openexchange.usm.api.session.ChangeState;
import com.openexchange.usm.api.session.ConflictResolution;
import com.openexchange.usm.api.session.DataObject;
import com.openexchange.usm.api.session.ProtocolInformation;
import com.openexchange.usm.api.session.SimSession;
import com.openexchange.usm.api.session.SyncResult;
import com.openexchange.usm.api.session.SynchronizationConflictException;
import com.openexchange.usm.configuration.ConfigurationProperties;
import com.openexchange.usm.journal.SimJournal;
import com.openexchange.usm.session.dataobject.DataObjectUtil;
import com.openexchange.usm.session.impl.SessionManagerImpl;
import com.openexchange.usm.session.sync.IncrementalContentSyncer;
import com.openexchange.usm.session.sync.SimContentSyncerStorage;
import com.openexchange.usm.session.sync.SimSyncContentType;
import com.openexchange.usm.session.sync.SimSyncContentTypeTransferHandler;

/**
 * @author ldo
 *
 */
public class IncrementalContentSyncerTest extends TestCase {
	private static interface ResolutionRunnable {
		void run(ConflictResolution resolution) throws Exception;
	}

	private static class TestProtocolInfo implements ProtocolInformation {
		private static final long serialVersionUID = 1L;

		private final String _data;

		public TestProtocolInfo(String data) {
			_data = data;
		}

		public String getData() {
			return _data;
		}

		@Override
		public String toString() {
			return "ProtocolInfo(" + _data + ')';
		}

		@Override
		public int hashCode() {
			final int prime = 31;
			int result = 1;
			result = prime * result + ((_data == null) ? 0 : _data.hashCode());
			return result;
		}

		@Override
		public boolean equals(Object obj) {
			if (this == obj)
				return true;
			if (obj == null)
				return false;
			if (!(obj instanceof TestProtocolInfo))
				return false;
			TestProtocolInfo other = (TestProtocolInfo) obj;
			if (_data == null) {
				if (other._data != null)
					return false;
			} else if (!_data.equals(other._data))
				return false;
			return true;
		}
	}

	private static final int INITIAL_SMALL_DATAOBJECT_COUNT = 2;
	private static final int INITIAL_LARGE_DATAOBJECT_COUNT = 100;
	private static final boolean WAIT_AT_END = false;

	List<DataObject> syncState = new ArrayList<DataObject>();
	List<DataObject> expectedSyncState = new ArrayList<DataObject>();
	List<DataObject> clientChanges = new ArrayList<DataObject>();
	List<DataObject> expectedClientChanges = new ArrayList<DataObject>();
	List<DataObject> dataOnServer = new ArrayList<DataObject>();
	SimContentSyncerStorage syncerStorage = new SimContentSyncerStorage();
	SimSession session = new SimSession();
	SimSyncContentType contentType = new SimSyncContentType();
	SimSyncContentTypeTransferHandler handler = (SimSyncContentTypeTransferHandler) contentType.getTransferHandler();
	IncrementalContentSyncer contentSyncer = new IncrementalContentSyncer(new SessionManagerImpl(new SimJournal()
			.getLog(IncrementalContentSyncerTest.class)),
			ConfigurationProperties.SYNC_CONCURRENT_MODIFICATION_MAX_RETRIES_DEFAULT);

	public void testNoChangesOnClientAndServer() {
		performMultiCheck(new ResolutionRunnable() {
			public void run(ConflictResolution resolution) throws Exception {
				initOriginalState();
				performSync(resolution, 0);
			}
		}, ConflictResolution.values());
	}

	public void testClientCreation() {
		performMultiCheck(new ResolutionRunnable() {
			public void run(ConflictResolution resolution) throws Exception {
				initOriginalState();
				TestProtocolInfo protocolInfo = new TestProtocolInfo("ClientProtInfo");
				DataObject o = createDataObject("clientID", "Client", protocolInfo, 1, false);
				clientChanges.add(o);
				DataObject o2 = o.createCopy(false);
				o2.setTimestamp(2L);
				o2.commitChanges();
				o2.setProtocolInformation(protocolInfo);
				expectedSyncState.add(o2);
				handler._writeNewToCall = true;
				handler._objectToSend = o;
				performSync(resolution, 1);
				assertFalse(o.isFailed());
			}
		}, ConflictResolution.values());
	}

	public void testServerCreation() {
		performMultiCheck(new ResolutionRunnable() {
			public void run(ConflictResolution resolution) throws Exception {
				initOriginalState(createDataObject("serverID", "Server", null, 1, true));
				performSync(resolution, 0);
			}
		}, ConflictResolution.values());
	}

	public void testClientDeletion() {
		performMultiCheck(new ResolutionRunnable() {
			public void run(ConflictResolution resolution) throws Exception {
				initOriginalState();
				DataObject o = syncState.get(1).createCopy(false);
				o.setChangeState(ChangeState.DELETED);
				clientChanges.add(o);
				handler._writeDeleteToCall = true;
				handler._objectToSend = o.createCopy(false);
				expectedSyncState.remove(1);
				performSync(resolution, 1);
			}
		}, ConflictResolution.values());
	}

	public void testServerDeletion() {
		performMultiCheck(new ResolutionRunnable() {
			public void run(ConflictResolution resolution) throws Exception {
				initOriginalState();
				DataObject o = dataOnServer.remove(1);
				o.setTimestamp(1L);
				o.commitChanges();
				o.setTimestamp(2L);
				o.setChangeState(ChangeState.DELETED);
				expectedClientChanges.add(o);
				o.setProtocolInformation(expectedSyncState.remove(1).getProtocolInformation());
				performSync(resolution, 0);
			}
		}, ConflictResolution.values());
	}

	public void testClientChange() {
		performMultiCheck(new ResolutionRunnable() {
			public void run(ConflictResolution resolution) throws Exception {
				initOriginalState();
				DataObject o = syncState.get(1).createCopy(false);
				o.setFieldContent("Field2", "ModifiedClient");
				o.setProtocolInformation(syncState.get(1).getProtocolInformation());
				clientChanges.add(o);
				DataObject o2 = o.createCopy(false);
				o2.setTimestamp(2L);
				o2.setProtocolInformation(o.getProtocolInformation());
				o2.commitChanges();
				expectedSyncState.set(1, o2);
				handler._writeUpdateToCall = true;
				handler._objectToSend = o;
				performSync(resolution, 1);
				assertFalse(o.isFailed());
			}
		}, ConflictResolution.values());
	}

	public void testServerChange() {
		performMultiCheck(new ResolutionRunnable() {
			public void run(ConflictResolution resolution) throws Exception {
				initOriginalState();
				DataObject o = dataOnServer.get(1);
				DataObject o2 = o.createCopy(false);
				o.setFieldContent("Field2", "ModifiedServer");
				o2.setTimestamp(1L);
				o2.commitChanges();
				o2.setTimestamp(2L);
				o2.setFieldContent("Field2", "ModifiedServer");
				o2.setProtocolInformation(syncState.get(1).getProtocolInformation());
				DataObject o3 = o2.createCopy(false);
				o3.setProtocolInformation(o2.getProtocolInformation());
				expectedClientChanges.add(o3);
				o2.commitChanges();
				expectedSyncState.set(1, o2);
				performSync(resolution, 0);
			}
		}, ConflictResolution.values());
	}

	public void testClientDeletionServerDeletion() {
		performMultiCheck(new ResolutionRunnable() {
			public void run(ConflictResolution resolution) throws Exception {
				initOriginalState();
				DataObject o = dataOnServer.remove(1);
				o.setChangeState(ChangeState.DELETED);
				clientChanges.add(o);
				expectedSyncState.remove(1);
				performSync(resolution, 0);
				assertFalse(o.isFailed());
			}
		}, ConflictResolution.values());
	}

	public void testClientChangeServerDeletionERROR() throws USMException {
		clientChangeServerDeletionExpectError(ConflictResolution.ERROR);
	}

	public void testClientChangeServerDeletionERROR_DELETE_OVER_CHANGE() throws USMException {
		clientChangeServerDeletionExpectDeletion(ConflictResolution.ERROR_DELETE_OVER_CHANGE);
	}

	public void testClientChangeServerDeletionCLIENT() throws USMException {
		clientChangeServerDeletionExpectError(ConflictResolution.USE_CLIENT);
	}

	public void testClientChangeServerDeletionCLIENT_DELETE_OVER_CHANGE() throws USMException {
		clientChangeServerDeletionExpectDeletion(ConflictResolution.USE_CLIENT_DELETE_OVER_CHANGE);
	}

	public void testClientChangeServerDeletionSERVER() throws USMException {
		clientChangeServerDeletionExpectDeletion(ConflictResolution.USE_SERVER);
	}

	public void testClientChangeServerDeletionSERVER_DELETE_OVER_CHANGE() throws USMException {
		clientChangeServerDeletionExpectDeletion(ConflictResolution.USE_SERVER_DELETE_OVER_CHANGE);
	}

	public void testClientDeletionServerChangeERROR() throws USMException {
		clientDeletionServerChangeExpectError(ConflictResolution.ERROR);
	}

	public void testClientDeletionServerChangeERROR_DELETE_OVER_CHANGE() throws USMException {
		clientDeletionServerChangeExpectDeletion(ConflictResolution.ERROR_DELETE_OVER_CHANGE);
	}

	public void testClientDeletionServerChangeCLIENT() throws USMException {
		clientDeletionServerChangeExpectDeletion(ConflictResolution.USE_CLIENT);
	}

	public void testClientDeletionServerChangeCLIENT_DELETE_OVER_CHANGE() throws USMException {
		clientDeletionServerChangeExpectDeletion(ConflictResolution.USE_CLIENT_DELETE_OVER_CHANGE);
	}

	public void testClientDeletionServerChangeSERVER() throws USMException {
		clientDeletionServerChangeExpectChange(ConflictResolution.USE_SERVER);
	}

	public void testClientDeletionServerChangeSERVER_DELETE_OVER_CHANGE() throws USMException {
		clientDeletionServerChangeExpectDeletion(ConflictResolution.USE_SERVER_DELETE_OVER_CHANGE);
	}

	public void testClientChangeServerChangeERROR() throws USMException {
		clientChangeServerChangeExpectError(ConflictResolution.ERROR);
	}

	public void testClientChangeServerChangeERROR_DELETE_OVER_CHANGE() throws USMException {
		clientChangeServerChangeExpectError(ConflictResolution.ERROR_DELETE_OVER_CHANGE);
	}

	public void testClientChangeServerChangeCLIENT() throws USMException {
		clientChangeServerChangeExpectClient(ConflictResolution.USE_CLIENT);
	}

	public void testClientChangeServerChangeCLIENT_DELETE_OVER_CHANGE() throws USMException {
		clientChangeServerChangeExpectClient(ConflictResolution.USE_CLIENT_DELETE_OVER_CHANGE);
	}

	public void testClientChangeServerChangeSERVER() throws USMException {
		clientChangeServerChangeExpectServer(ConflictResolution.USE_SERVER);
	}

	public void testClientChangeServerChangeSERVER_DELETE_OVER_CHANGE() throws USMException {
		clientChangeServerChangeExpectServer(ConflictResolution.USE_SERVER_DELETE_OVER_CHANGE);
	}

	public void testLimitedSyncWithManyChanges() throws USMException, InterruptedException {
		performMultiCheck(new ResolutionRunnable() {
			public void run(ConflictResolution resolution) throws Exception {
				syncState.clear();
				dataOnServer.clear();
				for (int i = 0; i < INITIAL_LARGE_DATAOBJECT_COUNT; i++) {
					DataObject o = createDataObject("syncID" + i, "Sync" + i, null, 1, true);
					o.setUUID(UUID.randomUUID());
					syncState.add(o);
					dataOnServer.add(createDataObject("syncID" + i, "Server" + i, null, 2, true));
					dataOnServer.add(createDataObject("syncID" + (i + INITIAL_LARGE_DATAOBJECT_COUNT), "New" + i, null,
							2, true));
				}
				for (int i = 0; i < 2 * INITIAL_LARGE_DATAOBJECT_COUNT; i++) {
					handler.resetServerCalls();
					expectedSyncState.clear();
					clientChanges.clear();
					expectedClientChanges.clear();
					for (DataObject o : dataOnServer) {
						o.setTimestamp(i + 2);
						o.commitChanges();
					}
					SyncResult r = performSync(ConflictResolution.ERROR, 0, i + 2L, 1,
							i < 2 * INITIAL_LARGE_DATAOBJECT_COUNT - 1);
					DataObject[] clientChanges = r.getChanges();
					assertEquals(1, clientChanges.length);
					int index = Integer.parseInt(clientChanges[0].getID().substring(6));
					DataObject o = null;
					if (i < INITIAL_LARGE_DATAOBJECT_COUNT) {
						o = createDataObject("syncID" + index, "Sync" + index, null, i + 1, true);
						o.setTimestamp(i + 2L);
						o.setFieldContent("Field2", "Server" + index);
					} else {
						o = createDataObject("syncID" + index, "New" + (index - INITIAL_LARGE_DATAOBJECT_COUNT), null,
								i + 2, true);
						o.setChangeState(ChangeState.CREATED);
					}
					expectedClientChanges.add(o.createCopy(false));
					o.commitChanges();
					for (DataObject d : syncState) {
						if (d.getID().equals(o.getID()))
							expectedSyncState.add(o);
						else {
							DataObject o2 = d.createCopy(false);
							o2.setTimestamp(i + 2L);
							o2.commitChanges();
							expectedSyncState.add(o2);
						}
					}
					if (index >= INITIAL_LARGE_DATAOBJECT_COUNT)
						expectedSyncState.add(o);
					checkStateEquality(expectedClientChanges, r.getChanges());
					checkStateEquality(expectedSyncState, syncerStorage.getServerDataToStore());
					syncState.clear();
					for (DataObject os : r.getNewState())
						syncState.add(os);
				}
				//		for (;;)
				//			Thread.sleep(1000L);
			}
		}, ConflictResolution.values());
		if (WAIT_AT_END) {
			for (;;) {
				Thread.sleep(10000L);
			}
		}
	}

	private void clientChangeServerChangeExpectServer(ConflictResolution resolution) throws USMException {
		DataObject o = initClientChangeServerChange();
		o.rollbackChanges();
		o.setTimestamp(1L);
		o.commitChanges();
		o.setTimestamp(2L);
		o.setFieldContent("Field2", "ModifiedServer");
		o.setProtocolInformation(syncState.get(1).getProtocolInformation());
		expectedClientChanges.add(o);
		o = o.createCopy(false);
		o.setProtocolInformation(syncState.get(1).getProtocolInformation());
		o.commitChanges();
		expectedSyncState.set(1, o);
		performSync(resolution, 0);
		assertTrue("Client change is not set to failed", clientChanges.get(0).isFailed());
	}

	private void clientChangeServerChangeExpectClient(ConflictResolution resolution) throws USMException {
		initClientChangeServerChange();
		handler._writeUpdateToCall = true;
		DataObject o = clientChanges.get(0).createCopy(false);
		handler._objectToSend = o.createCopy(false);
		o.setProtocolInformation(syncState.get(1).getProtocolInformation());
		o.setTimestamp(2L);
		o.commitChanges();
		expectedSyncState.set(1, o);
		performSync(resolution, 1);
		assertFalse("Client change is set to failed", clientChanges.get(0).isFailed());
	}

	private void clientChangeServerChangeExpectError(ConflictResolution resolution) throws USMException {
		DataObject o = initClientChangeServerChange();
		try {
			performSync(resolution, 0);
			fail("Conflict did not generate Exception");
		} catch (SynchronizationConflictException e) {
			ConflictingChange[] conflicts = e.getConflictingChanges();
			assertEquals(1, conflicts.length);
			assertEquals(clientChanges.get(0).toString(), conflicts[0].getClientChange().toString());
			assertEquals(o, conflicts[0].getServerChange());
		}
	}

	private void clientDeletionServerChangeExpectError(ConflictResolution resolution) throws USMException {
		DataObject o = initClientDeletionServerChange();
		try {
			performSync(resolution, 0);
			fail("Conflict did not generate Exception");
		} catch (SynchronizationConflictException e) {
			ConflictingChange[] conflicts = e.getConflictingChanges();
			assertEquals(1, conflicts.length);
			assertEquals(clientChanges.get(0).toString(), conflicts[0].getClientChange().toString());
			assertEquals(o, conflicts[0].getServerChange());
		}
	}

	private void clientDeletionServerChangeExpectChange(ConflictResolution resolution) throws USMException {
		DataObject o = initClientDeletionServerChange();
		DataObject o2 = o.createCopy(false);
		o2.setProtocolInformation(o.getProtocolInformation());
		o2.commitChanges();
		expectedSyncState.set(1, o2);
		o.rollbackChanges();
		o.setTimestamp(1L);
		o.commitChanges();
		o.setTimestamp(2L);
		o.setFieldContent("Field2", "ModifiedServer");
		expectedClientChanges.add(o);
		handler._writeDeleteToCall = false;
		handler._objectToSend = null;
		performSync(resolution, 0);
		assertTrue("Client change is not set to failed", clientChanges.get(0).isFailed());
	}

	private void clientDeletionServerChangeExpectDeletion(ConflictResolution resolution) throws USMException {
		initClientDeletionServerChange();
		expectedSyncState.remove(1);
		performSync(resolution, 1);
		assertFalse("Client change is set to failed", clientChanges.get(0).isFailed());
	}

	private void clientChangeServerDeletionExpectError(ConflictResolution resolution) throws USMException {
		DataObject o2 = initClientChangeServerDeletion();
		try {
			performSync(resolution, 0);
			fail("Conflict did not generate Exception");
		} catch (SynchronizationConflictException e) {
			ConflictingChange[] conflicts = e.getConflictingChanges();
			assertEquals(1, conflicts.length);
			assertEquals(clientChanges.get(0).toString(), conflicts[0].getClientChange().toString());
			assertEquals(o2.toString(), conflicts[0].getServerChange().toString());
		}
	}

	private void clientChangeServerDeletionExpectDeletion(ConflictResolution resolution) throws USMException {
		DataObject o2 = initClientChangeServerDeletion();
		o2.setTimestamp(2L);
		expectedClientChanges.add(o2);
		performSync(resolution, 0);
		assertTrue("Client change is not set to failed", clientChanges.get(0).isFailed());
	}

	private DataObject initClientChangeServerChange() {
		initOriginalState();
		DataObject o = syncState.get(1).createCopy(false);
		o.setFieldContent("Field2", "ModifiedClient");
		clientChanges.add(o);
		DataObject o2 = dataOnServer.get(1);
		o2.setFieldContent("Field2", "ModifiedServer");
		o = o2.createCopy(false);
		o2.commitChanges();
		return o;
	}

	private DataObject initClientDeletionServerChange() {
		initOriginalState();
		DataObject o = syncState.get(1).createCopy(false);
		o.setChangeState(ChangeState.DELETED);
		clientChanges.add(o);
		handler._writeDeleteToCall = true;
		handler._objectToSend = o.createCopy(false);
		DataObject o2 = dataOnServer.get(1);
		o2.setFieldContent("Field2", "ModifiedServer");
		o = o2.createCopy(false);
		o.setProtocolInformation(syncState.get(1).getProtocolInformation());
		o2.commitChanges();
		return o;
	}

	private DataObject initClientChangeServerDeletion() {
		initOriginalState();
		DataObject o = syncState.get(1).createCopy(false);
		o.setFieldContent("Field2", "ModifiedClient");
		clientChanges.add(o);
		DataObject o2 = dataOnServer.remove(1);
		o2.setTimestamp(1L);
		o2.commitChanges();
		o2.setProtocolInformation(syncState.get(1).getProtocolInformation());
		o2.setChangeState(ChangeState.DELETED);
		expectedSyncState.remove(1);
		return o2;
	}

	private void performMultiCheck(ResolutionRunnable r, ConflictResolution... resolutions) {
		for (ConflictResolution resolution : resolutions) {
			try {
				r.run(resolution);
			} catch (Exception e) {
				throw new RuntimeException("Error while checking " + resolution, e);
			}
		}
	}

	private DataObject createDataObject(String id, String field2, ProtocolInformation protocolInfo, long timestamp,
			boolean commit) {
		DataObject o = contentType.newDataObject(session);
		o.setID(id);
		o.setFieldContent("Field2", field2);
		o.setTimestamp(timestamp);
		o.setProtocolInformation(protocolInfo);
		if (commit)
			o.commitChanges();
		return o;
	}

	private SyncResult performSync(ConflictResolution resolution, int expectedServerCalls) throws USMException {
		SyncResult result = performSync(resolution, expectedServerCalls, 2L, 0, false);
		checkStateEquality(expectedClientChanges, result.getChanges());
		checkStateEquality(expectedSyncState, syncerStorage.getServerDataToStore());
		return result;
	}

	private SyncResult performSync(ConflictResolution resolution, int expectedServerCalls, long timestamp, int limit,
			boolean expectIncomplete) throws USMException {
		syncerStorage.setCurrentServerData(DataObjectUtil.toArray(dataOnServer));
		SyncResult result = contentSyncer.syncChangesWithServer(syncerStorage, resolution, DataObjectUtil
				.toArray(syncState), DataObjectUtil.toArray(clientChanges), 1, limit, null, null);
		assertEquals(timestamp, result.getTimestamp());
		assertEquals(expectIncomplete, result.isIncomplete());
		handler.checkCalls(expectedServerCalls);
		return result;
	}

	private void checkStateEquality(List<DataObject> expectedData, DataObject[] data) {
		assertEquals(expectedData.size(), data.length);
		for (int i = 0; i < data.length; i++) {
			DataObject e = expectedData.get(i);
			assertEquals(e.toString(), DataObjectUtil.findDataObject(e.getID(), data).toString());
		}
	}

	private void initOriginalState(DataObject... newServerObjects) {
		syncState.clear();
		dataOnServer.clear();
		expectedSyncState.clear();
		clientChanges.clear();
		expectedClientChanges.clear();
		handler.resetServerCalls();
		for (int i = 0; i < INITIAL_SMALL_DATAOBJECT_COUNT; i++) {
			ProtocolInformation info = new TestProtocolInfo("Existing" + i);
			DataObject o = createDataObject("syncID" + i, "Sync" + i, info, 1, true);
			o.setUUID(UUID.randomUUID());
			syncState.add(o);
			dataOnServer.add(createDataObject("syncID" + i, "Sync" + i, null, 2, true));
			expectedSyncState.add(createDataObject("syncID" + i, "Sync" + i, info, 2, true));
		}
		for (DataObject o : newServerObjects) {
			dataOnServer.add(o.createCopy(false));
			DataObject d = o.createCopy(false);
			d.setTimestamp(2L);
			d.setChangeState(ChangeState.CREATED);
			expectedClientChanges.add(d);
			d = d.createCopy(false);
			d.commitChanges();
			expectedSyncState.add(d);
		}
	}
}
