/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2010 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */
package com.openexchange.usm.mapping.impl;

import java.sql.*;
import java.util.*;
import java.util.Map.Entry;

import org.apache.commons.logging.Log;

import com.openexchange.usm.api.database.*;
import com.openexchange.usm.api.exceptions.USMSQLException;
import com.openexchange.usm.api.session.Session;
import com.openexchange.usm.mapping.FolderIdMapping;
import com.openexchange.usm.util.Toolkit;

public class FolderIDMappingImpl extends AbstractMappingDBAccess implements FolderIdMapping {

	private final Map<String, Integer> _longToShortMap = new HashMap<String, Integer>();
	private final Map<Integer, String> _shortToLongMap = new HashMap<Integer, String>();

	private DatabaseAccess _dbAccess;

	public FolderIDMappingImpl(Log journal, DatabaseAccess dbAccess, Session session) {
		super(journal, session);
		_dbAccess = dbAccess;
	}

	public void delete(String... id) throws DatabaseAccessException, USMSQLException {
		deleteOrRetain("IN (", id);
		for (String i : id) {
			Integer s = _longToShortMap.remove(i);
			if (s != null)
				_shortToLongMap.remove(s);
		}
	}

	public void retain(String... id) throws DatabaseAccessException, USMSQLException {
		deleteOrRetain("NOT IN (", id);
		Map<String, Object> idMap = new HashMap<String, Object>();
		for (String i : id)
			idMap.put(i, null);
		for (Iterator<Map.Entry<String, Integer>> i = _longToShortMap.entrySet().iterator(); i.hasNext();) {
			Entry<String, Integer> e = i.next();
			if (!idMap.containsKey(e.getKey())) {
				_shortToLongMap.remove(e.getValue());
				i.remove();
			}
		}
	}

	public String getLongID(int shortID) throws USMSQLException {
		Integer shortInteger = shortID;
		String longID = _shortToLongMap.get(shortInteger);
		if (longID == null) {
			try {
				readCacheFromDB();
			} catch (DatabaseAccessException e) {
				_journal.error(_session + " SQL error reading cache from DB", e);
			}
			longID = _shortToLongMap.get(shortInteger);
		}
		if (longID == null)
			_journal.warn(_session + " No long ID for short ID " + shortID);
		return longID;
	}

	public int getShortID(String longID) throws DatabaseAccessException, USMSQLException {
		Integer shortID = _longToShortMap.get(longID);
		if (shortID == null) {
			readCacheFromDB();
			shortID = _longToShortMap.get(longID);
			if (shortID == null) {
				EncapsulatedConnection con = null;
				PreparedStatement statement = null;
				try {
					shortID = _dbAccess.getNextIdFromSequence(_session.getContextId(),
							MappingSQLStatements.SEQUENCE_TABLE_NAME);
					con = _session.getWritableDBConnection();
					statement = con.prepareStatement(MappingSQLStatements.FULL_INSERT_FOLDER_ID_MAPPING);
					addSessionIdentifierParamsToStmt(statement);
					statement.setString(NUMBER_OF_SESSION_ID_FIELDS + 1, longID);
					if (shortID == null)
						statement.setInt(NUMBER_OF_SESSION_ID_FIELDS + 2, 0);
					else
						statement.setInt(NUMBER_OF_SESSION_ID_FIELDS + 2, shortID);
					if (statement.executeUpdate() != 1)
						_journal.error(_session + " Insert of short ID was not performed");
					_longToShortMap.put(longID, shortID);
					_shortToLongMap.put(shortID, longID);
				} catch (SQLException e) {
					String errorMessage = _session + " SQL error generating shortid for " + longID;
					_journal.error(errorMessage, e);
					throw new USMSQLException(MappingBundleErrorCodes.GENERATE_SHORT_ID_FAILED, errorMessage, e);
				} finally {
					Toolkit.close(statement);
					Toolkit.close(con);
				}
			}
		}
		if (shortID == null)
			throw new USMSQLException(MappingBundleErrorCodes.GENERATE_SHORT_ID_FAILED, _session
					+ " SQL error generating shortid for " + longID, null);
		return shortID;
	}

	private void readCacheFromDB() throws DatabaseAccessException, USMSQLException {
		EncapsulatedConnection con = _session.getReadOnlyDBConnection();
		_longToShortMap.clear();
		_shortToLongMap.clear();
		PreparedStatement statement = null;
		ResultSet result = null;
		try {
			statement = con.prepareStatement("select shortID,LongID from usmIdMapping where "
					+ MappingSQLStatements.UNIQUE_SESSION_IDENTIFIER);
			addSessionIdentifierParamsToStmt(statement);
			result = statement.executeQuery();
			while (result.next()) {
				Integer shortID = result.getInt(1);
				String longID = result.getString(2);
				_longToShortMap.put(longID, shortID);
				_shortToLongMap.put(shortID, longID);
			}
		} catch (SQLException e) {
			String errorMessage = _session + " SQL error reading cache from DB";
			_journal.error(errorMessage, e);
			throw new USMSQLException(MappingBundleErrorCodes.SELECT_FAILED_NUMBER1, errorMessage, e);
		} finally {
			Toolkit.close(result);
			Toolkit.close(statement);
			Toolkit.close(con);
		}
	}

	private void deleteOrRetain(String extraCommandText, String[] folderID) throws DatabaseAccessException,
			USMSQLException {
		deleteOrRetain("SQL error while deleting short ID mappings in folders",
				MappingSQLStatements.DELETE_FOLDER_ID_MAPPING + MappingSQLStatements.UNIQUE_SESSION_IDENTIFIER
						+ " and LongID ", extraCommandText, null, folderID);
	}
}
