/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2010 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */
package com.openexchange.usm.json.test.outlooklogs;

import java.io.*;
import java.util.ArrayList;
import java.util.List;

import org.json.JSONException;
import org.json.JSONObject;

import com.openexchange.usm.util.Toolkit;

public class OutlookLogAnalyzer {

	public static void main(String[] args) {
		try {
			File dir = new File(args[0]).getCanonicalFile();
			File[] files = dir.listFiles();
			for (File file : files) {
				if (file.isFile()) {
					OutlookLogAnalyzer o = new OutlookLogAnalyzer(file);
					o.analyze();
				}
			}
		} catch (Exception e) {
			e.printStackTrace();
		}
	}

	private final File fLogFile;

	private OutlookLogAnalyzer(File logFile) {
		super();
		fLogFile = logFile;
	}

	private void analyze() throws Exception {
		String[] lines = readLines();
		OutlookLogRoundtrip[] roundTrips = createRoundTrips(lines);
		OutlookLogSession[] sessions = createSessions(roundTrips);
		printSessions(sessions);
	}

	private void printSessions(OutlookLogSession[] sessions) throws Exception {
		if (sessions.length == 0) {
			return;
		}
		File dir = new File(fLogFile.getParentFile(), fLogFile.getName() + ".analyzed");
		deleteDirRecursive(dir);
		dir.mkdirs();
		for (OutlookLogSession session : sessions) {
			printSession(session, dir);
		}
	}

	private void printSession(OutlookLogSession session, File dir) throws Exception {
		session.getFolderHierarchySyncs();
		File f = new File(dir, session.getUserName() + "_" + session.fSessionId + ".log");
		BufferedWriter out = null;
		try {
			out = new BufferedWriter(new OutputStreamWriter(new FileOutputStream(f), "UTF-8"));
			for (OutlookLogRoundtrip trip : session.fRoundTrips) {
				trip.print(out, session);
			}
		} finally {
			Toolkit.close(out);
		}
		System.out.println("printed: " + f);
	}

	private OutlookLogSession[] createSessions(OutlookLogRoundtrip[] roundTrips) throws JSONException {
		List<OutlookLogSession> sessions = new ArrayList<OutlookLogSession>();
		OutlookLogSession tmpSession = null;
		for (OutlookLogRoundtrip trip : roundTrips) {
			tmpSession = findSession(sessions, trip);
			if (tmpSession == null) {
				tmpSession = new OutlookLogSession(trip.getSessionId());
				sessions.add(tmpSession);
			}
			tmpSession.fRoundTrips.add(trip);
		}
		return sessions.toArray(new OutlookLogSession[sessions.size()]);
	}

	private OutlookLogSession findSession(List<OutlookLogSession> sessions, OutlookLogRoundtrip trip)
			throws JSONException {
		String sessionId = trip.getSessionId();
		for (OutlookLogSession session : sessions) {
			if (session.fSessionId.equals(sessionId)) {
				return session;
			}
		}
		return null;
	}

	private OutlookLogRoundtrip[] createRoundTrips(String[] lines) throws JSONException {
		List<OutlookLogRoundtrip> trips = new ArrayList<OutlookLogRoundtrip>();
		OutlookLogRoundtrip tmpTrip = null;
		StringBuilder tmpJson = null;
		boolean isRequest = false;
		int idx;
		for (String line : lines) {
			if ((idx = line.indexOf("FCT=")) >= 0) {
				tmpTrip = new OutlookLogRoundtrip();
				trips.add(tmpTrip);
				tmpTrip.fCommand = line.substring(idx + 4, line.length());
				tmpJson = null;
			} else if ((idx = line.indexOf("ARG=")) >= 0) {
				tmpJson = new StringBuilder(256);
				isRequest = true;
				tmpTrip.fRequestTime = line.substring(0, 11);
			} else if ((idx = line.indexOf("RES=")) >= 0) {
				tmpJson = new StringBuilder(256);
				isRequest = false;
				tmpTrip.fResponseTime = line.substring(0, 11);
			} else if (tmpJson != null) {
				tmpJson.append(line);
				tmpJson.append("\r\n");
				if ("}".equals(line)) {
					JSONObject json = new JSONObject(tmpJson.toString());
					if (isRequest) {
						tmpTrip.fRequest = json;
					} else {
						tmpTrip.fResponse = json;
					}
					tmpJson = null;
				}
			}
		}
		return trips.toArray(new OutlookLogRoundtrip[trips.size()]);
	}

	private String[] readLines() throws IOException {
		List<String> logEntries = new ArrayList<String>();
		BufferedReader in = null;
		try {
			in = new BufferedReader(new InputStreamReader(new FileInputStream(fLogFile), "UTF-8"));
			String line = in.readLine();
			while (line != null) {
				logEntries.add(line);
				line = in.readLine();
			}
			return logEntries.toArray(new String[logEntries.size()]);
		} finally {
			Toolkit.close(in);
		}
	}

	private void deleteDirRecursive(File dir) throws IOException {
		deleteDirContents(dir);
		deleteFileOrDir(dir);
	}

	private void deleteDirContents(File dir) throws IOException {
		if (!dir.exists()) {
			return;
		}
		File[] children = dir.listFiles();
		for (File child : children) {
			if (child.isFile()) {
				deleteFileOrDir(child);
			} else {
				deleteDirRecursive(child);
			}
		}
	}

	private void deleteFileOrDir(File fileOrDir) throws IOException {
		if (!fileOrDir.exists()) {
			return;
		}
		boolean success = fileOrDir.delete();
		if (!success) {
			throw new IOException("Could not delete " + fileOrDir);
		}
	}

}
