/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2010 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */
package com.openexchange.usm.json.test;

import java.io.*;
import java.net.URL;
import java.util.ArrayList;
import java.util.List;

import junit.framework.TestCase;

import org.apache.commons.httpclient.Header;
import org.apache.commons.httpclient.HttpClient;
import org.apache.commons.httpclient.methods.PutMethod;
import org.apache.commons.httpclient.methods.StringRequestEntity;
import org.json.JSONException;
import org.json.JSONObject;

import com.openexchange.usm.util.Toolkit;

/**
 *
 * @author ibr
 *
 */
public class USMJsonTestCase extends TestCase {

	private static final String WAIT = "wait";
	private static final String TEXT_JAVASCRIPT_CONTENTTYPE = "text/javascript";
	private static final String CHARSET_UTF_8 = "UTF-8";

	private static class CallInfo {

		public final String _call;
		public final String _callRequest;
		public final JSONObject _expectedResponse;
		public boolean _parseFolders = false;
		public boolean _doNotPrintResult = false;

		CallInfo(String call, String callRequest, JSONObject expectedResponse) {
			_call = call;
			_callRequest = callRequest;
			_expectedResponse = expectedResponse;
		}

		@Override
		public String toString() {
			return new StringBuilder(2048).append(_call).append('\n').append(_callRequest).append('\n').append(
					_expectedResponse).toString();
		}
	}

	private final List<CallInfo> callList = new ArrayList<CallInfo>();
	private final HttpClient _httpclient = new HttpClient();
	private final JSONTestReplacer _replacer;
	private final USMJsonTestConfig _config;
	private final File _testFile;

	public USMJsonTestCase(USMJsonTestConfig cfg, File testFile, String testCaseName) {
		super(testCaseName);
		_config = cfg;
		_testFile = testFile;
		_replacer = new JSONTestReplacer(cfg);
	}

	@Override
	protected void setUp() {
		try {
			super.setUp();
			System.out.println();
			System.out.println();
			System.out
					.println("---------------------------------------------------------------------------------------------");
			System.out.println(_testFile);
			System.out
					.println("---------------------------------------------------------------------------------------------");
			System.out.println();
			loadCalls();
			JSONObject loginResponse = performUSMAccess(-2, "{user:\"" + _config.getUsmUser()
					+ "\", device:\"android\", password:\"" + _config.getUsmPassword() + "\"}", "login");
			String error = loginResponse.optString("oxError");
			if (Toolkit.provided(error)) {
				fail(error);
			}
			JSONObject logoutParams = new JSONObject();
			logoutParams.put("sessionid", loginResponse.getJSONObject("data").getString("sessionid"));
			logoutParams.put("endSynchronization", true);
			JSONObject logoutResponse = performUSMAccess(-1, logoutParams.toString(), "logout");
			assertEquals("Finalized logout during test setup failed", 1, logoutResponse.getInt("status"));
		} catch (Exception e) {
			e.printStackTrace();
			fail("Error while initializing test: " + e.getMessage());
		}
	}

	@Override
	public void runTest() throws Throwable {
		int i = 1;
		for (CallInfo callInfo : callList) {
			if (callInfo._call.startsWith(WAIT)) {
				long duration = 10000L;
				try {
					duration = Long.parseLong(callInfo._call.substring(WAIT.length()).trim());
				} catch (Exception ignored) {
				}
				try {
					System.out.printf("Waiting for %.0f seconds\n", duration / 1000.0);
					Thread.sleep(duration);
				} catch (InterruptedException ignored) {
				}
			} else {
				executeJSONCall(i++, callInfo);
			}
		}
	}

	public void executeJSONCall(int n, CallInfo callInfo) throws Exception {
		String request = callInfo._callRequest;
		try {
			JSONObject requestBody = new JSONObject(request);
			_replacer.replaceAll(requestBody, _testFile);
			request = requestBody.toString();
			logJSONObject("request", callInfo._call, n, requestBody);
		} catch (JSONException e) {
			System.out.println(e);
		}
		JSONObject result = performUSMAccess(n, request, callInfo._call);
		if (result == null)
			fail("the response object is null");
		try {
			if (callInfo._parseFolders) {
				_replacer.parseFolders(result, n);
			} else {
				_replacer.compareAndExtract(n, callInfo._expectedResponse, result);
			}
		} finally {
			if (!callInfo._doNotPrintResult)
				logJSONObject("result", callInfo._call, n, result);
		}
	}

	private void logJSONObject(String type, String call, int callNr, JSONObject obj) {
		System.out.println();
		System.out.println(call + " " + type + " (call " + callNr + "):");
		try {
			System.out.println(obj.toString(3));
		} catch (JSONException e) {
			e.printStackTrace();
		}
	}

	private JSONObject performUSMAccess(int n, String requestBody, String call) throws Exception {
		URL url = new URL(_config.getUsmBaseUrl(), call);
		String urlString = url.toExternalForm();
		PutMethod httppost = new PutMethod(urlString);
		httppost.setRequestEntity(new StringRequestEntity(requestBody, TEXT_JAVASCRIPT_CONTENTTYPE, CHARSET_UTF_8));
		httppost.setRequestHeader(new Header("Content-Type", TEXT_JAVASCRIPT_CONTENTTYPE));
		int statusCode = _httpclient.executeMethod(httppost);
		String responseBodyAsString = httppost.getResponseBodyAsString();
		assertEquals("Call " + n + ": HTTP status code mismatch", 200, statusCode);
		return new JSONObject(responseBodyAsString);
	}

	private void loadCalls() throws JSONException {
		readCalls(_testFile, 0);
	}

	private void readCalls(File testFile, int counter) throws JSONException {
		String call = null;
		String request = null;
		String line;
		BufferedReader reader = null;
		int linenumber = 0;
		try {
			reader = new BufferedReader(new FileReader(testFile));
			while ((line = reader.readLine()) != null) {
				linenumber++;
				if (line.trim().length() > 0 && line.charAt(0) != '#') {
					counter++;
					if (counter == 1) {
						call = line;
						if (line.startsWith(WAIT)) {
							counter = 0;
							callList.add(new CallInfo(call, null, null));
						}
					} else if (counter == 2) {
						request = line;
					} else if (counter == 3) {
						callList.add(new CallInfo(call, request, new JSONObject(line)));
						counter = 0;
					}
				} else if (line.startsWith("#include ")) {
					File includeFile = findFileRecursive(_config.getProjectDir(), line.substring(9));
					readCalls(includeFile, counter);
				} else if (line.startsWith("#parseFolders")) {
					CallInfo ci = new CallInfo(call, request, null);
					ci._parseFolders = true;
					if (line.contains("doNotPrintResult"))
						ci._doNotPrintResult = true;
					callList.add(ci);
					counter = 0;
				}
			}
		} catch (JSONException e) {
			fail("Invalid JSONObject at line " + linenumber);
		} catch (FileNotFoundException e) {
			fail("Could not load test file: " + testFile);
		} catch (IOException e) {
			fail("Could not load test file: " + testFile);
		} finally {
			Toolkit.close(reader);
		}
	}

	private File findFileRecursive(File folder, String filename) {
		if (".svn".equals(folder.getName())) {
			return null;
		}
		File[] children = folder.listFiles();
		for (File child : children) {
			if (child.isDirectory()) {
				File found = findFileRecursive(child, filename);
				if (found != null) {
					return found;
				}
			} else {
				if (child.getName().equals(filename)) {
					return child;
				}
			}
		}
		return null;
	}

}
