/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2010 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */
package com.openexchange.usm.json.test;

import java.io.*;

import junit.framework.TestCase;

import org.json.JSONException;
import org.json.JSONObject;

import com.openexchange.usm.api.contenttypes.ContentTypeField;
import com.openexchange.usm.api.datatypes.DataType;
import com.openexchange.usm.api.session.DataObject;
import com.openexchange.usm.mimemail.MimeMailBuilder;

/**
 * The JUnit test case class. This class overrides the TestCase.
 * The expected mime mail (as String) and JSON structure are read from test files.
 * From the JSON structure will be created the mime mail with the aid of MimeMailBuilder.
 * The expected String and created String will be compared.
 * @author ibr
 *
 */

public class MimeMailBuilderTestCase extends TestCase {

	private final String _fileName;
	private final File _fileDirectory;

	private String _jsonString;
	private String _expectedMimeStr;

	public MimeMailBuilderTestCase(String name, String fileName, File directory) {
		super(name);
		_fileName = fileName;
		_fileDirectory = directory;
	}

	@Override
	protected void setUp() {
		try {
			super.setUp();
			load();
		} catch (Exception e) {
			e.printStackTrace();
			fail("Error while initializing test: " + e.getMessage());
		}
	}

	@Override
	protected void tearDown() throws Exception {
		super.tearDown();
	}

	@Override
	public void runTest() throws Throwable {
		JSONObject jsonObj = new JSONObject(_jsonString);

		MimeMailBuilder builder = new MimeMailBuilder();
		String actualMimeStr = builder.convertJSONObjectToMimeMail(jsonObj, true);
		System.out.println(actualMimeStr);
		_expectedMimeStr = replaceBoundaries(actualMimeStr, _expectedMimeStr);
		assertEquals(_expectedMimeStr.trim(), actualMimeStr.trim());

	}

	private String replaceBoundaries(String mime, String originalMime) {
		if (mime.length() == 0)
			return originalMime;

		String boundaryPrefix = "------=_Part_";
		String boundaryInitPrefix = "boundary=\"----=_Part_";

		originalMime = replaceBoundary(mime, originalMime, boundaryPrefix/*, 37*/);

		originalMime = replaceBoundary(mime, originalMime, boundaryInitPrefix/*, 46*/);

		return originalMime;
	}

	private String replaceBoundary(String mime, String originalMime, String boundaryPrefix) {
		int posSimul = 0;
		int posOrigin = 0;

		int length = mime.length();
		int originalLength = originalMime.length();
		while (posSimul < length && posOrigin < originalLength) {
			int nextPosSimul = mime.indexOf(boundaryPrefix, posSimul);
			if (nextPosSimul < 0)
				break;
			int boundaryLen = boundaryPrefix.length();
			while (nextPosSimul + boundaryLen < length && !isEndBoundary(mime.charAt(nextPosSimul + boundaryLen)))
				boundaryLen++;

			String boundary = mime.substring(nextPosSimul, nextPosSimul + boundaryLen);
			int nextPosOrigin = originalMime.indexOf(boundaryPrefix, posOrigin);
			int firstCtrl = nextPosOrigin + boundaryPrefix.length();
			while (firstCtrl < originalLength && !isEndBoundary(originalMime.charAt(firstCtrl)))
				firstCtrl++;
			if (nextPosOrigin < 0)
				break;
			StringBuilder replaceBuffer = new StringBuilder(originalLength);
			replaceBuffer.append(originalMime.substring(0, nextPosOrigin)).append(boundary).append(
					originalMime.substring(firstCtrl, originalLength));
			originalMime = replaceBuffer.toString();
			originalLength = originalMime.length();
			// step
			posSimul = nextPosSimul + boundary.length();
			posOrigin = nextPosOrigin + boundary.length();
		}
		return originalMime;
	}

	private boolean isEndBoundary(char ch) {
		return ch != '.' && ch != '_' && ch != '-' && ch != '"' && !Character.isDigit(ch);
	}

	private void load() throws JSONException {
		BufferedReader reader = null;
		StringBuilder buffer = new StringBuilder(100);
		StringBuilder testMimeBuffer = new StringBuilder(100);
		String separator = "#####";
		try {
			reader = new BufferedReader(new FileReader(_fileDirectory.getPath() + "/" + _fileName));
			String line;

			boolean isJsonStr = true;
			while ((line = reader.readLine()) != null) {
				if (line.startsWith(separator)) {
					isJsonStr = false;
					continue;
				}
				if (isJsonStr)
					buffer.append(line);
				else
					testMimeBuffer.append(line).append("\r\n");
			}
			_jsonString = buffer.toString();
			_expectedMimeStr = testMimeBuffer.toString();

		} catch (FileNotFoundException e) {
			fail("Could not load test file: " + _fileDirectory.getPath() + "/" + _fileName);
		} catch (IOException e) {
			fail("Could not load test file: " + _fileDirectory.getPath() + "/" + _fileName);
		} finally {
			if (reader != null) {
				try {
					reader.close();
				} catch (IOException ignored) {
				}
			}
		}
	}

	protected void updateDataObjectFieldFromJSONObject(JSONObject source, ContentTypeField field, DataObject destination) {
		String fieldName = field.getFieldName();
		DataType<?> fieldType = field.getFieldType();
		try {
			destination.setFieldContent(fieldName, fieldType.extractFromJSONObject(destination.getSession(), source,
					fieldName));
		} catch (JSONException e) {
			fail("reading: " + fieldName + " with the " + fieldType + "  " + e.getMessage());
		}
	}
}
