/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2010 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */
package com.openexchange.usm.json.test;

import java.io.File;
import java.sql.Timestamp;
import java.util.*;

import junit.framework.AssertionFailedError;
import junit.framework.ComparisonFailure;

import org.json.*;

import com.openexchange.usm.util.JSONToolkit;

public class JSONTestReplacer {
	private static final String MARKER = "!!";
	private static final String INTEGER = "!!integer!!";
	private static final String INTEGER_STRING = "!!integer_string!!";
	private static final String UUID_KEY = "!!uuid!!";
	private static final String ANYTHING = "!!anything!!";
	private static final String NEW_UUID = "!!new:";
	private static final String USER = "!!user!!";
	private static final String PASSWORD = "!!password!!";
	private static final String IGNORED_ADDITIONAL_FIELDS = "!!ignored_additional_fields!!";
	private static final String CURRENT_TIMESTAMP = "!!current_timestamp"; // can be increased or decreased, e.g. !!current_timestamp-1000!!, !!current_timestamp+1000!!
	private static final String TEST_NAME = "!!testname!!"; // newly created name which can be used e.g. for calendar entry names, mail subjects etc
	private static final String MODIFY_UUID = "!!modify_uuid:"; // modify the UUID of a shared folder analogue to SessionManagerImpl.getModifedUUID(UUID, int)
	private static final String FIND = "!!find:";

	private final Map<String, Object> _storedValues = new HashMap<String, Object>();
	private final USMJsonTestConfig _config;

	JSONTestReplacer(USMJsonTestConfig config) {
		super();
		_config = config;
	}

	public void replaceAll(JSONObject data, File testfile) throws JSONException {
		for (String key : JSONToolkit.keys(data)) {
			Object v = data.opt(key);
			if (v instanceof String) {
				data.put(key, replaceValue((String) v, testfile));
			} else if (v instanceof JSONObject) {
				replaceAll((JSONObject) v, testfile);
			} else if (v instanceof JSONArray) {
				replaceAll((JSONArray) v, testfile);
			} else {
				// do nothing
			}
		}
	}

	public void replaceAll(JSONArray data, File testfile) throws JSONException {
		int len = data.length();
		for (int i = 0; i < len; i++) {
			Object v = data.opt(i);
			if (v instanceof String) {
				data.put(i, replaceValue((String) v, testfile));
			} else if (v instanceof JSONObject) {
				replaceAll((JSONObject) v, testfile);
			} else if (v instanceof JSONArray) {
				replaceAll((JSONArray) v, testfile);
			} else {
				// do nothing
			}
		}
	}

	private Object replaceValue(String v, File testfile) {
		if (!v.startsWith(MARKER) || !v.endsWith(MARKER))
			return v;
		if (USER.equals(v))
			return _config.getUsmUser();
		if (PASSWORD.equals(v))
			return _config.getUsmPassword();
		if (TEST_NAME.equals(v))
			return getRelativePath(testfile) + " " + new Timestamp(System.currentTimeMillis());
		if (v.startsWith(CURRENT_TIMESTAMP)) {
			long timestamp = System.currentTimeMillis();
			int endMarkerIdx = v.indexOf(MARKER, MARKER.length());
			String offsetString = v.substring(CURRENT_TIMESTAMP.length(), endMarkerIdx);
			if (offsetString.length() > 0) {
				if (offsetString.charAt(0) == '+')
					offsetString = offsetString.substring(1);
				long offset = Long.valueOf(offsetString);
				timestamp += offset;
			}
			return timestamp;
		}
		if (v.startsWith(MODIFY_UUID)) {
			String params = v.substring(MODIFY_UUID.length(), v.length() - MARKER.length());
			String uuidVariable = MARKER + params.split(",")[0] + MARKER;
			String userId = params.split(",")[1];
			if (!_storedValues.containsKey(uuidVariable))
				throw new AssertionFailedError("Variable " + uuidVariable + " was not retrieved in previous call");
			String oldUuid = _storedValues.get(uuidVariable).toString();
			String newUuid = getModifedUUID(oldUuid, userId);
			return newUuid;
		}
		Object o = _storedValues.get(v);
		if (o == null) {
			if (!v.startsWith(NEW_UUID))
				throw new AssertionFailedError("Pattern " + v + " was not retrieved in previous call");
			String realKey = MARKER + v.substring(NEW_UUID.length());
			if (_storedValues.containsKey(realKey))
				throw new AssertionFailedError("Pattern " + v + " was already generated in previous call");
			o = UUID.randomUUID().toString();
			_storedValues.put(realKey, o);
		}
		return o;
	}

	public void compareAndExtract(int n, JSONObject expectedResponse, JSONObject result) throws JSONException {
		doCompareAndExtract(expectedResponse, result);
		String s1 = expectedResponse.toString(2);
		String s2 = result.toString(2);
		if (!s1.equals(s2))
			throw new ComparisonFailure("Call " + n + " result mismatch:\n", s1, s2);
	}

	private JSONArray doCompareAndExtract(JSONArray o1, JSONArray o2) throws JSONException {
		int len = o1.length();
		for (int i = 0; i < len; i++) {
			Object v = o1.opt(i);
			if (v instanceof String) {
				String s = (String) v;
				if (o2 != null && i == len - 1 && s.equals(IGNORED_ADDITIONAL_FIELDS)) {
					int len2 = o2.length();
					for (; i < len2; i++) {
						o1.put(i, o2.opt(i));
					}
					return o1;
				} else if (s.equals(INTEGER)) {
					if (o2 != null) {
						try {
							o1.put(i, o2.getLong(i));
						} catch (JSONException ignored) {
						}
					}
				} else if (s.equals(INTEGER_STRING)) {
					if (o2 != null) {
						try {
							o1.put(i, String.valueOf(o2.getLong(i)));
						} catch (JSONException ignored) {
						}
					}
				} else if (s.equals(UUID_KEY)) {
					if (o2 != null) {
						try {
							String sv = o2.getString(i);
							UUID.fromString(sv);
							o1.put(i, sv);
						} catch (IllegalArgumentException ignored) {
						} catch (JSONException ignored) {
						}
					}
				} else if (s.equals(ANYTHING)) {
					if (o2 != null) {
						try {
							o1.put(i, o2.get(i));
						} catch (JSONException ignored) {
						}
					}
				} else if (s.startsWith(MARKER) && s.endsWith(MARKER)) {
					if (s.startsWith(FIND)) {
						searchAndStoreValue(buildSearchArray(s), o2);
						JSONArray ot = new JSONArray();
						for (int j = 0; j < len; j++) {
							if (j != i)
								ot.put(o1.get(j));
						}
						o1 = ot;
						i--;
						len = o1.length();
					} else {
						Object o = _storedValues.get(s);
						try {
							if (o != null)
								o1.put(i, o);
							else {
								if (o2 != null) {
									o = o2.get(i);
									o1.put(i, o);
									_storedValues.put(s, o);
								}
							}
						} catch (JSONException ignored) {
						}
					}
				}
			} else if (v instanceof JSONObject) {
				if (o2 != null)
					doCompareAndExtract((JSONObject) v, o2.optJSONObject(i));
			} else if (v instanceof JSONArray) {
				if (o2 != null)
					o1.put(i, doCompareAndExtract((JSONArray) v, o2.optJSONArray(i)));
			} else {
				// do nothing
			}
		}
		return o1;
	}

	private boolean searchAndStoreValue(String[] searchData, JSONArray a) {
		if (a != null) {
			int len = a.length();
			for (int i = 0; i < len; i++) {
				Object v = a.opt(i);
				if (v instanceof JSONArray) {
					if (searchAndStoreValue(searchData, (JSONArray) v))
						return true;
				} else if (v instanceof JSONObject) {
					if (searchAndStoreValue(searchData, (JSONObject) v))
						return true;
				}
			}
		}
		return false;
	}

	private boolean searchAndStoreValue(String[] searchData, JSONObject o) {
		if (o != null) {
			if (searchData[1].equals(o.optString(searchData[0], null))) {
				for (int i = 2; i < searchData.length; i += 2)
					_storedValues.put(MARKER + searchData[i] + MARKER, o.opt(searchData[i + 1]));
				return true;
			}
			for (String key : JSONToolkit.keys(o)) {
				Object v = o.opt(key);
				if (v instanceof JSONArray) {
					if (searchAndStoreValue(searchData, (JSONArray) v))
						return true;
				} else if (v instanceof JSONObject) {
					if (searchAndStoreValue(searchData, (JSONObject) v))
						return true;
				}
			}
		}
		return false;
	}

	private String[] buildSearchArray(String s) throws JSONException {
		try {
			s = s.substring(FIND.length(), s.length() - MARKER.length());
			String[] parts = s.split("[:=]");
			if ((parts.length & 1) == 0)
				return parts;
		} catch (Exception ignored) {
		}
		throw new JSONException("Illegal search string '" + s + "'");
	}

	private void doCompareAndExtract(JSONObject expected, JSONObject actual) throws JSONException {
		//see if in the expected result some keys need to be replaced
		JSONArray keys = new JSONArray();
		for (String key : JSONToolkit.keys(expected)) {
			keys.put(key);
		}
		try {
			int length = keys.length();
			for (int i = 0; i < length; i++) {
				String key = keys.getString(i);
				if (_storedValues.containsKey(key)) {
					Object value = expected.get(key);
					String newKey = (String) _storedValues.get(key);
					expected.remove(key);
					expected.put(newKey, value);
				}
			}
		} catch (JSONException ignored) {

		}
		//do the replace
		for (String key : JSONToolkit.copyOfKeys(expected)) {
			Object v = expected.opt(key);
			if (v instanceof String) {
				String s = (String) v;
				if (key.equals(IGNORED_ADDITIONAL_FIELDS)) {
					ignoreAdditionalFields(expected, actual, s);
				} else if (s.equals(INTEGER)) {
					if (actual != null) {
						try {
							expected.put(key, actual.getLong(key));
						} catch (JSONException ignored) {
						}
					}
				} else if (s.equals(INTEGER_STRING)) {
					if (actual != null) {
						try {
							expected.put(key, String.valueOf(actual.getLong(key)));
						} catch (JSONException ignored) {
						}
					}
				} else if (s.equals(UUID_KEY)) {
					if (actual != null) {
						try {
							String sv = actual.getString(key);
							UUID.fromString(sv);
							expected.put(key, sv);
						} catch (IllegalArgumentException ignored) {
						} catch (JSONException ignored) {
						}
					}
				} else if (s.equals(ANYTHING)) {
					if (actual != null) {
						try {
							expected.put(key, actual.get(key));
						} catch (JSONException ignored) {
						}
					}
				} else if (s.startsWith(MODIFY_UUID)) {
					String params = s.substring(MODIFY_UUID.length(), s.length() - MARKER.length());
					String uuidVariable = MARKER + params.split(",")[0] + MARKER;
					String userId = params.split(",")[1];
					if (!_storedValues.containsKey(uuidVariable))
						throw new AssertionFailedError("Variable " + uuidVariable
								+ " was not retrieved in previous call");
					String oldUuid = _storedValues.get(uuidVariable).toString();
					String newUuid = getModifedUUID(oldUuid, userId);
					expected.put(key, newUuid);
				} else if (s.startsWith(MARKER) && s.endsWith(MARKER)) {
					if (s.startsWith(FIND)) {
						searchAndStoreValue(buildSearchArray(s), actual);
						expected.remove(s);
					} else {
						Object o = _storedValues.get(s);
						try {
							if (o != null)
								expected.put(key, o);
							else {
								if (actual != null) {
									o = actual.get(key);
									expected.put(key, o);
									_storedValues.put(s, o);
								}
							}
						} catch (JSONException ignored) {
						}
					}
				}
			} else if (v instanceof JSONObject) {
				if (actual != null)
					doCompareAndExtract((JSONObject) v, actual.optJSONObject(key));
			} else if (v instanceof JSONArray) {
				if (actual != null)
					expected.put(key, doCompareAndExtract((JSONArray) v, actual.optJSONArray(key)));
			} else {
				// do nothing
			}
		}
	}

	private String getModifedUUID(String uuidString, String userIDString) {
		int userID = Integer.parseInt(userIDString);
		if (uuidString == null || userID == 0)
			return uuidString;
		UUID oldUuid = UUID.fromString(uuidString);
		UUID newUuid = new UUID(oldUuid.getMostSignificantBits() ^ userID, oldUuid.getLeastSignificantBits());
		return newUuid.toString();
	}

	private void ignoreAdditionalFields(JSONObject expected, JSONObject actual, String numberToIgnore)
			throws JSONException {
		expected.remove(IGNORED_ADDITIONAL_FIELDS);
		if (actual == null)
			return;
		int numberToIgnoreInt = "all".equals(numberToIgnore) ? Integer.MAX_VALUE : Integer.parseInt(numberToIgnore);
		for (String key : JSONToolkit.keys(actual)) {
			if (numberToIgnoreInt <= 0) {
				return;
			}
			if (!expected.has(key)) {
				expected.put(key, actual.get(key));
				numberToIgnoreInt--;
			}
		}
	}

	private String getRelativePath(File testfile) {
		return "/" + testfile.getParentFile().getName() + "/" + testfile.getName();
		//		String path = testfile.getName();
		//		File folder = testfile.getParentFile();
		//		while (folder != null && !folder.getName().startsWith("com.openexchange")) {
		//			path = folder.getName() + "/" + path;
		//			folder = folder.getParentFile();
		//		}
		//		return path;
	}

	void parseFolders(JSONObject result, int callNumber) throws JSONException {
		System.out.println();
		System.out.println("Folder overview:");
		JSONObject data = result.getJSONObject("data");
		_storedValues.put("!!syncid" + callNumber + "!!", data.getString("syncid"));
		JSONArray created = data.getJSONArray("created");
		int length = created.length();
		for (int i = 0; i < length; i++) {
			JSONObject folder = created.getJSONObject(i);
			printOverview(folder, i + 1, "type", "created_by", "uuid", "id");
			storeFolderValues(folder, "uuid", "id", "module", "type", "created_by", "permissions");
		}
	}

	private void storeFolderValues(JSONObject folder, String... keys) throws JSONException {
		String title = folder.getString("title");
		for (String key : keys) {
			String storeKey = "!!folder." + title + "." + key + "!!";
			String storeValue = folder.optString(key);
			_storedValues.put(storeKey, storeValue);
		}
	}

	private void printOverview(JSONObject folder, int number, String... keys) throws JSONException {
		String numberString = "(" + number + ")";
		while (numberString.length() < 6) {
			numberString = " " + numberString;
		}
		System.out.print(numberString);
		System.out.print(' ');
		String title = folder.getString("title");
		while (title.length() < 25) {
			title += " ";
		}
		System.out.print(title);
		System.out.print('	');
		for (String key : keys) {
			System.out.print(key);
			System.out.print(": ");
			System.out.print(folder.optString(key));
			System.out.print("   ");
		}
		System.out.println();
	}

}
