/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2010 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */
package com.openexchange.usm.database.ox;

import java.sql.*;
import java.util.ArrayList;
import java.util.List;

import org.apache.commons.logging.Log;
import org.osgi.service.component.ComponentContext;

import com.openexchange.groupware.delete.*;
import com.openexchange.usm.api.database.DatabaseAccess;
import com.openexchange.usm.journal.Journal;
import com.openexchange.usm.util.Toolkit;

public class USMOXDeleteListener implements DeleteListener {
	private Log _journal;

	/*
	 * (non-Javadoc)
	 * @see org.osgi.framework.BundleActivator#start(org.osgi.framework.BundleContext)
	 */
	public void activate(ComponentContext context) {
		Journal journal = (Journal) context.locateService(Journal.class.getName());
		_journal = journal.getLog(DatabaseAccess.class);
		_journal.info("USMOXDeleteListener activated");
	}

	/*
	 * (non-Javadoc)
	 * @see org.osgi.framework.BundleActivator#stop(org.osgi.framework.BundleContext)
	 */
	public void deactivate(ComponentContext context) {
		_journal.info("USMOXDeleteListener deactivated");
		_journal = null;
	}

	public void deletePerformed(DeleteEvent event, Connection readCon, Connection writeCon)
			throws DeleteFailedException {
		if (event.getType() == DeleteEvent.TYPE_USER) {
			int userId = event.getId();
			int contextId = event.getContext().getContextId();
			if (_journal.isDebugEnabled())
				_journal.debug("Removing USM data for user " + userId + ", cid " + contextId);
			DatabaseMetaData dbmd = getDBMetaData(writeCon);
			List<Integer> sessions = getAllSessionsForUser(dbmd, writeCon, contextId, userId);
			//remove the user from all tables
			for (String table : TableDefinition.TABLE_NAMES) {
				if (!TableDefinition.SESSION_TABLE_NAME.equals(table))
					removeFromTable(dbmd, writeCon, table, contextId, sessions);
			}
			//at the end remove the user from the session table (holder of the primary key)
			removeFromTable(dbmd, writeCon, TableDefinition.SESSION_TABLE_NAME, contextId, sessions);
		}
	}

	private DatabaseMetaData getDBMetaData(Connection writeCon) throws DeleteFailedException {
		try {
			return writeCon.getMetaData();
		} catch (SQLException e) {
			_journal.error("SQL error when retreiving DB metadata", e);
			throw new DeleteFailedException(DeleteFailedException.Code.SQL_ERROR, e);
		}
	}

	private List<Integer> getAllSessionsForUser(DatabaseMetaData dbmd, Connection writeCon, int contextId, int userId)
			throws DeleteFailedException {
		PreparedStatement stmt = null;
		ResultSet result = null;
		List<Integer> resultList = new ArrayList<Integer>();
		try {
			if (USMBaseUpdateTask.tableExists(TableDefinition.SESSION_TABLE_NAME, dbmd)) {
				stmt = writeCon.prepareStatement("SELECT usmSessionId FROM " + TableDefinition.SESSION_TABLE_NAME
						+ " WHERE cid = ? AND id = ?");
				stmt.setInt(1, contextId);
				stmt.setInt(2, userId);
				result = stmt.executeQuery();
				while (result.next()) {
					int sid = result.getInt(1);
					resultList.add(sid);
				}
			}
		} catch (SQLException e) {
			_journal.error("SQL error when retreiving USM sessions for user " + userId, e);
			throw new DeleteFailedException(DeleteFailedException.Code.SQL_ERROR, e);
		} finally {
			Toolkit.close(result);
			Toolkit.close(stmt);
		}
		return resultList;
	}

	private void removeFromTable(DatabaseMetaData dbmd, Connection writeCon, String table, int contextId,
			List<Integer> sessions) throws DeleteFailedException {
		PreparedStatement stmt = null;
		try {
			if (USMBaseUpdateTask.tableExists(table, dbmd)) {
				for (Integer sessionId : sessions) {
					stmt = writeCon.prepareStatement("DELETE FROM " + table + " WHERE cid = ? AND usmSessionId = ?");
					stmt.setInt(1, contextId);
					stmt.setInt(2, sessionId);
					stmt.executeUpdate();
				}
			}
		} catch (SQLException e) {
			_journal.error("SQL error deleting USM user data", e);
			throw new DeleteFailedException(DeleteFailedException.Code.SQL_ERROR, e);
		} finally {
			Toolkit.close(stmt);
		}
	}
}
