/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2010 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */
package com.openexchange.usm.api.contenttypes;

import java.io.Serializable;
import java.util.*;

import org.json.*;

import com.openexchange.usm.api.datatypes.DataType;
import com.openexchange.usm.api.session.*;

/**
 * This is a nearly 100% copy of SimpleDataObject.
 * 
 * Base implementation of DataObject. It stores ID and ParentFolderID directly in
 * fields provided by the ContentType, by default "id" for ID and "folder_id" for
 * ParentFolderID. If a ContentType uses other field names for that information,
 * it can  use the constructor with additional parameters to specify the correct
 * field names.
 * 
 * @author afe
 *
 */
public class SimDataObject implements DataObject {

	static boolean _formatToString = false;
	protected static final boolean SKIP_UNSET_VALUES_FOR_TOSTRING = false;

	public static void setToStringFormatted(boolean formatToString) {
		_formatToString = formatToString;
	}

	protected final Session _session;

	protected final ContentType _contentType;

	protected final Map<String, Integer> _fieldNameToIndexMap;

	protected ChangeState _changeState = ChangeState.CREATED;

	protected final Object[] _originalFieldContent;

	protected final Object[] _fieldContent;

	protected long _timestamp, _originalTimestamp;

	private final int _idIndex;

	private final int _parentFolderIDIndex;

	private boolean _failed;

	private UUID _uuid;

	private DataObject _source;

	private Folder _parentFolder;

	private ProtocolInformation _protocolInfo;

	public SimDataObject(Session session, ContentType contentType) {
		this(session, contentType, "id", "folder_id");
	}

	public SimDataObject(Session session, ContentType contentType, String idFieldName, String parentFolderIdFieldName) {
		if (session == null || contentType == null)
			throw new IllegalArgumentException("Session and ContentType must be specified");
		_source = this;
		_session = session;
		_contentType = contentType;
		ContentTypeField[] fields = _contentType.getFields();
		int fieldCount = fields.length;
		_originalFieldContent = new Object[fieldCount];
		_fieldContent = new Object[fieldCount];
		for (int i = 0; i < fieldCount; i++) {
			DataType<?> fieldType = fields[i].getFieldType();
			_originalFieldContent[i] = fieldType.createNewObject();
			_fieldContent[i] = fieldType.createNewObject();
		}
		_fieldNameToIndexMap = new HashMap<String, Integer>();
		for (int i = 0; i < fields.length; i++)
			_fieldNameToIndexMap.put(fields[i].getFieldName(), i);
		_idIndex = _fieldNameToIndexMap.get(idFieldName);
		_parentFolderIDIndex = _fieldNameToIndexMap.get(parentFolderIdFieldName);
	}

	protected SimDataObject(SimDataObject source, boolean linkUUIDs) {
		_source = linkUUIDs ? source._source : this;
		_uuid = source._uuid;
		_session = source._session;
		_contentType = source._contentType;
		_fieldNameToIndexMap = source._fieldNameToIndexMap;
		_changeState = source._changeState;
		_timestamp = source._timestamp;
		_originalTimestamp = source._originalTimestamp;
		_idIndex = source._idIndex;
		_parentFolderIDIndex = source._parentFolderIDIndex;
		_failed = source._failed;
		_fieldContent = new Object[source._fieldContent.length];
		_originalFieldContent = new Object[source._originalFieldContent.length];
		ContentTypeField[] fields = _contentType.getFields();
		for (int i = 0; i < _fieldContent.length; i++) {
			DataType<?> fieldType = fields[i].getFieldType();
			_fieldContent[i] = fieldType.createCopy(source._fieldContent[i]);
			_originalFieldContent[i] = fieldType.createCopy(source._originalFieldContent[i]);
		}
	}

	public Session getSession() {
		return _session;
	}

	public ContentType getContentType() {
		return _contentType;
	}

	public ChangeState getChangeState() {
		return _changeState;
	}

	public void setChangeState(ChangeState newState) {
		if (newState != ChangeState.CREATED && newState != ChangeState.DELETED)
			throw new IllegalArgumentException("Changing from " + _changeState + " to " + newState + " not permitted");
		_changeState = newState;
	}

	public String getID() {
		return convertObjectToString((getFieldContent(_idIndex)));
	}

	public String getOriginalID() {
		return convertObjectToString((getOriginalFieldContent(_idIndex)));
	}

	public void setID(String id) {
		setFieldContent(_idIndex, id);
	}

	public long getTimestamp() {
		return _timestamp;
	}

	public long getOriginalTimestamp() {
		return _originalTimestamp;
	}

	public void setTimestamp(long timestamp) {
		_timestamp = timestamp;
		updateChangeState(_timestamp != _originalTimestamp);
	}

	public String getParentFolderID() {
		return convertObjectToString((getFieldContent(_parentFolderIDIndex)));
	}

	public String getOriginalParentFolderID() {
		return convertObjectToString(getOriginalFieldContent(_parentFolderIDIndex));
	}

	public void setParentFolderID(String folderID) {
		setFieldContent(_parentFolderIDIndex, folderID);
	}

	public Object getFieldContent(int fieldIndex) {
		return _fieldContent[fieldIndex];
	}

	public Object getOriginalFieldContent(int fieldIndex) {
		return _originalFieldContent[fieldIndex];
	}

	public void setFieldContent(int fieldIndex, Object data) {
		_fieldContent[fieldIndex] = _contentType.getFields()[fieldIndex].getFieldType().checkValue(data);
		if (fieldIndex == _parentFolderIDIndex)
			_parentFolder = null;
		updateChangeState(fieldIndex);
	}

	public Object getFieldContent(String fieldName) {
		return getFieldContent(getFieldIndex(fieldName));
	}

	public Object getOriginalFieldContent(String fieldName) {
		return getOriginalFieldContent(getFieldIndex(fieldName));
	}

	public void setFieldContent(String fieldName, Object data) {
		setFieldContent(getFieldIndex(fieldName), data);
	}

	public void setParentFolder(Folder folder) {
		setParentFolderID(folder.getID());
		_parentFolder = folder;
	}

	public Folder getParentFolder() {
		return _parentFolder;
	}

	public boolean isFieldModified(int fieldIndex) {
		return !_contentType.getFields()[fieldIndex].getFieldType().isEqual(_fieldContent[fieldIndex],
				_originalFieldContent[fieldIndex]);
	}

	public boolean isFieldModified(String fieldName) {
		return isFieldModified(getFieldIndex(fieldName));
	}

	public boolean isModified() {
		return _changeState != ChangeState.UNMODIFIED;
	}

	public void rollbackChanges() {
		_timestamp = _originalTimestamp;
		System.arraycopy(_originalFieldContent, 0, _fieldContent, 0, _fieldContent.length);
		_changeState = ChangeState.UNMODIFIED;
		_failed = false;
	}

	public void commitChanges() {
		_originalTimestamp = _timestamp;
		System.arraycopy(_fieldContent, 0, _originalFieldContent, 0, _fieldContent.length);
		_changeState = ChangeState.UNMODIFIED;
		_failed = false;
	}

	protected void updateChangeState(boolean newFieldContentIsNotEqual) {
		if (newFieldContentIsNotEqual) {
			if (_changeState == ChangeState.UNMODIFIED)
				_changeState = ChangeState.MODIFIED;
		} else {
			if (_changeState == ChangeState.MODIFIED && _timestamp == _originalTimestamp) {
				for (int i = 0; i < _fieldContent.length; i++) {
					if (isFieldModified(i))
						return;
				}
				_changeState = ChangeState.UNMODIFIED;
			}
		}
	}

	protected void updateChangeState(int fieldIndex) {
		updateChangeState(isFieldModified(fieldIndex));
	}

	public int getFieldIndex(String fieldName) {
		Integer index = _fieldNameToIndexMap.get(fieldName);
		if (index != null)
			return index;
		throw new IllegalArgumentException("Unknown field " + fieldName + " in " + _contentType.getID());
	}

	@Override
	public String toString() {
		return toString(null, SKIP_UNSET_VALUES_FOR_TOSTRING);
	}

	public String toString(BitSet requestedFields) {
		return toString(requestedFields, SKIP_UNSET_VALUES_FOR_TOSTRING);
	}

	public String toString(BitSet requestedFields, boolean skipUnsetFields) {
		StringBuilder sb = new StringBuilder(256);
		sb.append(getDataObjectName()).append('(').append(_session).append(',').append(_contentType.getID())
				.append(',').append(_changeState).append("){");
		if (_formatToString)
			sb.append("\n   ");
		sb.append("Timestamp:").append(_originalTimestamp);
		if (_timestamp != _originalTimestamp)
			sb.append("=>").append(_timestamp);
		addSpecialFieldsToStringBuilder(sb);
		ContentTypeField[] fields = _contentType.getFields();
		for (int i = 0; i < fields.length; i++) {
			if (requestedFields == null || requestedFields.get(i)) {
				DataType<?> fieldType = fields[i].getFieldType();
				addElementToStringBuilder(fieldType, sb, fields[i].getFieldName(), _fieldContent[i],
						_originalFieldContent[i], skipUnsetFields);
			}
		}
		if (_formatToString)
			sb.append('\n');
		return sb.append(",data:").append(getProtocolInformation()).append('}').toString();
	}

	protected void addSpecialFieldsToStringBuilder(StringBuilder sb) {
		// This method can be overwritten by sub-classes to add other special fields in the toString()-method
	}

	/**
	 * This can be overwritten by sub classes to provide information about implementing class in toString()-method
	 * @return
	 */
	protected String getDataObjectName() {
		return getClass().getSimpleName();
	}

	protected void addElementToStringBuilder(DataType<?> type, StringBuilder sb, String name, Object current,
			Object old, boolean skipUnsetFields) {
		boolean changed = !type.isEqual(current, old);
		if (skipUnsetFields && !changed && type.isEqual(type.createNewObject(), current))
			return;
		sb.append(',');
		if (_formatToString)
			sb.append("\n   ");
		sb.append(name).append(':');
		appendValueToStringBuilder(type, sb, old);
		if (changed) {
			sb.append("=>");
			appendValueToStringBuilder(type, sb, current);
		}
	}

	protected void appendValueToStringBuilder(DataType<?> type, StringBuilder sb, Object v) {
		JSONArray a = new JSONArray();
		try {
			type.addToJSONArray(_session, a, v);
			Object val = a.opt(0);
			if (val instanceof JSONArray)
				sb.append(((JSONArray) val).toString(0).replaceAll("\n", ""));
			else if (val instanceof JSONObject)
				sb.append(((JSONObject) val).toString(0).replaceAll("\n", ""));
			else
				sb.append(String.valueOf(val).replaceAll("\"", ""));
		} catch (JSONException e) {
			sb.append("<JSON_ERROR>");
		}
	}

	@Override
	public int hashCode() {
		final int prime = 31;
		int result = 1;
		String id = getID();
		result = prime * result + ((id == null) ? 0 : id.hashCode());
		result = prime * result + ((_session == null) ? 0 : _session.hashCode());
		return result;
	}

	@Override
	public boolean equals(Object obj) {
		if (this == obj)
			return true;
		if (obj == null)
			return false;
		if (getClass() != obj.getClass())
			return false;
		SimDataObject other = (SimDataObject) obj;
		return internalEquals(other, false);
	}

	public boolean equalsWithoutID(DataObject other) {
		if (!(other instanceof SimDataObject))
			return false;
		return internalEquals((SimDataObject) other, true);
	}

	private boolean internalEquals(SimDataObject other, boolean ignoreID) {
		if (!_changeState.equals(other._changeState))
			return false;
		if (!_contentType.equals(other._contentType))
			return false;
		if (!_session.equals(other._session))
			return false;
		//		if (_timestamp != other._timestamp)
		//			return false;
		//		if (_originalTimestamp != other._originalTimestamp)
		//			return false;
		ContentTypeField[] fields = _contentType.getFields();
		for (int i = 0; i < _fieldContent.length; i++) {
			if (!ignoreID || i != _idIndex) {
				DataType<?> type = fields[i].getFieldType();
				if (!type.isEqual(_fieldContent[i], other._fieldContent[i]))
					return false;
				if (!type.isEqual(_originalFieldContent[i], other._originalFieldContent[i]))
					return false;
			}
		}
		return true;
	}

	public DataObject createCopy(boolean linkUUIDs) {
		return new SimDataObject(this, linkUUIDs);
	}

	protected String convertObjectToString(Object o) {
		return (o == null) ? null : o.toString();
	}

	public boolean isFailed() {
		return _failed;
	}

	public void setFailed(boolean failed) {
		_failed = failed;
	}

	public Serializable[] serialize() {
		Serializable[] result = new Serializable[_fieldContent.length];
		ContentTypeField[] fields = _contentType.getFields();
		for (int i = 0; i < _fieldContent.length; i++) {
			DataType<?> fieldType = fields[i].getFieldType();
			result[i] = fieldType.createCopy(_fieldContent[i]);
		}
		return result;
	}

	public void deserialize(long timestamp, Serializable[] data) throws IllegalArgumentException {
		_changeState = ChangeState.UNMODIFIED;
		_timestamp = timestamp;
		_originalTimestamp = timestamp;
		_failed = false;
		ContentTypeField[] fields = _contentType.getFields();
		for (int i = 0; i < _fieldContent.length; i++) {
			DataType<?> fieldType = fields[i].getFieldType();
			_fieldContent[i] = fieldType.createCopy(data[i]);
			_originalFieldContent[i] = fieldType.createCopy(data[i]);
		}
	}

	public UUID getUUID() {
		return (_source == this) ? _uuid : _source.getUUID();
	}

	public void setUUID(UUID uuid) {
		if (_source == this)
			_uuid = uuid;
		else
			_source.setUUID(uuid);
	}

	public void linkUUIDTo(DataObject source) {
		if (source == null)
			source = this;
		if (source == this)
			_uuid = _source.getUUID();
		_source = source;
	}

	public ProtocolInformation getProtocolInformation() {
		return (_source == this) ? _protocolInfo : _source.getProtocolInformation();
	}

	public void setProtocolInformation(ProtocolInformation info) {
		if (_source == this)
			_protocolInfo = info;
		else
			_source.setProtocolInformation(info);
	}

	public int getParentFolderOwnerID() {
		// TODO Auto-generated method stub
		return 0;
	}

	public void setParentFolderOwnerID(int id) {
		// TODO Auto-generated method stub

	}
}
