/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2010 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */
package com.openexchange.build.ant.tasks;

import java.io.File;
import java.util.*;

import org.apache.tools.ant.*;
import org.apache.tools.ant.types.Path;

import com.openexchange.build.ant.data.*;

public class ComputeBuildOrder extends Task {
	private String buildType;
	private File dir;
	private String propertyName;
	private String projectList;
	private Path classpath;

	public void setBuildType(String buildType) {
		this.buildType = buildType;
	}

	public final void setProjectList(final String projectList) {
		this.projectList = projectList;
	}

	public final void setPropertyName(final String propertyName) {
		this.propertyName = propertyName;
	}

	public final void setDir(final File dir) {
		this.dir = dir;
	}

	public final Path createClasspath() {
		classpath = new Path(getProject());
		return classpath;
	}

	@Override
	public void execute() throws BuildException {

		// Parse ClassPath-Jars
		log("using classpath " + classpath, Project.MSG_INFO);
		String[] classpathFiles = classpath.list();
		List<AbstractModule> classpathModules = new ArrayList<AbstractModule>(
				classpathFiles.length);
		for (int i = 0; i < classpathFiles.length; i++) {
			File classpathFile = new File(classpathFiles[i]);
			if (classpathFile.isFile())
				classpathModules.add(new JarModule(classpathFile));
			else {
				DirModule classpathModule = new DirModule(classpathFile);
				classpathModule.readLocalFiles(getProject(), classpathFile
						.getParentFile());
				classpathModules.add(classpathModule);
			}
		}

		// Parse Projects
		File tempDir = new File(getProject().getBaseDir(), "temp");
		tempDir.mkdir(); // create the temp dir for remote read

		List<AbstractModule> appModules = new ArrayList<AbstractModule>();
		String[] split = projectList.split(",");
		for (int i = 0; i < split.length; i++) {
			DirModule module = new DirModule(split[i]);
			if ("local".equalsIgnoreCase(buildType)) {
				module.readLocalFiles(getProject(), dir);
			} else {
				module.readRemoteFiles(getProject(), tempDir);
			}
			appModules.add(module);
		}

		tempDir.delete();

		List<AbstractModule> allModules = new ArrayList<AbstractModule>(
				classpathModules.size() + appModules.size());
		allModules.addAll(classpathModules);
		allModules.addAll(appModules);
		log("all modules: " + allModules, Project.MSG_INFO);

		// compute dependencies
		Map<String, AbstractModule> modulesByName = new HashMap<String, AbstractModule>();
		for (Iterator<AbstractModule> modulesIt = allModules.iterator(); modulesIt
				.hasNext();) {
			AbstractModule module = modulesIt.next();
			modulesByName.put(module.getName(), module);
		}

		Map<String, Set<AbstractModule>> modulesByPackage = new HashMap<String, Set<AbstractModule>>();
		for (Iterator<AbstractModule> modulesIt = allModules.iterator(); modulesIt
				.hasNext();) {
			modulesIt.next().addToExportMap(modulesByPackage);
		}
		for (Iterator<AbstractModule> modulesIt = appModules.iterator(); modulesIt
				.hasNext();) {
			modulesIt.next().computeDependencies(modulesByName,
					modulesByPackage);
		}
		for (Iterator<AbstractModule> modulesIt = appModules.iterator(); modulesIt
				.hasNext();) {
			modulesIt.next().computeDependenciesForFragments();
		}

		List<AbstractModule> sortedModules = new DependenciesSorter()
				.sortDependencies(allModules);
		log("sorted modules: " + sortedModules, Project.MSG_INFO);

		sortedModules.removeAll(classpathModules); // only keep the app modules

		String moduleNamesList = getModuleNamesList(sortedModules, ',');
		getProject().setInheritedProperty(propertyName, moduleNamesList);

		for (Iterator<AbstractModule> modulesIt = sortedModules.iterator(); modulesIt
				.hasNext();) {
			AbstractModule module = modulesIt.next();
			String requiredClasspath = module.getRequiredClasspath(',');
			log(module.getName() + ".requiredClasspath: " + requiredClasspath,
					Project.MSG_INFO);
			getProject().setInheritedProperty(
					module.getName() + ".requiredClasspath", requiredClasspath);
		}
	}

	private String getModuleNamesList(Collection<AbstractModule> modules,
			char delimiter) {
		StringBuffer buffer = new StringBuffer();
		for (Iterator<AbstractModule> modulesIt = modules.iterator(); modulesIt
				.hasNext();) {
			buffer.append(modulesIt.next().getName());
			if (modulesIt.hasNext()) {
				buffer.append(delimiter);
			}
		}
		return buffer.toString();
	}
}