/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2010 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */
package com.openexchange.build.ant.data;

import java.io.File;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.tools.ant.BuildException;
import org.apache.tools.ant.Project;
import org.jdom.Document;
import org.jdom.Element;
import org.jdom.input.SAXBuilder;
import org.tigris.subversion.svnant.SvnTask;
import org.tigris.subversion.svnant.commands.Cat;
import org.tigris.subversion.svnclientadapter.SVNUrl;

public class DirModule extends AbstractModule {
	private File dir;
	private List<String> classpathDependencies = new LinkedList<String>();
	private Set<String> exportedClasspath = new HashSet<String>();
	private Set<String> requiredClasspath = new HashSet<String>();

	public DirModule(final String name) {
		this.name = name;
	}

	public DirModule(final File dir) {
		this.name = dir.getName();
	}

	private void readFiles(File manifestFile, File classpathFile) {
		try {
			if (manifestFile.exists() && manifestFile.length() != 0) {
				this.osgiManifest = new OSGIManifest(manifestFile);

				// read Bundle-ClassPath:
				Set<String> manifestClasspath = osgiManifest.getListEntry(OSGIManifest.BUNDLE_CLASSPATH);
				for (Iterator<String> iterator = manifestClasspath.iterator(); iterator.hasNext();) {
					String classpathEntry = iterator.next();
					if (!classpathEntry.equals("."))
						exportedClasspath.add(classpathEntry);
				}
			}

			if (classpathFile.exists() && classpathFile.length() != 0) {
				Document d = new SAXBuilder().build(classpathFile);
				Element root = d.getRootElement();
				List list = root.getChildren("classpathentry");
				Iterator it = list.iterator();
				while (it.hasNext()) {
					Element entry = (Element) it.next();
					if (entry.getAttribute("combineaccessrules") != null) {
						String path = entry.getAttributeValue("path");
						classpathDependencies.add(path.substring(1));
					} else if (entry.getAttributeValue("kind").equals("lib")) {
						String path = entry.getAttributeValue("path");
						requiredClasspath.add(path);

						if (entry.getAttributeValue("exported") != null
								&& entry.getAttributeValue("exported").equals("true")) {
							exportedClasspath.add(path);
						}
					}
				}
			}
		} catch (Exception e) {
			throw new BuildException(e);
		}
	}

	public void readLocalFiles(Project project, File rootDir) {
		dir = new File(rootDir, name);
		project.log("Project " + name + " reads local files from " + dir);
		readFiles(new File(dir, "/META-INF/MANIFEST.MF"), new File(dir, ".classpath"));
	}

	private File cat(Project project, String path, File tempDir) {
		String svnTimestamp = project.getProperty("svn.timestamp");
		String svnUrl = project.getProperty(name + ".svn.url");
		File file = new File(tempDir, path.substring(path.lastIndexOf('/') + 1));

		try {
			SvnTask svnTask = new SvnTask();
			Cat catTask = new Cat();
			catTask.setTask(svnTask);
			catTask.setProject(project);
			catTask.setUrl(new SVNUrl(svnUrl + '/' + path));
			catTask.setRevision(svnTimestamp);
			catTask.setDestFile(file);

			svnTask.setProject(project);
			svnTask.setJavahl(false);
			svnTask.setUsername(project.getProperty("svn.username"));
			svnTask.setPassword(project.getProperty("svn.password"));
			svnTask.addCat(catTask);
			svnTask.execute();
		} catch (Exception e) {
			project.log("Error while reading remote " + path, Project.MSG_ERR);
		}

		return file;
	}

	public void readRemoteFiles(Project project, File tempDir) {
		dir = new File(project.getProperty("checkout.dir"), name);

		String svnTimestamp = project.getProperty("svn.timestamp");
		String svnUrl = project.getProperty(name + ".svn.url");
		project.log("Project " + name + " reads remote files from " + svnUrl + "@" + svnTimestamp);

		File manifestFile = cat(project, "META-INF/MANIFEST.MF", tempDir);
		File classpathFile = cat(project, ".classpath", tempDir);

		readFiles(manifestFile, classpathFile);

		classpathFile.delete();
		manifestFile.delete();
	}

	@Override
	public void computeDependencies(Map<String, AbstractModule> projectsByName,
			Map<String, Set<AbstractModule>> projectsByPackage) {

		for (Iterator<String> classpathIt = classpathDependencies.iterator(); classpathIt.hasNext();) {
			String classpathProject = (String) classpathIt.next();
			AbstractModule module = projectsByName.get(classpathProject);
			if (module != null && module != this)
				dependencies.add(module);
		}

		super.computeDependencies(projectsByName, projectsByPackage);
	}

	public String getExportedClasspath(char delimiter) {
		StringBuffer buffer = new StringBuffer();
		buffer.append(dir.getAbsolutePath() + "/<bin>");
		for (Iterator<String> iterator = exportedClasspath.iterator(); iterator.hasNext();) {
			String classpathEntry = iterator.next();
			buffer.append(delimiter);
			buffer.append(dir.getAbsolutePath() + '/' + classpathEntry);
		}
		return buffer.toString();
	}

	public String getRequiredClasspath(char delimiter) {
		if (requiredClasspath.isEmpty()) {
			return super.getRequiredClasspath(delimiter);
		} else {
			StringBuffer buffer = new StringBuffer();
			for (Iterator<String> classpathIt = requiredClasspath.iterator(); classpathIt.hasNext();) {
				String classpathEntry = classpathIt.next();
				if (classpathEntry.startsWith("/")) {
					buffer.append(dir.getParentFile().getAbsolutePath() + classpathEntry);
				} else {
					buffer.append(dir.getAbsolutePath() + '/' + classpathEntry);
				}
				if (classpathIt.hasNext()) {
					buffer.append(delimiter);
				}
			}
			String superRequired = super.getRequiredClasspath(delimiter);
			if (superRequired.length() != 0) {
				buffer.append(delimiter);
				buffer.append(superRequired);
			}
			return buffer.toString();
		}
	}
}
