/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2020 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.tools.servlet;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.security.Principal;
import java.util.Enumeration;
import java.util.Locale;
import java.util.Map;
import javax.servlet.RequestDispatcher;
import javax.servlet.ServletInputStream;
import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpSession;
import com.openexchange.config.ConfigTools;
import com.openexchange.config.ConfigurationService;
import com.openexchange.server.services.ServerServiceRegistry;
import com.openexchange.tools.stream.CountingInputStream;

/**
 * {@link CountingHttpServletRequest} - The HTTP Servlet request wrapper aware of <code>"com.openexchange.servlet.maxBodySize"</code>
 * property.
 * 
 * @author <a href="mailto:thorben.betten@open-xchange.com">Thorben Betten</a>
 */
public final class CountingHttpServletRequest implements HttpServletRequest {

    private final HttpServletRequest servletRequest;

    private final long max;

    private volatile ServletInputStream servletInputStream;

    /**
     * Initializes a new {@link CountingHttpServletRequest}.
     */
    public CountingHttpServletRequest(final HttpServletRequest servletRequest) {
        this(servletRequest, ConfigTools.getLongProperty("com.openexchange.servlet.maxBodySize", 104857600L, ServerServiceRegistry.getInstance().getService(ConfigurationService.class)));
    }

    /**
     * Initializes a new {@link CountingHttpServletRequest}.
     */
    public CountingHttpServletRequest(final HttpServletRequest servletRequest, final long max) {
        super();
        this.max = max;
        this.servletRequest = servletRequest;
    }

    
    public Object getAttribute(final String name) {
        return servletRequest.getAttribute(name);
    }

    
    public String getAuthType() {
        return servletRequest.getAuthType();
    }

    
    public Cookie[] getCookies() {
        return servletRequest.getCookies();
    }

    
    public Enumeration<?> getAttributeNames() {
        return servletRequest.getAttributeNames();
    }

    
    public long getDateHeader(final String name) {
        return servletRequest.getDateHeader(name);
    }

    
    public String getCharacterEncoding() {
        return servletRequest.getCharacterEncoding();
    }

    
    public void setCharacterEncoding(final String env) throws UnsupportedEncodingException {
        servletRequest.setCharacterEncoding(env);
    }

    
    public String getHeader(final String name) {
        return servletRequest.getHeader(name);
    }

    
    public int getContentLength() {
        return servletRequest.getContentLength();
    }

    
    public String getContentType() {
        return servletRequest.getContentType();
    }

    
    public Enumeration<?> getHeaders(final String name) {
        return servletRequest.getHeaders(name);
    }

    
    public ServletInputStream getInputStream() throws IOException {
        if (max <= 0) {
            return servletRequest.getInputStream();
        }
        ServletInputStream tmp = servletInputStream;
        if (null == tmp) {
            synchronized (servletRequest) {
                tmp = servletInputStream;
                if (null == tmp) {
                    servletInputStream =
                        tmp = new DelegateServletInputStream(new CountingInputStream(servletRequest.getInputStream(), max));
                }
            }
        }
        return tmp;
    }

    
    public String getParameter(final String name) {
        return servletRequest.getParameter(name);
    }

    
    public Enumeration<?> getHeaderNames() {
        return servletRequest.getHeaderNames();
    }

    
    public int getIntHeader(final String name) {
        return servletRequest.getIntHeader(name);
    }

    
    public Enumeration<?> getParameterNames() {
        return servletRequest.getParameterNames();
    }

    
    public String[] getParameterValues(final String name) {
        return servletRequest.getParameterValues(name);
    }

    
    public String getMethod() {
        return servletRequest.getMethod();
    }

    
    public String getPathInfo() {
        return servletRequest.getPathInfo();
    }

    
    public Map<?, ?> getParameterMap() {
        return servletRequest.getParameterMap();
    }

    
    public String getProtocol() {
        return servletRequest.getProtocol();
    }

    
    public String getPathTranslated() {
        return servletRequest.getPathTranslated();
    }

    
    public String getScheme() {
        return servletRequest.getScheme();
    }

    
    public String getServerName() {
        return servletRequest.getServerName();
    }

    
    public String getContextPath() {
        return servletRequest.getContextPath();
    }

    
    public int getServerPort() {
        return servletRequest.getServerPort();
    }

    
    public String getQueryString() {
        return servletRequest.getQueryString();
    }

    
    public BufferedReader getReader() throws IOException {
        return servletRequest.getReader();
    }

    
    public String getRemoteUser() {
        return servletRequest.getRemoteUser();
    }

    
    public String getRemoteAddr() {
        return servletRequest.getRemoteAddr();
    }

    
    public boolean isUserInRole(final String role) {
        return servletRequest.isUserInRole(role);
    }

    
    public String getRemoteHost() {
        return servletRequest.getRemoteHost();
    }

    
    public Principal getUserPrincipal() {
        return servletRequest.getUserPrincipal();
    }

    
    public void setAttribute(final String name, final Object o) {
        servletRequest.setAttribute(name, o);
    }

    
    public String getRequestedSessionId() {
        return servletRequest.getRequestedSessionId();
    }

    
    public String getRequestURI() {
        return servletRequest.getRequestURI();
    }

    
    public void removeAttribute(final String name) {
        servletRequest.removeAttribute(name);
    }

    
    public Locale getLocale() {
        return servletRequest.getLocale();
    }

    
    public StringBuffer getRequestURL() {
        return servletRequest.getRequestURL();
    }

    
    public Enumeration<?> getLocales() {
        return servletRequest.getLocales();
    }

    
    public String getServletPath() {
        return servletRequest.getServletPath();
    }

    
    public boolean isSecure() {
        return servletRequest.isSecure();
    }

    
    public RequestDispatcher getRequestDispatcher(final String path) {
        return servletRequest.getRequestDispatcher(path);
    }

    
    public HttpSession getSession(final boolean create) {
        return servletRequest.getSession(create);
    }

    
    public String getRealPath(final String path) {
        return servletRequest.getRealPath(path);
    }

    
    public HttpSession getSession() {
        return servletRequest.getSession();
    }

    
    public int getRemotePort() {
        return servletRequest.getRemotePort();
    }

    
    public boolean isRequestedSessionIdValid() {
        return servletRequest.isRequestedSessionIdValid();
    }

    
    public String getLocalName() {
        return servletRequest.getLocalName();
    }

    
    public String getLocalAddr() {
        return servletRequest.getLocalAddr();
    }

    
    public boolean isRequestedSessionIdFromCookie() {
        return servletRequest.isRequestedSessionIdFromCookie();
    }

    
    public int getLocalPort() {
        return servletRequest.getLocalPort();
    }

    
    public boolean isRequestedSessionIdFromURL() {
        return servletRequest.isRequestedSessionIdFromURL();
    }

    
    public boolean isRequestedSessionIdFromUrl() {
        return servletRequest.isRequestedSessionIdFromUrl();
    }

}
