/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2012 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.tools.file.internal;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.HashSet;
import java.util.Set;
import java.util.SortedSet;
import java.util.TreeSet;
import java.util.UUID;
import javax.activation.MimetypesFileTypeMap;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import com.openexchange.java.util.UUIDs;
import com.openexchange.tools.file.external.FileStorage;
import com.openexchange.tools.file.external.FileStorageException;
import com.openexchange.tools.file.external.FileStorageException.Code;

/**
 * A {@link HashingFileStorage} generates UUIDs for every file that is stored in it.
 *
 * @author <a href="mailto:francisco.laguna@open-xchange.com">Francisco Laguna</a>
 */
public class HashingFileStorage implements FileStorage {

    private static final Log LOG = com.openexchange.log.Log.valueOf(LogFactory.getLog(HashingFileStorage.class));

    private File storage;

    public HashingFileStorage(File storage) {
        this.storage = storage;
    }

    protected File file(String identifier) {
        return new File(storage, identifier);
    }

    public boolean deleteFile(String identifier) throws FileStorageException {
        return file(identifier).delete();
    }

    public Set<String> deleteFiles(String[] identifiers) throws FileStorageException {
        Set<String> notDeleted = new HashSet<String>();
        for (String identifier : identifiers) {
            if (!deleteFile(identifier)) {
                notDeleted.add(identifier);
            }
        }

        return notDeleted;
    }

    public InputStream getFile(String name) throws FileStorageException {
        try {
            return new BufferedInputStream(new FileInputStream(file(name)));
        } catch (FileNotFoundException e) {
            throw new FileStorageException(Code.FILE_NOT_FOUND, name);
        }
    }

    public SortedSet<String> getFileList() throws FileStorageException {
        final SortedSet<String> files = new TreeSet<String>();
        final int beginIndex = storage.getAbsolutePath().length()+1;
        visit(new Visitor() {

            public void visit(File f) {
                if(f.isFile()) {
                    files.add(f.getAbsolutePath().substring(beginIndex));
                }
            }

        });
        return files;
    }

    public long getFileSize(String name) throws FileStorageException {
        return file(name).length();
    }

    public String getMimeType(String name) throws FileStorageException {
        final MimetypesFileTypeMap map = new MimetypesFileTypeMap();
        return map.getContentType(file(name));
    }

    public void recreateStateFile() throws FileStorageException {
        // Nope, no state file used.
    }

    public void remove() throws FileStorageException {
        visit(new Visitor() {

            public void visit(File f) {
                f.delete();
            }

        });
    }

    public String saveNewFile(InputStream file) throws FileStorageException {
        String[] filestorePath = generateName();
        File path = new File(storage, filestorePath[0]);
        if (!path.exists() && !path.mkdirs() && !path.exists()) {
            throw new FileStorageException(FileStorageException.Code.CREATE_DIR_FAILED, path.toString());
        }

        BufferedOutputStream bufOut = null;
        BufferedInputStream bufIn = null;
        File filePath = new File(path, filestorePath[1]);
        try {
            bufIn = new BufferedInputStream(file);
            bufOut = new BufferedOutputStream(new FileOutputStream(filePath));

            int i = 0;
            while((i = bufIn.read()) != -1) {
                bufOut.write(i);
            }
        } catch (FileNotFoundException e) {
            throw new FileStorageException(Code.FILE_NOT_FOUND, filePath.toString());
        } catch (IOException e) {
            throw new FileStorageException(Code.IOERROR, e.toString());
        } finally {
            if (bufIn != null) {
                try {
                    bufIn.close();
                } catch (IOException e) {
                    LOG.error(e.getMessage(), e);
                }
            }
            if (bufOut != null) {
                try {
                    bufOut.close();
                } catch (IOException e) {
                    LOG.error(e.getMessage(), e);
                }
            }
        }

        return filestorePath[0]+"/"+filestorePath[1];
    }

    public String[] generateName() {
        String uuid = UUIDs.getUnformattedString(UUID.randomUUID());

        String prefix = Integer.toHexString(uuid.hashCode());

        StringBuilder b = new StringBuilder();

        int i = 0;
        for (; i < prefix.length() && i < 6; i++) {
            b.append(prefix.charAt(i));
            if (i % 2 == 1 && i > 0) {
                b.append('/');
            }
        }
        for (; i < 6; i++) {
            b.append('0');
            if (i % 2 == 1 && i > 0) {
                b.append('/');
            }
        }

        b.setLength(b.length()-1);

        return new String[] { b.toString(), uuid };
    }

    public boolean stateFileIsCorrect() throws FileStorageException {
        return true; // We're not using a state file
    }

    // Visits all nodes - depth first
    protected void visit(Visitor visitor) {
        recurse(storage, visitor);
    }

    protected void recurse(File f, Visitor visitor) {
        if(f.isFile()) {
            visitor.visit(f);
            return;
        }
        File[] files = f.listFiles();
        if(files == null) {
            return;
        }
        for (File file : files) {
            recurse(file, visitor);
        }
        visitor.visit(f);
    }

    private static interface Visitor {
        public void visit(File f);
    }

}
