/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2012 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.groupware.update.tasks;

import static com.openexchange.tools.sql.DBUtils.closeSQLStuff;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.Collections;
import java.util.HashSet;
import java.util.Set;
import com.openexchange.api2.OXException;
import com.openexchange.database.DBPoolingException;
import com.openexchange.groupware.container.FolderObject;
import com.openexchange.groupware.contexts.impl.ContextImpl;
import com.openexchange.groupware.update.SimpleUpdateTask;
import com.openexchange.tools.oxfolder.OXFolderException;
import com.openexchange.tools.oxfolder.OXFolderException.FolderCode;
import com.openexchange.tools.oxfolder.OXFolderSQL;

/**
 * {@link CorrectPublicFolderPermissions} - Removes non-context-admin permissions from editable public folders.
 *
 * @author <a href="mailto:thorben.betten@open-xchange.com">Thorben Betten</a>
 */
public final class CorrectPublicFolderPermissions extends SimpleUpdateTask {

    public CorrectPublicFolderPermissions() {
        super();
    }

    @Override
    protected void perform(final Connection con) throws SQLException {
        for (final int contextId : getAllContexts(con)) {
            handleContext(contextId, con);
        }
    }

    private static final int[] CHANGEABLE_PUBLIC_FOLDERS =
    { FolderObject.SYSTEM_PUBLIC_FOLDER_ID, FolderObject.SYSTEM_INFOSTORE_FOLDER_ID, FolderObject.SYSTEM_PUBLIC_INFOSTORE_FOLDER_ID };

    private void handleContext(final int contextId, final Connection con) throws SQLException {
        PreparedStatement stmt = null;
        try {
            final int admin = getContextAdminID(contextId, con);
            stmt = con.prepareStatement("DELETE FROM oxfolder_permissions AS op WHERE op.cid = ? AND op.fuid = ? AND op-permission_id <> ? AND op-permission_id <> ?");
            for (final int fuid : CHANGEABLE_PUBLIC_FOLDERS) {
                stmt.setLong(1, contextId);
                stmt.setLong(2, fuid);
                stmt.setLong(3, admin);
                stmt.setLong(4, 0);
                stmt.addBatch();
            }
            stmt.executeBatch();
        } catch (final OXException e) {
            // Cannot occur
            final SQLException sqle = new SQLException(e.getMessage());
            sqle.initCause(e);
            throw sqle;
        } finally {
            closeSQLStuff(stmt);
        }
    }

    private Set<Integer> getAllContexts(final Connection con) throws SQLException {
        PreparedStatement stmt = null;
        ResultSet rs = null;
        try {
            stmt = con.prepareStatement("SELECT cid FROM user");
            rs = stmt.executeQuery();
            if (!rs.next()) {
                return Collections.emptySet();
            }
            final Set<Integer> set = new HashSet<Integer>();
            do {
                set.add(Integer.valueOf(rs.getInt(1)));
            } while (rs.next());
            return set;
        } finally {
            closeSQLStuff(rs, stmt);
        }
    }

    private int getContextAdminID(final int cid, final Connection con) throws OXException {
        try {
            final int retval = OXFolderSQL.getContextAdminID(new ContextImpl(cid), con);
            if (retval == -1) {
                throw new OXFolderException(FolderCode.NO_ADMIN_USER_FOUND_IN_CONTEXT, Integer.valueOf(cid));
            }
            return retval;
        } catch (final DBPoolingException e) {
            throw new OXFolderException(FolderCode.DBPOOLING_ERROR, e, Integer.valueOf(cid));
        } catch (final SQLException e) {
            throw new OXFolderException(FolderCode.SQL_ERROR, e, e.getMessage());
        }
    }
}
