/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2012 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.secret;

import static org.junit.Assert.assertEquals;
import org.junit.Test;
import com.openexchange.crypto.CryptoException;
import com.openexchange.crypto.CryptoService;
import com.openexchange.crypto.SimCryptoService;
import com.openexchange.groupware.AbstractOXException;
import com.openexchange.secret.impl.CryptoSecretEncryptionFactoryService;
import com.openexchange.session.Session;
import com.openexchange.session.SimSession;
import com.openexchange.sim.SimBuilder;


/**
 * {@link SecretEncryptionServiceTest}
 *
 * @author <a href="mailto:francisco.laguna@open-xchange.com">Francisco Laguna</a>
 */
public class SecretEncryptionServiceTest {
    
    @Test
    public void testEncrypt() throws CryptoException {
        final SimCryptoService crypto = new SimCryptoService("Encrypted", "Plain");
        final CryptoSecretEncryptionFactoryService factory = new CryptoSecretEncryptionFactoryService(crypto, null, "'The old secret'", "'The newer secret'", "'The newest secret'");
        final SecretEncryptionService encryptionService = factory.createService(null);
        
        final Session session = new SimSession();
        assertEquals("Encrypted", encryptionService.encrypt(session, "Plain"));
        assertEquals("The newest secret", crypto.getPassword());
        
    }
    
    @Test
    public void testDecryptNormally() throws AbstractOXException {
        final SimCryptoService crypto = new SimCryptoService("Encrypted", "Plain");
        final CryptoSecretEncryptionFactoryService factory = new CryptoSecretEncryptionFactoryService(crypto, null, "'The old secret'", "'The newer secret'", "'The newest secret'");
        final SecretEncryptionService encryptionService = factory.createService(null);
        
        assertEquals("Plain", encryptionService.decrypt(null, "Encrypted"));
        assertEquals("The newest secret", crypto.getPassword());
    }
    
    @Test
    public void testRecryptOlder() throws AbstractOXException {
        final SimBuilder cryptoSimBuilder = new SimBuilder()
                                    .expectCall("decrypt", "Encrypted", "The newest secret").andThrow(new CryptoException(new AbstractOXException()))
                                    .expectCall("decrypt", "Encrypted", "The newer secret").andThrow(new CryptoException(new AbstractOXException()))
                                    .expectCall("decrypt", "Encrypted", "The old secret").andReturn("Plain")
                                    .expectCall("encrypt", "Plain", "The newest secret").andReturn("Encrypted");
        final CryptoService crypto = cryptoSimBuilder.getSim(CryptoService.class);
        
        
        final SimBuilder strategySimBuilder = new SimBuilder()
                                        .expectCall("update", "Encrypted", "field 12");
        
        final SecretEncryptionStrategy<String> strategy = strategySimBuilder.getSim(SecretEncryptionStrategy.class);
        
        final CryptoSecretEncryptionFactoryService factory = new CryptoSecretEncryptionFactoryService(crypto, null, "'The old secret'", "'The newer secret'", "'The newest secret'");
        final SecretEncryptionService<String> encryptionService = factory.createService(strategy);
        
        assertEquals("Plain", encryptionService.decrypt(null, "Encrypted", "field 12"));

        cryptoSimBuilder.assertAllWereCalled();
        strategySimBuilder.assertAllWereCalled();
    }
    
    @Test
    public void testFallbackToSecretService() throws AbstractOXException {
        final Session session = new SimSession();
        
        final SimBuilder cryptoSimBuilder = new SimBuilder()
                                      .expectCall("decrypt", "Encrypted", "The newest secret").andThrow(new CryptoException(new AbstractOXException()))
                                      .expectCall("decrypt", "Encrypted", "The newer secret").andThrow(new CryptoException(new AbstractOXException()))
                                      .expectCall("decrypt", "Encrypted", "The old secret").andThrow(new CryptoException(new AbstractOXException()))
                                      .expectCall("decrypt", "Encrypted", "The secret from the secret service").andReturn("Plain")
                                      .expectCall("encrypt", "Plain", "The newest secret").andReturn("Encrypted");
        final CryptoService crypto = cryptoSimBuilder.getSim(CryptoService.class);


        final SimBuilder strategySimBuilder = new SimBuilder()
                                        .expectCall("update", "Encrypted", "field 12");

        final SimBuilder secretServiceBuilder = new SimBuilder()
                                        .expectCall("getSecret", session).andReturn("The secret from the secret service");
        
        final SecretService secretService = secretServiceBuilder.getSim(SecretService.class);
        
        final SecretEncryptionStrategy<String> strategy = strategySimBuilder.getSim(SecretEncryptionStrategy.class);

        final CryptoSecretEncryptionFactoryService factory = new CryptoSecretEncryptionFactoryService(crypto, secretService, "'The old secret'", "'The newer secret'", "'The newest secret'");
        final SecretEncryptionService<String> encryptionService = factory.createService(strategy);

        assertEquals("Plain", encryptionService.decrypt(session, "Encrypted", "field 12"));

        cryptoSimBuilder.assertAllWereCalled();
        strategySimBuilder.assertAllWereCalled();
    }
    
    // Error Cases
    @Test(expected = CryptoException.class)
    public void testNoWordFound() throws AbstractOXException {
        final Session session = new SimSession();
        
        final SimBuilder cryptoSimBuilder = new SimBuilder()
                                      .expectCall("decrypt", "Encrypted", "The newest secret").andThrow(new CryptoException(new AbstractOXException()))
                                      .expectCall("decrypt", "Encrypted", "The newer secret").andThrow(new CryptoException(new AbstractOXException()))
                                      .expectCall("decrypt", "Encrypted", "The old secret").andThrow(new CryptoException(new AbstractOXException()))
                                      .expectCall("decrypt", "Encrypted", "The secret from the secret service").andThrow(new CryptoException(new AbstractOXException()));
        final CryptoService crypto = cryptoSimBuilder.getSim(CryptoService.class);


        final SimBuilder secretServiceBuilder = new SimBuilder()
                                        .expectCall("getSecret", session).andReturn("The secret from the secret service");
        
        final SecretService secretService = secretServiceBuilder.getSim(SecretService.class);
        

        final CryptoSecretEncryptionFactoryService factory = new CryptoSecretEncryptionFactoryService(crypto, secretService, "The old secret", "The newer secret", "The newest secret");
        final SecretEncryptionService<String> encryptionService = factory.createService(null);

        assertEquals("Plain", encryptionService.decrypt(session, "Encrypted"));

        cryptoSimBuilder.assertAllWereCalled();
    }
    
    
}
