/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2012 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.imap;

import java.util.AbstractQueue;
import java.util.Collection;
import java.util.Iterator;
import java.util.NoSuchElementException;
import java.util.PriorityQueue;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.ReentrantLock;
import javax.mail.MessagingException;
import com.sun.mail.imap.IMAPStore;

/**
 * {@link UnboundedIMAPStoreContainer} - The unbounded {@link IMAPStoreContainer}.
 * 
 * @author <a href="mailto:thorben.betten@open-xchange.com">Thorben Betten</a>
 */
public class UnboundedIMAPStoreContainer extends AbstractIMAPStoreContainer {

    private final PriorityBlockingQueue queue;

    protected final String server;

    protected final int port;

    protected final String login;

    protected final String pw;

    /**
     * Initializes a new {@link UnboundedIMAPStoreContainer}.
     */
    public UnboundedIMAPStoreContainer(final String server, final int port, final String login, final String pw) {
        super();
        queue = new PriorityBlockingQueue();
        this.login = login;
        this.port = port;
        this.pw = pw;
        this.server = server;
    }

    /**
     * Initializes an empty {@link UnboundedIMAPStoreContainer}.
     */
    protected UnboundedIMAPStoreContainer() {
        super();
        queue = null;
        this.login = null;
        this.port = 0;
        this.pw = null;
        this.server = null;
    }

    /**
     * Gets the backing blocking queue.
     * 
     * @return The queue
     */
    protected BlockingQueue<IMAPStoreWrapper> getQueue() {
        return queue;
    }

    public IMAPStore getStore(final javax.mail.Session imapSession, final IMAPValidity validity) throws MessagingException, InterruptedException {
        /*
         * Retrieve and remove the head of this queue
         */
        final IMAPStoreWrapper imapStoreWrapper = queue.poll();
        IMAPStore imapStore = null == imapStoreWrapper ? null : imapStoreWrapper.imapStore;
        if (null == imapStore) {
            imapStore = newStore(server, port, login, pw, imapSession, validity);
            if (DEBUG) {
                LOG.debug("IMAPStoreContainer.getStore(): Returning newly established IMAPStore instance.");
            }
        } else if (DEBUG) {
            LOG.debug("IMAPStoreContainer.getStore(): Returning _cached_ IMAPStore instance.");
        }
        return imapStore;
    }

    public void backStore(final IMAPStore imapStore, final IMAPValidity validity) {
        final long currentValidity = validity.getCurrentValidity();
        if (currentValidity > 0 && imapStore.getValidity() < currentValidity) {
            validity.clearCachedConnections();
            closeSafe(imapStore);
        }
        backStoreNoValidityCheck(imapStore);
    }

    protected void backStoreNoValidityCheck(final IMAPStore imapStore) {
        if (!queue.offer(new IMAPStoreWrapper(imapStore))) {
            closeSafe(imapStore);
        } else if (DEBUG) {
            LOG.debug("IMAPStoreContainer.backStore(): Added IMAPStore instance to cache.");
        }
    }

    public void closeElapsed(final long stamp, final StringBuilder debugBuilder) {
        if (DEBUG) {
            LOG.debug("IMAPStoreContainer.closeElapsed(): " + queue.size() + " IMAPStore instances in queue for " + login + "@" + server + ":" + port);
        }
        IMAPStoreWrapper imapStoreWrapper;
        do {
            imapStoreWrapper = queue.pollIfElapsed(stamp);
            if (null == imapStoreWrapper) {
                break;
            }
            try {
                if (null == debugBuilder) {
                    closeSafe(imapStoreWrapper.imapStore);
                } else {
                    final String info = imapStoreWrapper.imapStore.toString();
                    closeSafe(imapStoreWrapper.imapStore);
                    debugBuilder.setLength(0);
                    LOG.debug(debugBuilder.append("IMAPStoreContainer.closeElapsed(): Closed elapsed IMAP store: ").append(info).toString());
                }
            } catch (final IllegalStateException e) {
                // Ignore
            }
        } while (true);
    }

    public void clear() {
        if (null == queue) {
            return;
        }
        IMAPStoreWrapper wrapper;
        while ((wrapper = queue.poll()) != null) {
            closeSafe(wrapper.imapStore);
        }
    }

    private static class PriorityBlockingQueue extends AbstractQueue<IMAPStoreWrapper> implements BlockingQueue<IMAPStoreWrapper>, java.io.Serializable {

        private static final long serialVersionUID = 1337510919245408276L;

        final transient PriorityQueue<IMAPStoreWrapper> q;

        final ReentrantLock lock;

        private final Condition notEmpty;

        /**
         * Creates a <tt>PriorityBlockingQueue</tt> with the default initial capacity (11) that orders its elements according to their
         * {@linkplain Comparable natural ordering}.
         */
        public PriorityBlockingQueue() {
            super();
            q = new PriorityQueue<IMAPStoreWrapper>();
            lock = new ReentrantLock(true);
            notEmpty = lock.newCondition();
        }

        @Override
        public boolean add(final IMAPStoreWrapper e) {
            return offer(e);
        }

        public boolean offer(final IMAPStoreWrapper e) {
            final ReentrantLock lock = this.lock;
            lock.lock();
            try {
                final boolean ok = q.offer(e);
                assert ok;
                notEmpty.signal();
                return true;
            } finally {
                lock.unlock();
            }
        }

        public void put(final IMAPStoreWrapper e) {
            offer(e);
        }

        public boolean offer(final IMAPStoreWrapper e, final long timeout, final TimeUnit unit) {
            return offer(e);
        }

        /**
         * Retrieves and removes the head of this queue if elapsed compared to given time stamp, or returns <code>null</code> if this queue
         * is empty or head is not elapsed.
         * 
         * @param stamp The time stamp
         * @return The elapsed head of this queue
         */
        public IMAPStoreWrapper pollIfElapsed(final long stamp) {
            final ReentrantLock lock = this.lock;
            lock.lock();
            try {
                final IMAPStoreWrapper e = q.peek();
                if ((null != e) && (e.lastAccessed < stamp)) {
                    return q.poll();
                }
                return null;
            } finally {
                lock.unlock();
            }
        }

        public IMAPStoreWrapper poll() {
            final ReentrantLock lock = this.lock;
            lock.lock();
            try {
                return q.poll();
            } finally {
                lock.unlock();
            }
        }

        public IMAPStoreWrapper take() throws InterruptedException {
            final ReentrantLock lock = this.lock;
            lock.lockInterruptibly();
            try {
                try {
                    while (q.size() == 0) {
                        notEmpty.await();
                    }
                } catch (final InterruptedException ie) {
                    notEmpty.signal(); // propagate to non-interrupted thread
                    throw ie;
                }
                final IMAPStoreWrapper x = q.poll();
                assert x != null;
                return x;
            } finally {
                lock.unlock();
            }
        }

        public IMAPStoreWrapper poll(final long timeout, final TimeUnit unit) throws InterruptedException {
            long nanos = unit.toNanos(timeout);
            final ReentrantLock lock = this.lock;
            lock.lockInterruptibly();
            try {
                for (;;) {
                    final IMAPStoreWrapper x = q.poll();
                    if (x != null) {
                        return x;
                    }
                    if (nanos <= 0) {
                        return null;
                    }
                    try {
                        nanos = notEmpty.awaitNanos(nanos);
                    } catch (final InterruptedException ie) {
                        notEmpty.signal(); // propagate to non-interrupted thread
                        throw ie;
                    }
                }
            } finally {
                lock.unlock();
            }
        }

        public IMAPStoreWrapper peek() {
            final ReentrantLock lock = this.lock;
            lock.lock();
            try {
                return q.peek();
            } finally {
                lock.unlock();
            }
        }

        @Override
        public int size() {
            final ReentrantLock lock = this.lock;
            lock.lock();
            try {
                return q.size();
            } finally {
                lock.unlock();
            }
        }

        public int remainingCapacity() {
            return Integer.MAX_VALUE;
        }

        @Override
        public boolean remove(final Object o) {
            final ReentrantLock lock = this.lock;
            lock.lock();
            try {
                return q.remove(o);
            } finally {
                lock.unlock();
            }
        }

        @Override
        public boolean contains(final Object o) {
            final ReentrantLock lock = this.lock;
            lock.lock();
            try {
                return q.contains(o);
            } finally {
                lock.unlock();
            }
        }

        @Override
        public Object[] toArray() {
            final ReentrantLock lock = this.lock;
            lock.lock();
            try {
                return q.toArray();
            } finally {
                lock.unlock();
            }
        }

        @Override
        public String toString() {
            final ReentrantLock lock = this.lock;
            lock.lock();
            try {
                return q.toString();
            } finally {
                lock.unlock();
            }
        }

        public int drainTo(final Collection<? super IMAPStoreWrapper> c) {
            if (c == null) {
                throw new NullPointerException();
            }
            if (c == this) {
                throw new IllegalArgumentException();
            }
            final ReentrantLock lock = this.lock;
            lock.lock();
            try {
                int n = 0;
                IMAPStoreWrapper e;
                while ((e = q.poll()) != null) {
                    c.add(e);
                    ++n;
                }
                return n;
            } finally {
                lock.unlock();
            }
        }

        public int drainTo(final Collection<? super IMAPStoreWrapper> c, final int maxElements) {
            if (c == null) {
                throw new NullPointerException();
            }
            if (c == this) {
                throw new IllegalArgumentException();
            }
            if (maxElements <= 0) {
                return 0;
            }
            final ReentrantLock lock = this.lock;
            lock.lock();
            try {
                int n = 0;
                IMAPStoreWrapper e;
                while (n < maxElements && (e = q.poll()) != null) {
                    c.add(e);
                    ++n;
                }
                return n;
            } finally {
                lock.unlock();
            }
        }

        @Override
        public void clear() {
            final ReentrantLock lock = this.lock;
            lock.lock();
            try {
                q.clear();
            } finally {
                lock.unlock();
            }
        }

        @Override
        public <T> T[] toArray(final T[] a) {
            final ReentrantLock lock = this.lock;
            lock.lock();
            try {
                return q.toArray(a);
            } finally {
                lock.unlock();
            }
        }

        @Override
        public Iterator<IMAPStoreWrapper> iterator() {
            return new Itr(toArray());
        }

        /**
         * Snapshot iterator that works off copy of underlying q array.
         */
        private class Itr implements Iterator<IMAPStoreWrapper> {

            final Object[] array; // Array of all elements

            int cursor; // index of next element to return;

            int lastRet; // index of last element, or -1 if no such

            Itr(final Object[] array) {
                lastRet = -1;
                this.array = array;
            }

            public boolean hasNext() {
                return cursor < array.length;
            }

            @SuppressWarnings("unchecked")
            public IMAPStoreWrapper next() {
                if (cursor >= array.length) {
                    throw new NoSuchElementException();
                }
                lastRet = cursor;
                return (IMAPStoreWrapper) array[cursor++];
            }

            public void remove() {
                if (lastRet < 0) {
                    throw new IllegalStateException();
                }
                final Object x = array[lastRet];
                lastRet = -1;
                // Traverse underlying queue to find == element,
                // not just a .equals element.
                lock.lock();
                try {
                    for (final Iterator<IMAPStoreWrapper> it = q.iterator(); it.hasNext();) {
                        if (it.next() == x) {
                            it.remove();
                            return;
                        }
                    }
                } finally {
                    lock.unlock();
                }
            }
        }

        private void writeObject(final java.io.ObjectOutputStream s) throws java.io.IOException {
            lock.lock();
            try {
                s.defaultWriteObject();
            } finally {
                lock.unlock();
            }
        }

    } // End of class

}
