/**
 *
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Matthias Biggeleben <matthias.biggeleben@open-xchange.com>
 *
 */

/*jslint bitwise: false, nomen: false, onevar: false, plusplus: false, regexp: false, white: true, browser: true, devel: true, evil: true, forin: true, undef: true, eqeqeq: true, immed: true */

/*global ox, jQuery, _, register, oxProductInfo, formatError, formatDate, triggerEvent */

ox.UINotifier = (function ($) {

    // vars
    var container;
    var isVisible = {
        ok: true,
        log: true,
        info: true,
        note: true,
        warn: true,
        error: true
    };

    /*
     * History
     */

    // history object
    var history = [], isNested = ox.api.window.isNested, core;

    // nested?
    if (isNested) {
        core = ox.api.window.core;
        // replace with objects from core window
        history = core.ox.UINotifier.history;
        isVisible = core.ox.UINotifier.isVisible;
    }

    /*
     * Init
     */
    var init = $.noop;
    init = function () {
        // get offset
        var o = $("#notheader").offset() || { top: 0, left: 0 };
        // create container
        container = $("<div/>", { id: "notificationContainer" }).css({
            position: "absolute",
            top: (o.top + 10) + "px",
            right: "10px",
            width: "450px",
            height: "auto",
            zIndex: 65300 // panel hover is 9900/9901
        }).appendTo(document.body);
        // remove init
        init = $.noop;
    };

    /*
     * Report management
     */
    var addLine, updateReport, report = null, lines = null;

    addLine = function (str) {
        return $("<div/>")
            .css({
                fontFamily: "monospace",
                fontSize: "12px",
                lineHeight: "15px",
                paddingLeft: "2em",
                textIndent: "-2em"
            })
            .text(String(str))
            .appendTo(lines);
    };

    updateReport = function () {

        if (lines !== null) {

            // clear
            lines.empty();

            // loop over history
            var i = 0, $i = history.length, obj, message, count = 0;
            for (; i < $i; i++) {
                obj = history[i];
                if (obj.type === "error" || obj.type === "warn") {
                    message = obj.data !== undefined ?
                        formatError(obj.data, "%2$s (%1$s, %3$s)") : obj.text;
                    addLine(
                        (i + 1) + ". " + formatDate(obj.time, "datetime") + " GMT"
                    )
                        .css({
                            color: "#555"
                        });
                    addLine(message)
                        .css({
                            fontFamily: "Arial, Helvetica, sans-serif",
                            fontWeight: "bold",
                            fontSize: "14px",
                            color: obj.type === "error" ? "#800" : "#c80",
                            margin: "0 0 1em 0"
                        });
                    count++;
                }
            }

            // no errors or warnings?
            if (count === 0) {
                addLine(_("No error occured"))
                    .css({
                        fontFamily: "Arial, Helvetica, sans-serif",
                        fontWeight: "bold",
                        fontSize: "14px",
                        color: "#080",
                        margin: "0 0 1em 0"
                    });
            }

            addLine("\u00a0");
            addLine("Date: " + new Date());
            addLine("Host: " + window.location.href);
            addLine("Version: " + oxProductInfo.pversion + " (UI), " + ox.api.config.get("serverVersion") + " (Server)");
            addLine("Browser: " + navigator.userAgent);
            addLine("\u00a0");
            var folders = "";
            jQuery.each(ox.api.config.get("folder", false), function(module, folder_id) {
                folders += module + "=" + folder_id;
            });
            addLine("Module: " + ox.UIController.getModule() +
                ", View: " + ox.UIController.getView() +
                ", Folder: " + ox.UIController.getFolder() +
                ", Default folders: " + folders +
                ", Landing page: " + ox.api.config.get("gui.global.landing_page.module", "") +
                ", Tree: " + (ox.api.config.get("modules.folder.tree") === 1 ? "New" : "Classic") +
                ", Expert: " + (ox.api.config.get("gui.global.expert_tree") === 1 ? "Yes" : "No") +
                ", Language: " + ox.api.config.get("language")
                );
            addLine("\u00a0");
            addLine("Loaded modules: " +
                ox.util.keys(ox.api.config.get("modules"), false).sort().join(", ")
                );
        }
    };

    /*
     * Template
     */
    var template = $("<div/>").css({
        position: "relative",
        width: "450px",
        height: "60px",
        marginBottom: "10px"
    }).append(
        $("<div/>").css({
            position: "absolute",
            top: "0px",
            right: "0px",
            bottom: "0px",
            left: "0px",
            backgroundColor: "black",
            opacity: 0.85,
            borderRadius: "20px",
            MozBorderRadius: "20px",
            webkitBorderRadius: "20px",
            boxShadow: "5px 5px 15px #111",
            MozBoxShadow: "5px 5px 15px #111",
            webkitBoxShadow: "5px 5px 15px #111"
        })
    ).append(
        $("<div/>").css({
            position: "absolute",
            top: "0px",
            right: "0px",
            bottom: "0px",
            left: "0px",
            backgroundColor: "transparent",
            border: "3px solid #ccc",
            padding: "10px 15px 10px 15px",
            textAlign: "center",
            fontSize: "9pt",
            fontWeight: "bold",
            lineHeight: "1.5em",
            color: "white",
            overflow: "hidden",
            MozBorderRadius: "20px",
            webkitBorderRadius: "20px",
            borderRadius: "20px 20px 20px 20px",
            cursor: "default"
        }).append(
            $("<span/>")
        ).append(
            $("<br/>")
        ).append(
            $("<span/>").css({ color: "white" })
        ).append(
            $("<span/>").text("\u00a0 ")
        ).append(
            $("<a/>").css({
                color: "#ccc",
                fontWeight: "normal",
                textDecoration: "underline",
                whiteSpace: "nowrap",
                cursor: "pointer"
            })
        )
    );

    var styles = {
        "log": { color: "#ccc", borderColor: "#ccc #aaa #aaa #ccc" },
        "ok": { color: "#688f42", borderColor: "#82b252 #688f42 #46602d #688f42" },
        "note": { color: "#ff8700", borderColor: "#ffa400 #ff8700 #ff8700 #ffa400" },
        "warn": { color: "#ff8700", borderColor: "#ffa400 #ff8700 #ff8700 #ffa400" },
        "error": { color: "#c00", borderColor: "#c00 #800 #800 #a00" }
    };

    var labels = {
        "log": _("Info"), /*i18n*/
        "ok": _("Info"), /*i18n*/
        "note": _("Info"), /*i18n*/
        "warn": _("Warning"), /*i18n*/
        "error": _("Error") /*i18n*/
    };

    var lastText = '';

    var out = function (type, text, data, customLabel) {
        if (isVisible[type]) {
            // always init
            init();
            // vars
            var node, timer, fader,
                label = customLabel !== undefined ? String(customLabel) : labels[type] + ": ";
            // don't show the very same message again
            if (container.children().length > 0 && (label + ' ' + text) === lastText) {
                return;
            }
            // don't show more than 3 errors at once
            if ((type === 'error' || type === 'warn') && container.find('.notification-error, .notification-warn').length >= 3) {
                container.children().first().remove();
            }
            // clone template
            node = template.clone(false);
            // customize
            node.addClass('notification-' + type)
                .children().eq(1)
                .css(styles[type])
                .children().eq(0)
                .text(label)
                .end().eq(2)
                .text(String(text));
            // remember
            lastText = label + ' ' + text;
            // error/warn
            if (type === "error" || type === "warn") {
                node.children().eq(1).children().eq(4)
                    .text(String(_("Open error report")))
                    .bind("click", function () {
                        // show report in core window
                        ox.api.window.select("OX.0");
                        ox.api.window.core.ox.UINotifier.showReport();
                    });
            } else {
                node.children().eq(1).children().eq(4).remove();
            }
            // add
            container.append(node);
            // adjust height
            var height = Math.max(60, node.children().get(1).scrollHeight + 10);
            node.css("height", height + "px");
            // fader
            fader = function () {
                if (node) {
                    // clear
                    $(document).add(node).unbind("mousedown keydown", fader);
                    clearTimeout(timer);
                    // fade out
                    node.fadeOut(500, function () {
                        // remove node
                        if (node !== null) {
                            node.remove();
                            node = null;
                        }
                    });
                }
            };

            // add to history
            if (type !== "ok") {
                history.push({
                    type: type,
                    text: labels[type] + ": " + text,
                    time: ox.util.now(),
                    data: data !== null ? data : { code: "UI-0000", error_id: "0", error: String(text) } // all but null works
                });
                // update report
                updateReport();
            }

            // fade out by click & key
            $(document).add(node).bind("mousedown keypress", fader);

            // time to live
            if (type === "ok" || type === "log") {
                timer = setTimeout(fader, 7000); // 7 seconds
            }
        }
    };

    // public methods/properties
    var notifier = {

        log: function (text, data, label) {
            out("log", text, data, label);
        },

        info: function (text, data, label) {
            out("log", text, data, label); // alias for log
        },

        error: function (text, data, label) {
            out("error", text, data, label);
        },

        ok: function (text, data, label) {
            out("ok", text, data, label);
        },

        warn: function (text, data, label) {
            out("warn", text, data, label);
        },

        note: function (text, data, label) {
            out("note", text, data, label);
        },

        history: history,

        isVisible: isVisible,

        getVisible: function () {
            return isVisible;
        },

        setVisible: function (type, state) {
            if (isVisible.hasOwnProperty(type)) {
                isVisible[type] = state;
            } else {
                return false;
            }
        },

        updateReport: updateReport,

        showReport: function () {

            var fnSend = function () {
                // vars
                var sendHTML = !ox.api.config.get("gui.mail.formatmessage").match(/text\/plain/i),
                    mailtext = "",
                    tmp = [];
                // html or text mails?
                if (sendHTML) {
                    mailtext = lines.html();
                } else {
                    // get content
                    lines.children().each(function (i) {
                        tmp.push($(this).text());
                    });
                    mailtext = tmp.join("<br/>");
                }
                // send mail
                ox.api.mail.compose({
                    data: {
                        to: ox.api.config.get("ui.global.notifier.report.mailAddress", ""),
                        subject: "Automatic bug report for " + oxProductInfo.pversion,
                        mailtext: mailtext
                    }
                });
                // clean up
                mailtext = tmp = null;
            };

            var fnClose = function () {
                report.popup("disable");
                report = lines = null;
            };

            if (report === null) {

                report = $("<div/>", { id: "error-report" })
                    .css({
                        color: "black"
                    })
                    .append(
                        $("<div/>")
                            .css({ fontSize: "16pt", color: "#888", margin: "0.5em 0 1em 0" })
                            .text(String(_("History of error messages")))
                    )
                    .append(
                        lines = $("<div/>")
                            .css({
                                paddingRight: "1em",
                                height: "300px",
                                overflow: "auto"
                            })
                    )
                    .append(
                        $("<div/>")
                            .css({
                                height: "30px",
                                margin: "10px 0 0 0"
                            })
                            .append(
                                // close button
                                $.button({
                                    title: _("Close"),
                                    click: fnClose
                                }).css({
                                    marginLeft: "1em",
                                    "float": "right"
                                })
                            )
                            .append(function() {
                                // feature disabled so hide button
                                if (ox.api.config.get("ui.global.notifier.report.sendEnabled", true) === false) {
                                    return "";
                                }

                                // send mail button
                                return $.button({
                                    title: _("Send as E-Mail"),
                                    click: function () {
                                        fnSend();
                                        fnClose();
                                    }
                                }).css("float", "right");
                            })
                    );
            }

            updateReport();

            report.popup({
                center: true,
                modal: true,
                width: "700px",
                height: "380px",
                overflow: "hidden"
            });
        }
    };

    /*
     * Register events
     */

    var eventHandler = function (type) {
        return function (priority, message, data) {
            // trigger global event?
            if (type === "error") {
                var trigger = isNested ? core.triggerEvent : triggerEvent;
                trigger("OX_Global_Error", priority, message || "", data);
            }
            // do not show notifier for specific errors
            if (data && data.code && data.code.match(/^SES-02..$/)) {
                return;
            }
            // show notification popup
            notifier[type](message || "", data);
        };
    };

    register("OX_New_Error", eventHandler("error"));
    register("OX_New_Info", eventHandler("ok"));

    // publish
    return notifier;

}(jQuery));
