/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2012 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.server.osgiservice;

import java.util.Dictionary;
import java.util.LinkedList;
import java.util.List;
import org.osgi.framework.BundleContext;
import org.osgi.framework.Filter;
import org.osgi.framework.ServiceReference;
import org.osgi.framework.ServiceRegistration;
import org.osgi.util.tracker.ServiceTracker;
import org.osgi.util.tracker.ServiceTrackerCustomizer;
import com.openexchange.server.ServiceLookup;


/**
 * A {@link HousekeepingActivator} helps with housekeeping tasks like remembering service trackers or service registrations
 * and cleaning them up later.
 *
 * @author <a href="mailto:francisco.laguna@open-xchange.com">Francisco Laguna</a>
 */
public abstract class HousekeepingActivator extends DeferredActivator {

    /**
     * Puts/removes tracked service from activator's service look-up as the appear/disappear.
     */
    private static final class ServiceLookupTrackerCustomizer<S> implements ServiceTrackerCustomizer {

        private final Class<S> clazz;
        private final HousekeepingActivator activator;
        private final BundleContext context;

        /**
         * Initializes a new {@link ServiceTrackerCustomizerImplementation}.
         *
         * @param clazz The service's class to look-up
         * @param activator The activator
         * @param context The bundle context
         */
        protected ServiceLookupTrackerCustomizer(final Class<S> clazz, final HousekeepingActivator activator, final BundleContext context) {
            super();
            this.clazz = clazz;
            this.activator = activator;
            this.context = context;
        }

        public Object addingService(final ServiceReference reference) {
            final Object service = context.getService(reference);
            activator.addService(clazz, service);
            return service;
        }

        public void modifiedService(final ServiceReference reference, final Object service) {
            // Ignore
        }

        public void removedService(final ServiceReference reference, final Object service) {
            activator.removeService(clazz);
            context.ungetService(reference);
        }
    }

    private final List<ServiceTracker> serviceTrackers = new LinkedList<ServiceTracker>();
    private final List<ServiceRegistration> serviceRegistrations = new LinkedList<ServiceRegistration>();

    @Override
    protected void handleAvailability(final Class<?> clazz) {
        // Override if needed
    }

    @Override
    protected void handleUnavailability(final Class<?> clazz) {
        // Override if needed
    }

    @Override
    public void start(BundleContext context) throws Exception {
        super.start(context);
        /*
         * Invoking ServiceTracker.open() more than once is a no-op, therefore it can be safely called from here.
         */
        if (!serviceTrackers.isEmpty()) {
            openTrackers();
        }
    }

    @Override
    protected void stopBundle() throws Exception {
        cleanUp();
    }

    protected <T> void registerService(final Class<T> klass, final T service, final Dictionary properties) {
        serviceRegistrations.add(context.registerService(klass.getName(), service, properties));
    }

    protected <T> void registerService(final Class<T> klass, final T service) {
        registerService(klass, service, null);
    }

    protected void rememberTracker(final ServiceTracker tracker) {
        serviceTrackers.add(tracker);
    }

    protected void forgetTracker(final ServiceTracker tracker) {
        serviceTrackers.remove(tracker);
    }

    protected ServiceTracker track(final Class<?> klass, final ServiceTrackerCustomizer customizer) {
        final ServiceTracker tracker = new ServiceTracker(context, klass.getName(), customizer);
        rememberTracker(tracker);
        return tracker;
    }

    protected ServiceTracker track(final Filter filter, final ServiceTrackerCustomizer customizer) {
        final ServiceTracker tracker = new ServiceTracker(context, filter, customizer);
        rememberTracker(tracker);
        return tracker;
    }

    protected ServiceTracker track(final Class<?> klass) {
        return track(klass, (ServiceTrackerCustomizer) null);
    }

    protected ServiceTracker track(final Filter filter) {
        return track(filter, (ServiceTrackerCustomizer) null);
    }

    protected <T> ServiceTracker track(final Class<?> klass, final SimpleRegistryListener<T> listener) {
        return track(klass, new ServiceTrackerCustomizer() {

            public Object addingService(final ServiceReference arg0) {
                final Object service = context.getService(arg0);
                listener.added(arg0, (T) service);
                return service;
            }

            public void modifiedService(final ServiceReference arg0, final Object arg1) {
                // Don't care

            }

            public void removedService(final ServiceReference arg0, final Object arg1) {
                listener.removed(arg0, (T) arg1);
                context.ungetService(arg0);
            }

        });
    }

    protected <T> ServiceTracker track(final Filter filter, final SimpleRegistryListener<T> listener) {
        return track(filter, new ServiceTrackerCustomizer() {

            public Object addingService(final ServiceReference arg0) {
                final Object service = context.getService(arg0);
                listener.added(arg0, (T) service);
                return service;
            }

            public void modifiedService(final ServiceReference arg0, final Object arg1) {
                // TODO Auto-generated method stub

            }

            public void removedService(final ServiceReference arg0, final Object arg1) {
                listener.removed(arg0, (T) arg1);
                context.ungetService(arg0);
            }

        });
    }

    /**
     * Creates and remembers a new {@link ServiceTracker}. The tracked service is automatically {@link #addService(Class, Object) added to}/
     * {@link #removeService(Class) removed} from tracked services and thus available/disappearing when using this activator as
     * {@link ServiceLookup service look-up}.
     * <p>
     * <b>NOTE</b>: Don't forget to open tracker(s) with {@link #openTrackers()}.
     *
     * @param clazz The class of the tracked service
     * @return The newly created {@link ServiceTracker} instance
     */
    protected <S> ServiceTracker trackService(final Class<S> clazz) {
        final ServiceTracker tracker = new ServiceTracker(context, clazz.getName(), new ServiceLookupTrackerCustomizer<S>(clazz, this, context));
        rememberTracker(tracker);
        return tracker;
    }

    protected void openTrackers() {
        for (final ServiceTracker tracker : new LinkedList<ServiceTracker>(serviceTrackers)) {
            tracker.open();
        }
    }

    protected void closeTrackers() {
        for (final ServiceTracker tracker : new LinkedList<ServiceTracker>(serviceTrackers)) {
            tracker.close();
        }
    }

    protected void clearTrackers() {
        serviceTrackers.clear();
    }

    protected void unregisterServices() {
        for (final ServiceRegistration reg : serviceRegistrations) {
            reg.unregister();
        }
        serviceRegistrations.clear();
    }

    protected void cleanUp() {
        closeTrackers();
        clearTrackers();
        unregisterServices();
    }

}
