/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2012 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.file.storage.json.servlets;

import java.io.Closeable;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.security.NoSuchAlgorithmException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import javax.servlet.ServletOutputStream;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import org.metastatic.rsync.ChecksumPair;
import org.metastatic.rsync.Rdiff;
import com.openexchange.ajax.MultipleAdapterServletNew;
import com.openexchange.ajax.helper.DownloadUtility;
import com.openexchange.ajax.requesthandler.AJAXRequestData;
import com.openexchange.file.storage.DefaultFile;
import com.openexchange.file.storage.File;
import com.openexchange.file.storage.FileStorageFileAccess;
import com.openexchange.file.storage.composition.IDBasedFileAccess;
import com.openexchange.file.storage.json.actions.files.AJAXInfostoreRequest;
import com.openexchange.file.storage.json.actions.files.AbstractFileAction;
import com.openexchange.file.storage.json.actions.files.FileActionFactory;
import com.openexchange.groupware.AbstractOXException;
import com.openexchange.tools.servlet.AjaxException;
import com.openexchange.tools.servlet.http.Tools;
import com.openexchange.tools.session.ServerSession;

/**
 * {@link FileServlet}
 *
 * @author <a href="mailto:francisco.laguna@open-xchange.com">Francisco Laguna</a>
 */
public class FileServlet extends MultipleAdapterServletNew {

    private static final long serialVersionUID = 6959876535438535648L;

    private static final String PARAMETER_CONTENT_DISPOSITION = "content_disposition";

    private static interface Handler {
        void handle(AJAXRequestData data, HttpServletRequest req, HttpServletResponse res, ServerSession session) throws IOException, AbstractOXException;
    }

    private static final Map<String, Handler> HANDLERS;

    static {
        final Map<String, Handler> m = new HashMap<String, Handler>(4);
        m.put("document", new Handler() {
            
            public void handle(final AJAXRequestData data, final HttpServletRequest req, final HttpServletResponse res, final ServerSession session) throws IOException, AbstractOXException {
                final AJAXInfostoreRequest request = new AJAXInfostoreRequest(data, session);
                request.require(AbstractFileAction.Param.ID);

                final String contentType = req.getParameter(PARAMETER_CONTENT_TYPE);
                final String userAgent = req.getHeader("user-agent");
                final String contentDisposition = req.getParameter(PARAMETER_CONTENT_DISPOSITION);

                final IDBasedFileAccess fileAccess = request.getFileAccess();
                final File fileMetadata = fileAccess.getFileMetadata(request.getId(), request.getVersion());

                InputStream documentData = null;
                ServletOutputStream outputStream = null;
                try {
                    documentData = fileAccess.getDocument(request.getId(), request.getVersion());
                    outputStream = res.getOutputStream();
                    if (SAVE_AS_TYPE.equals(contentType)) {
                        if (null == contentDisposition) {
                            final StringBuilder sb = new StringBuilder(32).append("attachment");
                            DownloadUtility.appendFilenameParameter(fileMetadata.getFileName(), SAVE_AS_TYPE, userAgent, sb);
                            res.setHeader("Content-Disposition", sb.toString());
                        } else {
                            final StringBuilder sb = new StringBuilder(32).append(contentDisposition.trim());
                            DownloadUtility.appendFilenameParameter(fileMetadata.getFileName(), SAVE_AS_TYPE, userAgent, sb);
                            res.setHeader("Content-Disposition", sb.toString());
                            // Tools.setHeaderForFileDownload(userAgent, res, fileMetadata.getFileName(), contentDisposition);
                        }
                        res.setContentType(contentType);
                    } else {
                        final com.openexchange.ajax.helper.DownloadUtility.CheckedDownload checkedDownload =
                            DownloadUtility.checkInlineDownload(
                                documentData,
                                fileMetadata.getFileName(),
                                fileMetadata.getFileMIMEType(),
                                contentDisposition,
                                userAgent);
                        if (contentDisposition != null) {
                            if (contentDisposition.indexOf(';') < 0) {
                                final String disposition = checkedDownload.getContentDisposition();
                                final int pos = disposition.indexOf(';');
                                if (pos >= 0) {
                                    res.setHeader("Content-Disposition", contentDisposition + disposition.substring(pos));
                                } else {
                                    res.setHeader("Content-Disposition", contentDisposition);
                                }
                            } else {
                                res.setHeader("Content-Disposition", contentDisposition);
                            }
                        } else {
                            res.setHeader("Content-Disposition", checkedDownload.getContentDisposition());
                        }
                        if (contentType != null) {
                            res.setContentType(contentType);
                        } else {
                            res.setContentType(checkedDownload.getContentType());
                        }
                        documentData = checkedDownload.getInputStream();
                    }
                    // Browsers doesn't like the Pragma header the way we usually set
                    // this. Especially if files are sent to the browser. So removing
                    // pragma header
                    Tools.removeCachingHeader(res);

                    final int buflen = 2048;
                    final byte[] buf = new byte[buflen];
                    for (int read; (read = documentData.read(buf, 0, buflen)) > 0;) {
                        outputStream.write(buf, 0, read);
                    }
                    outputStream.flush();

                } finally {
                    AbstractFileAction.close(documentData);
                    AbstractFileAction.close(outputStream);
                    fileAccess.finish();
                }
            }
        });
        m.put("documentdelta", new Handler() {
            
            public void handle(final AJAXRequestData data, final HttpServletRequest req, final HttpServletResponse res, final ServerSession session) throws IOException, AbstractOXException {
                final AJAXInfostoreRequest request = new AJAXInfostoreRequest(data, session);
                request.require(AbstractFileAction.Param.ID);

                final IDBasedFileAccess fileAccess = request.getFileAccess();

                InputStream documentStream = null;
                InputStream requestStream = null;
                OutputStream deltaOut = null;
                try {
                    documentStream = fileAccess.getDocument(request.getId(), request.getVersion());
                    requestStream = data.getUploadStream();

                    final Rdiff rdiff = new Rdiff();
                    // Read in signature
                    final List<ChecksumPair> signatures = rdiff.readSignatures(requestStream);
                    // Create delta against document and write it directly to HTTP output stream
                    deltaOut = res.getOutputStream();
                    rdiff.makeDeltas(signatures, documentStream, deltaOut);
                    deltaOut.flush();
                } catch (final NoSuchAlgorithmException e) {
                    // Cannot occur
                    throw new AjaxException(AjaxException.Code.UnexpectedError, e, e.getMessage());
                } finally {
                    AbstractFileAction.close(documentStream);
                    AbstractFileAction.close(requestStream);
                    AbstractFileAction.close(deltaOut);
                }
            }
        });
        m.put("documentsig", new Handler() {
            
            public void handle(final AJAXRequestData data, final HttpServletRequest req, final HttpServletResponse res, final ServerSession session) throws IOException, AbstractOXException {
                final AJAXInfostoreRequest request = new AJAXInfostoreRequest(data, session);
                request.require(AbstractFileAction.Param.ID);

                final IDBasedFileAccess fileAccess = request.getFileAccess();

                InputStream documentStream = null;
                OutputStream sigOut = null;
                try {
                    documentStream = fileAccess.getDocument(request.getId(), request.getVersion());

                    final Rdiff rdiff = new Rdiff();
                    // Make signature
                    sigOut = res.getOutputStream();
                    rdiff.makeSignatures(documentStream, sigOut);
                    sigOut.flush();
                } catch (final NoSuchAlgorithmException e) {
                    // Cannot occur
                    throw new AjaxException(AjaxException.Code.UnexpectedError, e, e.getMessage());
                } finally {
                    AbstractFileAction.close(documentStream);
                    AbstractFileAction.close(sigOut);
                }
            }
        });
        m.put("documentpatch", new Handler() {
            
            public void handle(final AJAXRequestData data, final HttpServletRequest req, final HttpServletResponse res, final ServerSession session) throws IOException, AbstractOXException {
                final AJAXInfostoreRequest request = new AJAXInfostoreRequest(data, session);
                request.require(AbstractFileAction.Param.ID);

                final IDBasedFileAccess fileAccess = request.getFileAccess();

                final List<Closeable> closeables = new ArrayList<Closeable>(4);
                java.io.File baseFile = null;
                java.io.File patchedFile = null;
                try {
                    final int buflen = 8192;
                    /*
                     * Stream document to temporary file
                     */
                    baseFile = AbstractFileAction.newTempFile();
                    {
                        final InputStream documentStream = fileAccess.getDocument(request.getId(), request.getVersion());
                        closeables.add(0, documentStream);
                        final OutputStream baseOut = new FileOutputStream(baseFile);
                        closeables.add(0, baseOut);
                        // Buffer
                        final byte[] buf = new byte[buflen];
                        for (int read; (read = documentStream.read(buf, 0, buflen)) > 0;) {
                            baseOut.write(buf, 0, read);
                        }
                        baseOut.flush();
                        AbstractFileAction.close(baseOut);
                        closeables.remove(0);
                        AbstractFileAction.close(documentStream);
                        closeables.remove(0);
                    }
                    /*
                     * Stream patch to patchOut
                     */
                    patchedFile = AbstractFileAction.newTempFile();
                    final OutputStream patchOut = new FileOutputStream(patchedFile);
                    closeables.add(patchOut);
                    final InputStream requestStream = data.getUploadStream();
                    closeables.add(0, requestStream);
                    final Rdiff rdiff = new Rdiff();
                    rdiff.rebuildFile(baseFile, requestStream, patchOut);
                    patchOut.flush();
                    AbstractFileAction.close(requestStream);
                    closeables.remove(0);
                    AbstractFileAction.close(patchOut);
                    closeables.remove(0);
                    /*
                     * Update
                     */
                    final InputStream patchIn = new FileInputStream(patchedFile);
                    closeables.add(patchIn);
                    
                    final DefaultFile fileAddress = new DefaultFile();
                    fileAddress.setId(request.getId());
                    fileAddress.setVersion(request.getVersion());
                    if (request.has("folder")) {
                        fileAddress.setFolderId(request.getFolderId());
                    } else {
                        final String folderId = request.getFileAccess().getFileMetadata(request.getId(), request.getVersion()).getFolderId();
                        fileAddress.setFolderId(folderId);
                    }
                    fileAccess.saveDocument(fileAddress, patchIn, FileStorageFileAccess.DISTANT_FUTURE);
                } finally {
                    while (!closeables.isEmpty()) {
                        AbstractFileAction.close(closeables.remove(0));
                    }
                    AbstractFileAction.delete(baseFile);
                    AbstractFileAction.delete(patchedFile);
                }
            }
        });
        
        m.put("debug", new Handler() {

            public void handle(final AJAXRequestData data, final HttpServletRequest req, final HttpServletResponse res, final ServerSession session) throws IOException, AbstractOXException {
                final byte[] buf = new byte[123];
                final ServletOutputStream outputStream = res.getOutputStream();
                try {
                    for (int i = 0; i < 100; i++) {
                        outputStream.write(buf, 0, buf.length);
                    }
                } catch (final Throwable t) {
                    t.printStackTrace();
                }
            }
            
        });
        HANDLERS = Collections.unmodifiableMap(m);
    }

    /**
     * Initializes a new {@link FileServlet}.
     *
     * @param factory
     */
    public FileServlet() {
        super(FileActionFactory.INSTANCE);
    }

    @Override
    protected boolean hasModulePermission(final ServerSession session) {
        return true;
    }

    @Override
    protected boolean handleIndividually(final String action, final AJAXRequestData data, final HttpServletRequest req, final HttpServletResponse res) throws IOException, AbstractOXException {
        final Handler handler = HANDLERS.get(action.toLowerCase(Locale.ENGLISH));
        if (null == handler) {
            return false;
        }
        handler.handle(data, req, res, getSessionObject(req));
        return true;
    }
    
    @Override
    protected boolean preferStream(final String action, final HttpServletRequest req, final HttpServletResponse resp, final boolean suggestion) {
        return action.equals("documentpatch") || action.equals("documentdelta")  || suggestion;
    }

}
