package com.openexchange.contacts.ldap;

import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.util.List;
import java.util.Properties;

import junit.framework.TestCase;

import com.openexchange.contacts.ldap.contacts.ContactSearchtermLdapConverter;
import com.openexchange.contacts.ldap.exceptions.LdapConfigurationException;
import com.openexchange.contacts.ldap.property.Mappings;
import com.openexchange.contacts.ldap.property.PropertyHandler;
import com.openexchange.groupware.contact.helpers.ContactField;
import com.openexchange.search.CompositeSearchTerm;
import com.openexchange.search.Operation;
import com.openexchange.search.SearchTerm;
import com.openexchange.search.SingleSearchTerm;
import com.openexchange.search.CompositeSearchTerm.CompositeOperation;
import com.openexchange.search.SingleSearchTerm.SingleOperation;
import com.openexchange.search.internal.operands.ColumnOperand;
import com.openexchange.search.internal.operands.ConstantOperand;

public class ContactSearchtermLdapConverterTest extends TestCase {


	private Mappings mappings;
	private ContactSearchtermLdapConverter converter;

	@Override
    public void setUp() throws LdapConfigurationException, FileNotFoundException, IOException {
        Properties props = new Properties();
        String mappingfile = "etc/contacts-ldap/mapping.ads.properties.example";
        props.load(new FileInputStream(mappingfile));
        mappings = Mappings.getMappingsFromProperties(props, PropertyHandler.bundlename + "mapping.ads", mappingfile);
        converter = new ContactSearchtermLdapConverter();
        converter.setMappings(mappings);
    }

	protected void testFilterTranslations(SingleOperation op, String expected){
		SingleSearchTerm term = new SingleSearchTerm(op);
		term.addOperand(new ConstantOperand<String>("a"));
		term.addOperand(new ConstantOperand<String>("b"));
		
		test(term, expected);
	}
	
	protected void test(SearchTerm term, String expected){
		converter.parse(term);
		String actualString = converter.getQueryString();
		
		assertEquals(expected, actualString);
	}
	
	public void testEquals(){
		testFilterTranslations(SingleSearchTerm.SingleOperation.EQUALS, "(a=b)");
	}
	
	public void testGreaterOrEqual(){
		testFilterTranslations(SingleSearchTerm.SingleOperation.GREATER_OR_EQUAL, "(a>=b)");
	}
	
	public void testLessOrEqual(){
		testFilterTranslations(SingleSearchTerm.SingleOperation.LESS_OR_EQUAL, "(a<=b)");
	}
	
	public void testGreater(){
		testFilterTranslations(SingleSearchTerm.SingleOperation.GREATER_THAN, "(&(a>=b)(!(a=b)))");
	}
	
	public void testLess(){
		testFilterTranslations(SingleSearchTerm.SingleOperation.LESS_THAN, "(&(a<=b)(!(a=b)))");
	}

	public void testSingleSearchTermWithPotentialInjection(){
		SingleSearchTerm equals = new SingleSearchTerm(SingleSearchTerm.SingleOperation.EQUALS);
		equals.addOperand(new ConstantOperand<String>("fie*ldn\\ame"));
		equals.addOperand(new ConstantOperand<String>("\u0000"));
		
		test(equals, "(fie*ldn\\5came=\\00)");
	}
	
	public void testSingleSearchTermWithNegations(){
		SingleSearchTerm equals = new SingleSearchTerm(SingleSearchTerm.SingleOperation.EQUALS);
		equals.addOperand(new ConstantOperand<String>("fieldname"));
		equals.addOperand(new ConstantOperand<String>("value"));
		
		CompositeSearchTerm not = new CompositeSearchTerm(CompositeSearchTerm.CompositeOperation.NOT);
		not.addSearchTerm(equals);
		
		test(not,"(!(fieldname=value))");
	}
	
	public void testUseOfLikeInsteadOfEqualsInCaseOfAsteriskSearch(){
		SingleSearchTerm term = new SingleSearchTerm(SingleSearchTerm.SingleOperation.EQUALS);
		term.addOperand(new ConstantOperand<String>("fieldname"));
		term.addOperand(new ConstantOperand<String>("value*"));
		
		test(term, "(fieldname=value*)");
	}

	public void testComplexSearchTermStringForStrings(){
		SingleSearchTerm term1 = new SingleSearchTerm(SingleSearchTerm.SingleOperation.EQUALS);
		term1.addOperand(new ConstantOperand<String>("field1name"));
		term1.addOperand(new ConstantOperand<String>("value1"));
		
		SingleSearchTerm term2 = new SingleSearchTerm(SingleSearchTerm.SingleOperation.GREATER_OR_EQUAL);
		term2.addOperand(new ConstantOperand<String>("field2name"));
		term2.addOperand(new ConstantOperand<String>("value2"));
		
		CompositeSearchTerm term = new CompositeSearchTerm(CompositeOperation.OR);
		term.addSearchTerm(term1);
		term.addSearchTerm(term2);
		
		test(term, "(|(field1name=value1)(field2name>=value2))");
	}
	
	/**
	 * This is for fields that have been named in a way that is not used in OX
	 */
	public void testJsonLdapDifferentFieldnameTranslation(){
		ContactField field = ContactField.GIVEN_NAME;
		
		SingleSearchTerm term = new SingleSearchTerm(SingleSearchTerm.SingleOperation.EQUALS);
		term.addOperand(new ColumnOperand(field.getAjaxName()));
		term.addOperand(new ConstantOperand<String>("value1"));
		
		test(term, "(" + mappings.get(field) + "=value1)");
	}
	

	/**
	 * This is for fields which name can be directly mapped to an OX field name
	 */
	public void testJsonLdapRegularFieldnameTranslation(){
		ContactField field = ContactField.POSTAL_CODE_BUSINESS;
		
		SingleSearchTerm term = new SingleSearchTerm(SingleSearchTerm.SingleOperation.EQUALS);
		term.addOperand(new ColumnOperand(field.getAjaxName()));
		term.addOperand(new ConstantOperand<String>("value1"));
		
		test(term, "(" + mappings.get(field) + "=value1)");
	}
	
	/**
	 * This is for querying something like "is the value null?"
	 */
	public void testIsNullOperator(){
		ContactField field = ContactField.GIVEN_NAME;

		SingleSearchTerm term = new SingleSearchTerm(SingleSearchTerm.SingleOperation.ISNULL);
		term.addOperand(new ColumnOperand(field.getAjaxName()));
	
		test(term, "(!(" + mappings.get(field) + "=*))");
	}

	/**
	 * The display name is a field that can be stored in different ldap fields 
	 * (depending whether it is a contact or a distribution list) so we need 
	 * to make sure we search in all of them. And only search for them if distributionlists are activated
	 */
	public void testSearchForDisplaynameActivated(){
		ContactField field = ContactField.DISPLAY_NAME;
		
		converter.setDistributionlistActive(true);
		
		SingleSearchTerm term = new SingleSearchTerm(SingleSearchTerm.SingleOperation.EQUALS);
		term.addOperand(new ColumnOperand(field.getAjaxName()));
		term.addOperand(new ConstantOperand<String>("value1"));
		
		test(term, "(|(" + mappings.get(field) + "=value1)(" + mappings.getDistributionlistname() + "=value1))");
	}

	/**
	 * The display name is a field that can be stored in different ldap fields 
	 * (depending whether it is a contact or a distribution list) so we need 
	 * to make sure we search in all of them. And only search for them if distributionlists are activated
	 */
	public void testSearchForDisplaynameDeactivated(){
	    ContactField field = ContactField.DISPLAY_NAME;
	    
	    SingleSearchTerm term = new SingleSearchTerm(SingleSearchTerm.SingleOperation.EQUALS);
	    term.addOperand(new ColumnOperand(field.getAjaxName()));
	    term.addOperand(new ConstantOperand<String>("value1"));
	    
	    test(term, "(" + mappings.get(field) + "=value1)");
	}

	/**
	 * Distributionlists don't have a given name so we have to search in their name field too
	 */
	public void testGivenNameDistri() {
	    ContactField field = ContactField.GIVEN_NAME;
	    
	    converter.setDistributionlistActive(true);

	    SingleSearchTerm term = new SingleSearchTerm(SingleSearchTerm.SingleOperation.EQUALS);
	    term.addOperand(new ColumnOperand(field.getAjaxName()));
	    term.addOperand(new ConstantOperand<String>("value1"));

	    test(term, "(|(" + mappings.get(field) + "=value1)(" + mappings.getDistributionlistname() + "=value1))");
	}
	
	/**
	 * As above but this time without active distributionlist to check if things aren't added if not enabled
	 */
	public void testGivenNameDistriNotActive() {
	    ContactField field = ContactField.GIVEN_NAME;
	    
	    SingleSearchTerm term = new SingleSearchTerm(SingleSearchTerm.SingleOperation.EQUALS);
	    term.addOperand(new ColumnOperand(field.getAjaxName()));
	    term.addOperand(new ConstantOperand<String>("value1"));
	    
	    test(term, "(" + mappings.get(field) + "=value1)");
	}
	
	/**
	 * Distributionlists don't have a last name so we have to search in their name field too
	 */
	public void testLastNameDistri() {
	    ContactField field = ContactField.SUR_NAME;
	    
	    converter.setDistributionlistActive(true);
	    
	    SingleSearchTerm term = new SingleSearchTerm(SingleSearchTerm.SingleOperation.EQUALS);
	    term.addOperand(new ColumnOperand(field.getAjaxName()));
	    term.addOperand(new ConstantOperand<String>("value1"));
	    
	    test(term, "(|(" + mappings.get(field) + "=value1)(" + mappings.getDistributionlistname() + "=value1))");
	}
	
	/**
	 * As above but this time without active distributionlist to check if things aren't added if not enabled
	 */
	public void testLastNameDistriNotActive() {
	    ContactField field = ContactField.SUR_NAME;
	    
	    SingleSearchTerm term = new SingleSearchTerm(SingleSearchTerm.SingleOperation.EQUALS);
	    term.addOperand(new ColumnOperand(field.getAjaxName()));
	    term.addOperand(new ConstantOperand<String>("value1"));
	    
	    test(term, "(" + mappings.get(field) + "=value1)");
	}
	
	/**
	 * Test which checks if different field in an and term aren't recognized for the LDAP workaround 
	 */
	public void testDiffentFieldsInAnd(){
	    String folderFieldName = ContactField.FOLDER_ID.getAjaxName();
	    final ContactField givenName = ContactField.GIVEN_NAME;
	    final ContactField surName = ContactField.SUR_NAME;
	    final String lastName = givenName.getAjaxName();
	    
	    SingleSearchTerm term1 = new SingleSearchTerm(SingleSearchTerm.SingleOperation.EQUALS);
	    term1.addOperand(new ColumnOperand(folderFieldName));
	    term1.addOperand(new ConstantOperand<String>("1"));
	    
	    SingleSearchTerm term2 = new SingleSearchTerm(SingleSearchTerm.SingleOperation.GREATER_OR_EQUAL);
	    term2.addOperand(new ColumnOperand(surName.getAjaxName()));
	    final String firstLetter = "A";
	    term2.addOperand(new ConstantOperand<String>(firstLetter));
	    
	    SingleSearchTerm term3 = new SingleSearchTerm(SingleSearchTerm.SingleOperation.LESS_THAN);
	    term3.addOperand(new ColumnOperand(lastName));
	    final String secondLetter = "C";
	    term3.addOperand(new ConstantOperand<String>(secondLetter));
	    
	    
	    CompositeSearchTerm term = new CompositeSearchTerm(CompositeOperation.AND);
	    term.addSearchTerm(term1);
	    term.addSearchTerm(term2);
	    term.addSearchTerm(term3);
	    
	    converter.parse(term);
	    
	    final String finalLetter = String.valueOf((char)(secondLetter.charAt(0)));
	    assertEquals( "(&" +
	        "(" + mappings.get(surName) + ">=" + firstLetter + ")"+
	        "(&" +
	        "(" + mappings.get(givenName) + "<=" + finalLetter + ")"+
	        "(!(" + mappings.get(givenName) + "=" + finalLetter + "))"+
	        ")" +
	        ")" , converter.getQueryString());
	    
	}
	
	/**
	 * Test which tests if a normal GUI letter search operation shows the right result 
	 */
	public void testNormalOperation(){
	    String folderFieldName = ContactField.FOLDER_ID.getAjaxName();
	    final ContactField givenName = ContactField.GIVEN_NAME;
        final String lastName = givenName.getAjaxName();
	    
	    SingleSearchTerm term1 = new SingleSearchTerm(SingleSearchTerm.SingleOperation.EQUALS);
	    term1.addOperand(new ColumnOperand(folderFieldName));
	    term1.addOperand(new ConstantOperand<String>("1"));
	    
	    SingleSearchTerm term2 = new SingleSearchTerm(SingleSearchTerm.SingleOperation.GREATER_OR_EQUAL);
	    term2.addOperand(new ColumnOperand(lastName));
	    final String firstLetter = "A";
        term2.addOperand(new ConstantOperand<String>(firstLetter));
	    
	    SingleSearchTerm term3 = new SingleSearchTerm(SingleSearchTerm.SingleOperation.LESS_THAN);
        term3.addOperand(new ColumnOperand(lastName));
	    final String secondLetter = "C";
        term3.addOperand(new ConstantOperand<String>(secondLetter));
	    
	    
	    CompositeSearchTerm term = new CompositeSearchTerm(CompositeOperation.AND);
	    term.addSearchTerm(term1);
	    term.addSearchTerm(term2);
	    term.addSearchTerm(term3);
	    
	    converter.parse(term);
	    
	    final String finalLetter = String.valueOf((char)(secondLetter.charAt(0) - 1));
        assertEquals( "(&(|(" + mappings.get(givenName) + "=" + firstLetter + "*)(" + mappings.get(givenName) + "=" + finalLetter + "*)))" , converter.getQueryString());
	}
	
	/**
	 * Test which tests if a normal GUI letter search operation shows the right result 
	 */
	public void testNormalOperationJapanese(){
	    String folderFieldName = ContactField.FOLDER_ID.getAjaxName();
	    final ContactField givenName = ContactField.GIVEN_NAME;
	    final String lastName = givenName.getAjaxName();
	    
	    SingleSearchTerm term1 = new SingleSearchTerm(SingleSearchTerm.SingleOperation.EQUALS);
	    term1.addOperand(new ColumnOperand(folderFieldName));
	    term1.addOperand(new ConstantOperand<String>("1"));
	    
	    SingleSearchTerm term2 = new SingleSearchTerm(SingleSearchTerm.SingleOperation.GREATER_OR_EQUAL);
	    term2.addOperand(new ColumnOperand(lastName));
	    final String firstLetter = "\u30a2";
	    term2.addOperand(new ConstantOperand<String>(firstLetter));
	    
	    SingleSearchTerm term3 = new SingleSearchTerm(SingleSearchTerm.SingleOperation.LESS_THAN);
	    term3.addOperand(new ColumnOperand(lastName));
	    final String secondLetter = "\u30aa";
	    term3.addOperand(new ConstantOperand<String>(secondLetter));
	    
	    
	    CompositeSearchTerm term = new CompositeSearchTerm(CompositeOperation.AND);
	    term.addSearchTerm(term1);
	    term.addSearchTerm(term2);
	    term.addSearchTerm(term3);
	    
	    converter.parse(term);
	    
	    final String finalLetter = String.valueOf((char)(secondLetter.charAt(0) - 1));
	    assertEquals( "(&" +
	        "(|" + 
	        "(" + mappings.get(givenName) + "=" + firstLetter + "*)" +
	        "(" + mappings.get(givenName) + "=" + '\u30a3' + "*)" +
	        "(" + mappings.get(givenName) + "=" + '\u30a4' + "*)" +
	        "(" + mappings.get(givenName) + "=" + '\u30a5' + "*)" +
	        "(" + mappings.get(givenName) + "=" + '\u30a6' + "*)" +
	        "(" + mappings.get(givenName) + "=" + '\u30a7' + "*)" +
	        "(" + mappings.get(givenName) + "=" + '\u30a8' + "*)" +
	        "(" + mappings.get(givenName) + "=" + finalLetter + "*)" +
	        "))" , converter.getQueryString());
	}
	
	/**
	 * Test which tests if a normal GUI letter search operation with one more argument works correctly 
	 */
	public void testThreeOperations(){
	    String folderFieldName = ContactField.FOLDER_ID.getAjaxName();
	    final ContactField givenName = ContactField.GIVEN_NAME;
	    final String lastName = givenName.getAjaxName();
	    
	    SingleSearchTerm term1 = new SingleSearchTerm(SingleSearchTerm.SingleOperation.EQUALS);
	    term1.addOperand(new ColumnOperand(folderFieldName));
	    term1.addOperand(new ConstantOperand<String>("1"));
	    
	    SingleSearchTerm term2 = new SingleSearchTerm(SingleSearchTerm.SingleOperation.GREATER_OR_EQUAL);
	    term2.addOperand(new ColumnOperand(lastName));
	    final String firstLetter = "A";
	    term2.addOperand(new ConstantOperand<String>(firstLetter));
	    
	    SingleSearchTerm term3 = new SingleSearchTerm(SingleSearchTerm.SingleOperation.LESS_THAN);
	    term3.addOperand(new ColumnOperand(lastName));
	    final String secondLetter = "C";
	    term3.addOperand(new ConstantOperand<String>(secondLetter));

	    SingleSearchTerm term4 = new SingleSearchTerm(SingleSearchTerm.SingleOperation.EQUALS);
	    term4.addOperand(new ColumnOperand(lastName));
	    final String thirdLetter = "D";
	    term4.addOperand(new ConstantOperand<String>(thirdLetter));
	    
	    
	    CompositeSearchTerm term = new CompositeSearchTerm(CompositeOperation.AND);
	    term.addSearchTerm(term1);
	    term.addSearchTerm(term2);
	    term.addSearchTerm(term3);
	    term.addSearchTerm(term4);
	    
	    converter.parse(term);
	    
	    final String finalLetter = String.valueOf((char)(secondLetter.charAt(0) - 1));
	    assertEquals("(&" + 
	        "(|(" + mappings.get(givenName) + "=" + firstLetter + "*)(" + mappings.get(givenName) + "=" + finalLetter + "*))" +
	        "(" + mappings.get(givenName) + "=" + thirdLetter + ")" +
	        ")" , converter.getQueryString());
	}
	
	/**
	 * Test which test if a normal GUI letter search operation shows the right result 
	 */
	public void testNormalOperationDistrilists(){
	    converter.setDistributionlistActive(true);
	    
	    String folderFieldName = ContactField.FOLDER_ID.getAjaxName();
	    final ContactField givenName = ContactField.GIVEN_NAME;
	    final String lastName = givenName.getAjaxName();
	    
	    SingleSearchTerm term1 = new SingleSearchTerm(SingleSearchTerm.SingleOperation.EQUALS);
	    term1.addOperand(new ColumnOperand(folderFieldName));
	    term1.addOperand(new ConstantOperand<String>("1"));
	    
	    SingleSearchTerm term2 = new SingleSearchTerm(SingleSearchTerm.SingleOperation.GREATER_OR_EQUAL);
	    term2.addOperand(new ColumnOperand(lastName));
	    final String firstLetter = "A";
	    term2.addOperand(new ConstantOperand<String>(firstLetter));
	    
	    SingleSearchTerm term3 = new SingleSearchTerm(SingleSearchTerm.SingleOperation.LESS_THAN);
	    term3.addOperand(new ColumnOperand(lastName));
	    final String secondLetter = "C";
	    term3.addOperand(new ConstantOperand<String>(secondLetter));
	    
	    
	    CompositeSearchTerm term = new CompositeSearchTerm(CompositeOperation.AND);
	    term.addSearchTerm(term1);
	    term.addSearchTerm(term2);
	    term.addSearchTerm(term3);
	    
	    converter.parse(term);
	    
	    final String finalLetter = String.valueOf((char)(secondLetter.charAt(0) - 1));
	    assertEquals( "(&(|" +
	        "(|(" + mappings.get(givenName) + "=" + firstLetter + "*)(" + mappings.getDistributionlistname() + "=" + firstLetter + "*))" +
	        "(|(" + mappings.get(givenName) + "=" + finalLetter + "*)(" + mappings.getDistributionlistname() + "=" + finalLetter + "*))" +
	        "))" , converter.getQueryString());
	    
	}
	
	/**
	 * Folders have no place in an ldap query 
	 */
	public void testFolderRemoval(){
		String folderFieldName = ContactField.FOLDER_ID.getAjaxName();
		ContactField distractingField = ContactField.CITY_HOME;
		
		SingleSearchTerm term1 = new SingleSearchTerm(SingleSearchTerm.SingleOperation.EQUALS);
		term1.addOperand(new ColumnOperand(folderFieldName));
		term1.addOperand(new ConstantOperand<String>("1"));

		SingleSearchTerm term2 = new SingleSearchTerm(SingleSearchTerm.SingleOperation.LESS_OR_EQUAL);
		term2.addOperand(new ConstantOperand("distraction"));
		term2.addOperand(new ConstantOperand<String>("not-a-folder"));

		SingleSearchTerm term3 = new SingleSearchTerm(SingleSearchTerm.SingleOperation.GREATER_THAN);
		term3.addOperand(new ColumnOperand(folderFieldName));
		term3.addOperand(new ConstantOperand<String>("2"));

		SingleSearchTerm term4 = new SingleSearchTerm(SingleSearchTerm.SingleOperation.GREATER_THAN);
		term4.addOperand(new ColumnOperand(folderFieldName));
		term4.addOperand(new ConstantOperand<String>("strings-can-be-folder-names-too"));

		CompositeSearchTerm term = new CompositeSearchTerm(CompositeOperation.OR);
		term.addSearchTerm(term1);
		term.addSearchTerm(term2);
		term.addSearchTerm(term3);
		term.addSearchTerm(term4);

		converter.parse(term);
		
		assertEquals( "(|(distraction<=not-a-folder))" , converter.getQueryString());

		List<String> folders = converter.getFolders();
		assertEquals(3, folders.size());
		assertTrue(folders.contains("1"));
		assertTrue(folders.contains("2"));
		assertTrue(folders.contains("strings-can-be-folder-names-too"));
	}

}
