/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2012 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.carddav;

import gnu.trove.map.TIntObjectMap;
import gnu.trove.map.hash.TIntObjectHashMap;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import javax.servlet.http.HttpServletResponse;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import com.openexchange.api2.ContactSQLInterface;
import com.openexchange.api2.OXException;
import com.openexchange.api2.RdbContactSQLImpl;
import com.openexchange.carddav.reports.Syncstatus;
import com.openexchange.config.cascade.ConfigCascadeException;
import com.openexchange.config.cascade.ConfigView;
import com.openexchange.config.cascade.ConfigViewFactory;
import com.openexchange.folderstorage.FolderException;
import com.openexchange.folderstorage.FolderExceptionErrorMessage;
import com.openexchange.folderstorage.FolderResponse;
import com.openexchange.folderstorage.FolderService;
import com.openexchange.folderstorage.FolderStorage;
import com.openexchange.folderstorage.Permission;
import com.openexchange.folderstorage.Type;
import com.openexchange.folderstorage.UserizedFolder;
import com.openexchange.folderstorage.database.contentType.ContactContentType;
import com.openexchange.folderstorage.type.PrivateType;
import com.openexchange.folderstorage.type.PublicType;
import com.openexchange.folderstorage.type.SharedType;
import com.openexchange.groupware.AbstractOXException;
import com.openexchange.groupware.container.Contact;
import com.openexchange.groupware.container.DataObject;
import com.openexchange.groupware.container.FolderObject;
import com.openexchange.groupware.contexts.Context;
import com.openexchange.groupware.contexts.impl.ContextException;
import com.openexchange.groupware.ldap.User;
import com.openexchange.groupware.ldap.UserException;
import com.openexchange.session.Session;
import com.openexchange.tools.iterator.SearchIterator;
import com.openexchange.tools.oxfolder.OXFolderAccess;
import com.openexchange.tools.oxfolder.OXFolderException;
import com.openexchange.tools.oxfolder.OXFolderManager;
import com.openexchange.tools.session.SessionHolder;
import com.openexchange.user.UserService;
import com.openexchange.webdav.protocol.WebdavCollection;
import com.openexchange.webdav.protocol.WebdavPath;
import com.openexchange.webdav.protocol.WebdavProtocolException;
import com.openexchange.webdav.protocol.WebdavResource;
import com.openexchange.webdav.protocol.WebdavStatusImpl;
import com.openexchange.webdav.protocol.helpers.AbstractWebdavFactory;

/**
 * {@link GroupwareCarddavFactory}
 * 
 * @author <a href="mailto:francisco.laguna@open-xchange.com">Francisco Laguna</a>
 * @author <a href="mailto:tobias.friedrich@open-xchange.com">Tobias Friedrich</a>
 */
public class GroupwareCarddavFactory extends AbstractWebdavFactory {

	private static final String OVERRIDE_NEXT_SYNC_TOKEN_PROPERTY = "com.openexchange.carddav.overridenextsynctoken";
	
	private final static int[] FIELDS_FOR_ALL_REQUEST = { 
		DataObject.OBJECT_ID, 
		DataObject.LAST_MODIFIED, 
		DataObject.CREATION_DATE, 
		Contact.MARK_AS_DISTRIBUTIONLIST, 
		Contact.UID 
	};

	private static final Log LOG = LogFactory.getLog(GroupwareCarddavFactory.class);

	public static final CarddavProtocol PROTOCOL = new CarddavProtocol();

	private final FolderService folders;

	private final SessionHolder sessionHolder;

	private final ThreadLocal<State> stateHolder = new ThreadLocal<State>();

	private final ConfigViewFactory configs;

	private final UserService users;
	
	private static final int SC_DELETED = 404;
	
	public GroupwareCarddavFactory(final FolderService folders, final SessionHolder sessionHolder, 
			final ConfigViewFactory configs, final UserService users) {
		super();
		this.folders = folders;
		this.sessionHolder = sessionHolder;
		this.configs = configs;
		this.users = users;
	}
	
	@Override
	public void beginRequest() {
		super.beginRequest();
		stateHolder.set(new State(this));
	}

	@Override
	public void endRequest(final int status) {
		stateHolder.set(null);
		super.endRequest(status);
	}

	public CarddavProtocol getProtocol() {
		return PROTOCOL;
	}

	public WebdavCollection resolveCollection(final WebdavPath url) throws WebdavProtocolException {
		if (url.size() > 1) {
			throw new WebdavProtocolException(url, SC_DELETED);
		}
		if (isRoot(url)) {
			return mixin(new RootCollection(this));
		}
		return mixin(new AggregatedCollection(url, this));
	}

	// TODO: i18n

	public boolean isRoot(final WebdavPath url) {
		return url.size() == 0;
	}

	public WebdavResource resolveResource(final WebdavPath url) throws WebdavProtocolException {
		if (url.size() == 2) {
			return mixin(((AggregatedCollection) resolveCollection(url.parent())).getChild(url.name()));
		}
		return resolveCollection(url);
	}

	public FolderService getFolderService() {
		return folders;
	}

	public OXFolderManager getOXFolderManager() throws OXFolderException {
		return OXFolderManager.getInstance(getSession());
	}

	public Context getContext() {
		return sessionHolder.getContext();
	}

	public Session getSession() {
		return sessionHolder.getSessionObject();
	}

	public User getUser() {
		return sessionHolder.getUser();
	}

	public ContactSQLInterface getContactInterface() throws ContextException {
		return new RdbContactSQLImpl(getSession());
	}

	public State getState() {
		return stateHolder.get();
	}

	public OXFolderAccess getOXFolderAccess() {
		return new OXFolderAccess(getContext());
	}

	public ConfigView getConfigView() throws ConfigCascadeException {
		return configs.getView(sessionHolder.getSessionObject().getUserId(), sessionHolder.getSessionObject().getContextId());
	}

	public User resolveUser(final int uid) throws WebdavProtocolException {
		try {
			return users.getUser(uid, getContext());
		} catch (final UserException e) {
			LOG.error(e.getMessage(), e);
			throw new WebdavProtocolException(new WebdavPath(), 500);
		}
	}

	public Syncstatus<WebdavResource> getSyncStatus(final Date since) throws WebdavProtocolException {
		final Syncstatus<WebdavResource> multistatus = new Syncstatus<WebdavResource>();
		final TIntObjectMap<Date> contentsLastModified = new TIntObjectHashMap<Date>();
		Date nextSyncToken = new Date(since.getTime());
		final AggregatedCollection collection = new AggregatedCollection(new WebdavPath().append("Contacts"), this);
		try {
			/*
			 * new and modified contacts
			 */
			final List<Contact> modifiedContacts = this.getState().getModifiedContacts(since);
			for (final Contact contact : modifiedContacts) {
				// add contact resource to multistatus
				final ContactResource resource = new ContactResource(collection, this, contact);
				final int status = contact.getCreationDate().after(since) ? HttpServletResponse.SC_CREATED : HttpServletResponse.SC_OK;
				multistatus.addStatus(new WebdavStatusImpl<WebdavResource>(status, resource.getUrl(), resource));
				// remember aggregated last modified for parent folder								
				final Date contactLastModified = contact.getLastModified();
				final int folderID = contact.getParentFolderID();
				if (false == contentsLastModified.containsKey(folderID) || contactLastModified.after(contentsLastModified.get(folderID))) {
					contentsLastModified.put(folderID, contactLastModified);
				}
				// remember aggregated last modified for next sync token 
				nextSyncToken = Tools.getLatestModified(nextSyncToken, contactLastModified);
			}
			/*
			 * deleted contacts
			 */
			final List<Contact> deletedContacts = this.getState().getDeletedContacts(since);
			for (final Contact contact : deletedContacts) {
				// add contact resource to multistatus
				final ContactResource resource = new ContactResource(collection, this, contact);
				multistatus.addStatus(new WebdavStatusImpl<WebdavResource>(SC_DELETED, resource.getUrl(), resource));
				// remember aggregated last modified for parent folder								
				final Date contactLastModified = contact.getLastModified();
				final int folderID = contact.getParentFolderID();
				if (false == contentsLastModified.containsKey(folderID) || contactLastModified.after(contentsLastModified.get(folderID))) {
					contentsLastModified.put(folderID, contactLastModified);
				}
				// remember aggregated last modified for next sync token 								
				nextSyncToken = Tools.getLatestModified(nextSyncToken, contactLastModified);
			}
			/*
			 * folders
			 */			
			final List<UserizedFolder> folders = this.getState().getFolders();
			for (final UserizedFolder folder : folders) {
				// determine effective last modified of folder and its content
				Date folderLastModified = folder.getLastModifiedUTC();
//				Date folderLastModified = folder.getLastModified();
				final Date folderContentsLastModified = contentsLastModified.get(Integer.parseInt(folder.getID()));
				if (null != folderContentsLastModified && folderContentsLastModified.after(folderLastModified)) {
					folderLastModified = folderContentsLastModified;
				}
				if (folderLastModified.after(since)) {
					// add folder resource to multistatus
					final FolderGroupResource resource = new FolderGroupResource(collection, this, folder);
					resource.overrrideLastModified(folderLastModified);
					final int status = folder.getCreationDate().after(since) ? HttpServletResponse.SC_CREATED : HttpServletResponse.SC_OK;
					multistatus.addStatus(new WebdavStatusImpl<WebdavResource>(status, resource.getUrl(), resource));
					// remember aggregated last modified for next sync token
					nextSyncToken = Tools.getLatestModified(nextSyncToken, folderLastModified);
				}				
			}
			multistatus.setToken(String.valueOf(nextSyncToken.getTime()));
			// TODO: Deleted Folders
			return multistatus;
		} catch (AbstractOXException e) {
			LOG.error(e.getMessage(), e);
			throw new WebdavProtocolException(new WebdavPath(), HttpServletResponse.SC_INTERNAL_SERVER_ERROR);
		}
	}
	
	/**
	 * Sets the next sync token for the current user to <code>"0"</code>, enforcing the next sync status report to contain all changes 
	 * independently of the sync token supplied by the client, thus emulating some kind of slow-sync this way. 
	 */
	public void overrideNextSyncToken() {
		this.setOverrideNextSyncToken("0");		
	}

	/**
	 * Sets the next sync token for the current user to the supplied value.
	 * @param value
	 */
	public void setOverrideNextSyncToken(final String value) {
		try {
			this.users.setUserAttribute(OVERRIDE_NEXT_SYNC_TOKEN_PROPERTY, value, this.getUser().getId(), this.getContext());
		} catch (final UserException e) {
			LOG.error(e.getMessage(), e);
		}
	}
	
	/**
	 * Gets a value indicating the overridden sync token for the current user if defined 
	 * @return
	 */
	public String getOverrideNextSyncToken() {
		try {
			return this.users.getUserAttribute(OVERRIDE_NEXT_SYNC_TOKEN_PROPERTY, this.getUser().getId(), this.getContext());
		} catch (final UserException e) {
			LOG.error(e.getMessage(), e);
		}
		return null;
	}
	
	/**
	 * 
	 * @param token
	 * @return
	 * @throws WebdavProtocolException
	 */
	public Syncstatus<WebdavResource> getSyncStatusSince(String token) throws WebdavProtocolException {
		long since = 0;
		if (null != token && 0 < token.length()) {
			final String overrrideSyncToken = this.getOverrideNextSyncToken();
			if (null != overrrideSyncToken && 0 < overrrideSyncToken.length()) {
				this.setOverrideNextSyncToken(null);
				token = overrrideSyncToken;				
				LOG.debug("Overriding sync token to '" + token + "' for user '" + this.getUser() + "'.");
			}
			try {
				since = Long.parseLong(token);
			} catch (final NumberFormatException e) {
				LOG.warn("Invalid sync token: '" + token + "', falling back to '0'.");								
			}
		}		
		return this.getSyncStatus(new Date(since));				
	}	
	
//	public Syncstatus<WebdavResource> getSyncStatusSinceOLD(String token) throws WebdavProtocolException {
//		if (token.length() == 0) {
//			token = null;
//		}
//		final Date lastModified = token != null ? new Date(
//				Long.parseLong(token)) : new Date(0);
//
//		try {
//
//			final ContactSQLInterface contactInterface = getContactInterface();
//			final Syncstatus<WebdavResource> multistatus = new Syncstatus<WebdavResource>();
//
//			final Collection<Contact> contacts = getState().getContacts();
////			final List<Contact> contacts = getState().getAggregatedContacts();
//
//			final AggregatedCollection collection = new AggregatedCollection(
//					new WebdavPath().append("Contacts"), this);
//
//			Date youngest = lastModified;
//
//			final TIntObjectMap<Date> fastPass = new TIntObjectHashMap<Date>();
//
//			for (final Contact contact : contacts) {
//				final long time1 = contact.getLastModified().getTime();
//				final long time2 = lastModified.getTime();
//				final long diff = time1 - time2;
//				if (diff <= 0) {
//					continue;
//				}
//				final Date date = fastPass.get(contact.getParentFolderID());
//				if (date != null) {
//					if (date.after(contact.getLastModified())) {
//						fastPass.put(contact.getParentFolderID(), date);
//					}
//				} else {
//					fastPass.put(contact.getParentFolderID(),
//							contact.getLastModified());
//				}
//				if (time1 > youngest.getTime()) {
//					youngest = contact.getLastModified();
//				}
//
//				final CarddavResource resource = new CarddavResource(
//						collection, contact, this);
//				int status = 200;
//				if (contact.getCreationDate().after(lastModified)) {
//					status = 201;
//				}
//				multistatus.addStatus(new WebdavStatusImpl<WebdavResource>(
//						status, resource.getUrl(), resource));
//			}
//
//			final List<UserizedFolder> allFolders = getState().getFolders();
//			for (final UserizedFolder f : allFolders) {
//				final int folderId = Integer.parseInt(f.getID());
//				final SearchIterator<Contact> deleted = contactInterface
//						.getDeletedContactsInFolder(folderId,
//								FIELDS_FOR_ALL_REQUEST, lastModified);
//				while (deleted.hasNext()) {
//					final Contact contact = deleted.next();
//					fastPass.put(contact.getParentFolderID(), new Date());
//					final CarddavResource resource = new CarddavResource(
//							collection, contact, this);
//					multistatus.addStatus(new WebdavStatusImpl<WebdavResource>(
//							404, resource.getUrl(), resource));
//					youngest = new Date();
//				}
//				int status = 200;
//				if (f.getCreationDate().after(lastModified)) {
//					status = 201;
//				}
//				final FolderGroupResource fgr = new FolderGroupResource(
//						collection, f, this);
//				fgr.assumeLastModified(fastPass.get(folderId));
//				if (fgr.getLastModified().after(lastModified)) {
//
//					if (fgr.getLastModified().after(youngest)) {
//						youngest = fgr.getLastModified();
//					}
//					multistatus.addStatus(new WebdavStatusImpl<WebdavResource>(
//							status, fgr.getUrl(), fgr));
//				}
//			}
//
//			multistatus.setToken(String.valueOf(youngest.getTime()));
//
//			// TODO: Deleted Folders
//
//			return multistatus;
//		} catch (final Exception x) {
//			LOG.error(x.getMessage(), x);
//			throw new WebdavProtocolException(new WebdavPath(), 500);
//		}
//	}
	
	public static final class State {

		private final GroupwareCarddavFactory factory;

		private Map<String, Contact> uidCache = null;

		private List<UserizedFolder> allFolders = null;
		
		private HashSet<String> folderBlacklist = null;
		
		private int defaultFolderId = Integer.MIN_VALUE;

		/**
		 * Initializes a new {@link State}
		 * @param factory
		 */
		public State(final GroupwareCarddavFactory factory) {
			super();
			this.factory = factory;
		}

		/**
		 * Loads a {@link Contact} containing all data identified by the supplied uid
		 * @param uid
		 * @return
		 * @throws AbstractOXException
		 */
		public Contact load(final String uid) throws AbstractOXException {
			final Contact contact = this.get(uid);
			if (null == contact) {
				if (LOG.isDebugEnabled()) {
					LOG.debug("Contact '" + uid + "' not found, unable to load.");
				}
				return null;
			} else {
				return this.load(contact);
			}
		}

		/**
		 * Loads a {@link Contact} containing all data identified by object- 
		 * and parent folder id found in the supplied contact.
		 * @param contact
		 * @return
		 * @throws OXException
		 * @throws ContextException
		 */
		public Contact load(final Contact contact) throws OXException, ContextException {
			if (null == contact) {
				throw new IllegalArgumentException("contact is null");
			} else if (false == contact.containsObjectID() || false == contact.containsParentFolderID()) {
				throw new IllegalArgumentException("need at least object- and parent folder id");
			}
			return this.load(contact.getObjectID(), contact.getParentFolderID());
		}
		
		/**
		 * Gets all contacts, each containing the basic information as defined by
		 * the <code>FIELDS_FOR_ALL_REQUEST</code> array.
		 * @return
		 * @throws AbstractOXException
		 */
		public Collection<Contact> getContacts() throws AbstractOXException {
			return this.getUidCache().values();			
		}
		
		/**
		 * Gets all contacts from the folder with the supplied id, each 
		 * containing the basic information as defined by the 
		 * <code>FIELDS_FOR_ALL_REQUEST</code> array.
		 * @return
		 * @throws AbstractOXException
		 */
		public List<Contact> getContacts(final int folderId) throws AbstractOXException {
			final List<Contact> contacts = new ArrayList<Contact>();
			final Collection<Contact> allContacts = this.getContacts();
			for (final Contact contact : allContacts) {
				if (folderId == contact.getParentFolderID()) {
					contacts.add(contact);
				}
			}			
			return contacts;
		}
		
		/**
		 * Gets the id of the default contact folder.
		 * @return
		 * @throws OXException
		 */
		public int getDefaultFolderId() throws OXException {
			if (Integer.MIN_VALUE == this.defaultFolderId) {
				this.defaultFolderId = this.factory.getOXFolderAccess().getDefaultFolder(
						this.factory.getUser().getId(), FolderObject.CONTACT).getObjectID();
			}
			return this.defaultFolderId;
		}
		
		/**
		 * Gets the folder identified with the supplied id.
		 * @param id
		 * @return
		 * @throws OXException 
		 * @throws ConfigCascadeException 
		 * @throws FolderException 
		 */
		public UserizedFolder getFolder(final String id) throws FolderException, ConfigCascadeException, OXException {
			final List<UserizedFolder> folders = this.getFolders();
			for (final UserizedFolder folder : folders) {
				if (id.equals(folder.getID())) {
					return folder;
				}
			}
			LOG.warn("Folder '" + id + "' not found.");
			return null;
		}

		/**
		 * Gets a list of all folders.
		 * 
		 * @return
		 * @throws FolderException
		 * @throws ConfigCascadeException
		 * @throws OXException
		 */
		public synchronized List<UserizedFolder> getFolders() throws FolderException, OXException {
			if (null == this.allFolders) {
				if (CarddavProtocol.REDUCED_FOLDER_SET) {
					this.allFolders = getReducedFolders();
				} else {
					this.allFolders = getVisibleFolders();
				}
			}
			return this.allFolders;
		}
		
	    private List<UserizedFolder> getReducedFolders() throws FolderException, OXException {
	    	final List<UserizedFolder> folders = new ArrayList<UserizedFolder>();
			final FolderService folderService = this.factory.getFolderService();
			final UserizedFolder globalAddressBookFolder = folderService.getFolder(
					FolderStorage.REAL_TREE_ID, FolderStorage.GLOBAL_ADDRESS_BOOK_ID, this.factory.getSession(), null);
			if (false == this.blacklisted(globalAddressBookFolder)) {
				folders.add(globalAddressBookFolder);
			}
			final UserizedFolder defaultContactsFolder = folderService.getFolder(
					FolderStorage.REAL_TREE_ID, Integer.toString(this.getDefaultFolderId()), this.factory.getSession(), null);
			if (false == this.blacklisted(defaultContactsFolder)) {
				folders.add(defaultContactsFolder);
			}
			return folders;
	    }
		
		/**
		 * Gets a list of all visible folders.
		 * @return
		 * @throws FolderException
		 */
	    private List<UserizedFolder> getVisibleFolders() throws FolderException {
	    	final List<UserizedFolder> folders = new ArrayList<UserizedFolder>();
	    	folders.addAll(this.getVisibleFolders(PrivateType.getInstance()));
	    	folders.addAll(this.getVisibleFolders(PublicType.getInstance()));
	    	folders.addAll(this.getVisibleFolders(SharedType.getInstance()));
	    	return folders;
	    }
		
		/**
		 * Gets a list containing all visible folders of the given {@link Type}.
		 * @param type
		 * @return
		 * @throws FolderException 
		 */
	    private List<UserizedFolder> getVisibleFolders(final Type type) throws FolderException {
	    	final List<UserizedFolder> folders = new ArrayList<UserizedFolder>();
			final FolderService folderService = this.factory.getFolderService();
			final FolderResponse<UserizedFolder[]> visibleFoldersResponse = folderService.getVisibleFolders(
					FolderStorage.REAL_TREE_ID, ContactContentType.getInstance(), type, true, 
					this.factory.getSession(), null);
            final UserizedFolder[] response = visibleFoldersResponse.getResponse();
            for (final UserizedFolder folder : response) {
                if (Permission.READ_OWN_OBJECTS < folder.getOwnPermission().getReadPermission() && false == this.blacklisted(folder)) {
                	folders.add(folder);                	
                }
            }
            return folders;
	    }		
		
		/**
		 * Gets an aggregated {@link Date} representing the last modification 
		 * time of all resources. 
		 * @return
		 * @throws AbstractOXException
		 */
//		public Date getLastModified2() throws AbstractOXException {
//			Date lastModified = new Date(0);
//			final Collection<Contact> contacts = this.getContacts();
//			for (final Contact contact : contacts) {
//				lastModified = Tools.getLatestModified(lastModified, contact);
//			}
//			final List<UserizedFolder> folders = this.getFolders();
//			for (final UserizedFolder folder : folders) {
//				lastModified = this.getYoungerDeletion(folder.getID(), lastModified);				
//			}
//			return lastModified;
//		}
		
		/**
		 * Gets an aggregated {@link Date} representing the last modification time of all resources. 
		 * @return
		 * @throws AbstractOXException
		 */
		public Date getLastModified() throws AbstractOXException {
			Date lastModified = new Date(0);
			final List<UserizedFolder> folders = this.getFolders();
			for (final UserizedFolder folder : folders) {
				lastModified = Tools.getLatestModified(lastModified, this.getLastModified(folder));
			}
			return lastModified;
		}	
		
		/**
		 * Gets the last modification time of the supplied folder, including its contents. 
		 * This covers the folder's last modification time itself, and both updated and 
		 * deleted items inside the folder.
		 * @param folderId
		 * @return
		 * @throws AbstractOXException
		 */
		public Date getLastModified(final UserizedFolder folder) throws AbstractOXException {
			final int[] cols = { DataObject.LAST_MODIFIED };
			Date lastModified = folder.getLastModifiedUTC();
//			Date lastModified = folder.getLastModified();
			final int folderId = Integer.parseInt(folder.getID());
			final ContactSQLInterface contactInterface = this.factory.getContactInterface();
			SearchIterator<Contact> iterator = null;
			try {
				iterator = contactInterface.getModifiedContactsInFolder(folderId, cols, lastModified);				
//				iterator = contactInterface.getContactsInFolder(folderId, 0, 1, DataObject.LAST_MODIFIED, 
//						com.openexchange.groupware.search.Order.DESCENDING, null, cols);				
				while (iterator.hasNext()) {
					lastModified = Tools.getLatestModified(lastModified, iterator.next());
				}
			} finally {
				close(iterator);
			}				
			try {
				iterator = contactInterface.getDeletedContactsInFolder(folderId, cols, lastModified);				
				while (iterator.hasNext()) {
					lastModified = Tools.getLatestModified(lastModified, iterator.next());
				}
			} finally {
				close(iterator);
			}				
			return lastModified;
		}
		
		/**
		 * Gets the last modification time of the supplied folder, including its contents. This covers both updated and deleted items.
		 * @param folderId
		 * @return
		 * @throws AbstractOXException
		 */
		public Date getLastModified(final int folderId) throws AbstractOXException {
			final List<UserizedFolder> folders = this.getFolders();
			for (final UserizedFolder folder : folders) {
				final int id = Integer.parseInt(folder.getID());
				if (id == folderId) {
					return this.getLastModified(folder);
				}
			}
			throw FolderExceptionErrorMessage.INVALID_FOLDER_ID.create(folderId);
		}	
		
		/**
		 * Gets an aggregated list of all modified contacts in all folders since the supplied {@link Date}.
		 * @param since
		 * @return
		 * @throws AbstractOXException
		 */
		public List<Contact> getModifiedContacts(final Date since) throws AbstractOXException  {
			final List<Contact> contacts = new ArrayList<Contact>();
			final List<UserizedFolder> folders = this.getFolders();
			for (final UserizedFolder folder : folders) {
				final int folderId = Integer.parseInt(folder.getID());
				contacts.addAll(this.getModifiedContacts(since, folderId));
			}
			return contacts;
		}
		
		/**
		 * Gets a list of all modified contacts in a folder since the supplied {@link Date}.
		 * @param since
		 * @param folderId
		 * @return
		 * @throws AbstractOXException
		 */
		public List<Contact> getModifiedContacts(final Date since, final int folderId) throws AbstractOXException  {
			final List<Contact> contacts = new ArrayList<Contact>();
			final ContactSQLInterface contactInterface = this.factory.getContactInterface();
			SearchIterator<Contact> iterator = null;
			try {
				iterator = contactInterface.getModifiedContactsInFolder(folderId, FIELDS_FOR_ALL_REQUEST, since);
				while (iterator.hasNext()) {
					final Contact contact = iterator.next();
					contacts.add(contact);						
				}
			} finally {
				close(iterator);
			}
			return contacts;
		}
		
		/**
		 * Gets an aggregated list of all deleted contacts in all folders since the supplied {@link Date}.
		 * @param since
		 * @return
		 * @throws AbstractOXException
		 */
		public List<Contact> getDeletedContacts(final Date since) throws AbstractOXException  {
			final List<Contact> contacts = new ArrayList<Contact>();
			final List<UserizedFolder> folders = this.getFolders();
			for (final UserizedFolder folder : folders) {
				final int folderId = Integer.parseInt(folder.getID());
				contacts.addAll(this.getDeletedContacts(since, folderId));
			}
			return contacts;
		}
		
		/**
		 * Gets an aggregated list of all deleted contacts in a folder since the supplied {@link Date}.
		 * @param since
		 * @param folderId
		 * @return
		 * @throws AbstractOXException
		 */
		public List<Contact> getDeletedContacts(final Date since, final int folderId) throws AbstractOXException  {
			final List<Contact> contacts = new ArrayList<Contact>();
			final ContactSQLInterface contactInterface = this.factory.getContactInterface();
			SearchIterator<Contact> iterator = null;
			try {
				iterator = contactInterface.getDeletedContactsInFolder(folderId, FIELDS_FOR_ALL_REQUEST, since);
				while (iterator.hasNext()) {
					contacts.add(iterator.next());						
				}
			} finally {
				close(iterator);
			}
			return contacts;
		}
		
		private static void close(final SearchIterator<Contact> iterator) {
			if (null != iterator) {
				try { 
					iterator.close();
				} catch (final AbstractOXException e) { 
					LOG.error(e.getMessage(), e); 
				}
			}
		}
		
		/**
		 * Puts the supplied {@link Contact} into the internal uid cache.
		 * @param contact
		 * @return
		 */
//		private void cache(final Contact contact) {
//			if (null == contact || false == contact.containsUid()) {
//				throw new IllegalArgumentException("contact");
//			} else if (null == this.uidCache) {
//				this.uidCache = new HashMap<String, Contact>();
//			}
//			this.uidCache.put(contact.getUid(), contact);			
//		}
		
//		private Date getYoungerDeletion(final String folderId, final Date comparisonDate) throws AbstractOXException {
//			final ContactSQLInterface contactInterface = this.factory.getContactInterface();
//			SearchIterator<Contact> iterator = null;
//			try {
//				iterator = contactInterface.getDeletedContactsInFolder(Integer.parseInt(folderId), FIELDS_FOR_ALL_REQUEST, comparisonDate);
//				if (iterator.hasNext()) {
//					return iterator.next().getLastModified();
////					return new Date();
//				}
//			} finally {
//				close(iterator);
//			}
//			return comparisonDate;
//		}
		
		/**
		 * Determines whether the supplied folder is blacklisted and should be ignored or not.
		 * @param userizedFolder
		 * @return
		 */
		private boolean blacklisted(final UserizedFolder userizedFolder) {
			if (null == this.folderBlacklist) {
				String ignoreFolders = null;
				try {
					ignoreFolders = factory.getConfigView().get("com.openexchange.carddav.ignoreFolders", String.class);
				} catch (final ConfigCascadeException e) {
			        LOG.error(e.getMessage(), e);
				}
				if (null == ignoreFolders || 0 >= ignoreFolders.length()) {
					this.folderBlacklist = new HashSet<String>(0);
				} else {
					this.folderBlacklist = new HashSet<String>(Arrays.asList(ignoreFolders.split("\\s*,\\s*")));
				}
			}
			return this.folderBlacklist.contains(userizedFolder.getID());
		}
		
		/**
		 * Gets a contact object containing the basic information as defined by
		 * the <code>FIELDS_FOR_ALL_REQUEST</code> array.
		 * @param uid
		 * @return
		 * @throws AbstractOXException 
		 */
		public Contact get(final String uid) throws AbstractOXException {			
			return this.getUidCache().get(uid);
		}
		
		/**
		 * 
		 * @param objectId
		 * @param inFolder
		 * @return
		 * @throws OXException
		 * @throws ContextException
		 */
		private Contact load(final int objectId, final int inFolder) throws OXException, ContextException {
			final Contact contact = this.factory.getContactInterface().getObjectById(objectId, inFolder);
			if (null == contact) {
				LOG.warn("Contact '" + objectId + "' in folder '" + inFolder + "' not found.");
			}
			return contact;
		}
		
		private synchronized Map<String, Contact> getUidCache() throws AbstractOXException {
			if (null == this.uidCache) {
				this.uidCache = generateUidCache();
			}
			return this.uidCache;
		}		
		
		private Map<String, Contact> generateUidCache() throws AbstractOXException {
			final HashMap<String, Contact> cache = new HashMap<String, Contact>();			
			final List<UserizedFolder> folders = this.getFolders(); 
			for (final UserizedFolder folder : folders) {
				final int folderId = Integer.parseInt(folder.getID());
				final ContactSQLInterface contactInterface = factory.getContactInterface();
				SearchIterator<Contact> iterator = null;
				try {
					iterator = contactInterface.getContactsInFolder(folderId, 0, 0, -1, null, null, FIELDS_FOR_ALL_REQUEST);
					while (iterator.hasNext()) {
						final Contact contact = iterator.next();
						if (contact.getMarkAsDistribtuionlist()) {
							continue;
						} 
						contact.setParentFolderID(folderId);
						if (false == contact.containsUid() && false == this.tryAddUID(contact, folder)) {
							LOG.warn("No UID found in contact '" + contact.toString() + "', skipping.");
							continue;
						}
						cache.put(contact.getUid(), contact);
					}
				} finally {
					close(iterator);
				}
			}	
			return cache;
		}
		
		private boolean tryAddUID(final Contact contact, final UserizedFolder folder) {
            if (Permission.WRITE_OWN_OBJECTS < folder.getOwnPermission().getWritePermission()) {
            	LOG.debug("Adding uid for contact '" + contact.toString() + "'.");
				try {
					final ContactSQLInterface contactInterface = factory.getContactInterface();
					contact.setUid(UUID.randomUUID().toString());			
					contactInterface.updateContactObject(contact, Integer.parseInt(folder.getID()), contact.getLastModified());
					return true;
				} catch (final AbstractOXException e) {
					LOG.error(e.getMessage(), e);
				}
            }
			return false;
		}
	}
}
