/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2012 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.calendar.itip;

import java.sql.Connection;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import com.openexchange.api.OXObjectNotFoundException;
import com.openexchange.api2.AppointmentSQLInterface;
import com.openexchange.api2.OXException;
import com.openexchange.calendar.AppointmentDiff;
import com.openexchange.groupware.AbstractOXException;
import com.openexchange.groupware.calendar.CalendarDataObject;
import com.openexchange.groupware.calendar.OXCalendarException;
import com.openexchange.groupware.container.Appointment;
import com.openexchange.groupware.container.GroupParticipant;
import com.openexchange.groupware.container.Participant;
import com.openexchange.groupware.container.UserParticipant;
import com.openexchange.groupware.contexts.Context;
import com.openexchange.groupware.ldap.User;
import com.openexchange.groupware.ldap.UserException;
import com.openexchange.groupware.search.AppointmentSearchObject;
import com.openexchange.groupware.search.Order;
import com.openexchange.server.ServiceLookup;
import com.openexchange.session.Session;
import com.openexchange.tools.iterator.SearchIterator;
import com.openexchange.user.UserService;

public class ITipConsistencyCalendar extends ITipCalendarWrapper implements AppointmentSQLInterface {
	protected AppointmentSQLInterface delegate;

	private UserService users;


	static interface ITipStrategy {

		void beforeUpdate(CalendarDataObject cdao)
				throws OXObjectNotFoundException, OXException;

		void afterUpdate(CalendarDataObject cdao);

		void delete(CalendarDataObject appointmentObject, int inFolder,
				Date clientLastModified, boolean checkPermissions)
				throws OXException;

	}

	private class InternalOrganizerStrategy implements ITipStrategy {

		public void beforeUpdate(CalendarDataObject cdao)
				throws OXObjectNotFoundException, OXException {
			// Increase the Sequence Number if dates, recurrences, full_time
			// change or if this is a create exception
			if (!cdao.containsSequence()) {
				try {
					CalendarDataObject loaded = getObjectById(cdao
							.getObjectID());
					AppointmentDiff diff = AppointmentDiff
							.compare(loaded, cdao);
					if (diff.anyFieldChangedOf(Appointment.START_DATE,
							Appointment.END_DATE,
							Appointment.RECURRENCE_POSITION,
							Appointment.RECURRENCE_ID,
							Appointment.RECURRENCE_DATE_POSITION,
							Appointment.RECURRENCE_COUNT,
							Appointment.RECURRENCE_START,
							Appointment.FULL_TIME, Appointment.TITLE,
							Appointment.LOCATION, Appointment.PARTICIPANTS)) {
						cdao.setSequence(loaded.getSequence() + 1);
					}
					if (cdao.getRecurrenceType() == 0) {
						if (loaded.getRecurrenceType() != 0) {
							cdao.setSequence(loaded.getSequence() + 1);
						}
					}
				} catch (SQLException e) {
					throw new OXCalendarException(
							OXCalendarException.Code.SQL_ERROR, e);
				}
			}
		}

		public void afterUpdate(CalendarDataObject cdao) {

		}

		public void delete(CalendarDataObject appointmentObject, int inFolder,
				Date clientLastModified, boolean checkPermissions)
				throws OXException {
			delegate.deleteAppointmentObject(appointmentObject, inFolder,
					clientLastModified, checkPermissions);
		}

	}

	private class ExternalOrganizerStrategy implements ITipStrategy {

		public void beforeUpdate(CalendarDataObject cdao) {
			// TODO Auto-generated method stub

		}

		public void afterUpdate(CalendarDataObject cdao) {
			// TODO Auto-generated method stub

		}

		public void delete(CalendarDataObject appointmentObject, int inFolder,
				Date clientLastModified, boolean checkPermissions)
				throws OXObjectNotFoundException, OXException {
			try {
				CalendarDataObject original = delegate
						.getObjectById(appointmentObject.getObjectID());
				if (onlyOneParticipantRemaining(original)) {
					delegate.deleteAppointmentObject(appointmentObject,
							inFolder, clientLastModified, checkPermissions);
				} else {
					removeCurrentUserFromParticipants(original);
					cleanOccurrencesAndUntil(original);
					original.setExternalOrganizer(true);
					original.setIgnoreConflicts(true);
					delegate.updateAppointmentObject(original, inFolder,
							clientLastModified, checkPermissions);
				}
			} catch (SQLException e) {
				throw new OXCalendarException(
						OXCalendarException.Code.SQL_ERROR);
			}
		}
		
        private void cleanOccurrencesAndUntil(CalendarDataObject original) {
            if (original.containsOccurrence()) {
                original.removeUntil();
            }
        }

        private void removeCurrentUserFromParticipants(
				CalendarDataObject original) {
			// New participants are all externals + all resources + all resolved
			// users from user participants - the current user participant
			List<Participant> participants = new ArrayList<Participant>();
			Participant[] p = original.getParticipants();
			if (p != null) {
				for (Participant participant : p) {
					if (!(participant instanceof GroupParticipant)
							&& !(participant instanceof UserParticipant)) {
						participants.add(participant);
					}
				}
			}

			UserParticipant[] u = original.getUsers();
			List<UserParticipant> newUserParticipants = new ArrayList<UserParticipant>();

			if (u != null) {
				for (UserParticipant userParticipant : u) {
					if (userParticipant.getIdentifier() != session.getUserId()) {
						participants.add(userParticipant);
						newUserParticipants.add(userParticipant);
					}
				}
			}

			original.setParticipants(participants);
			original.setUsers(newUserParticipants);
		}

		private boolean onlyOneParticipantRemaining(CalendarDataObject original) {
			Participant[] participants = original.getParticipants();
			if (participants != null) {
				for (Participant p : participants) {
					if (p instanceof UserParticipant) {
						UserParticipant up = (UserParticipant) p;
						if (up.getIdentifier() != session.getUserId()) {
							return false;
						}
					}
				}
				return true;
			}

			UserParticipant[] userParticipants = original.getUsers();
			if (userParticipants != null) {
				if (userParticipants.length > 1) {
					return false;
				}

				if (userParticipants.length == 0) {
					return true;
				}

				UserParticipant up = userParticipants[0];
				return up.getIdentifier() == session.getUserId();
			}
			return true;
		}

	}

	public ITipConsistencyCalendar(AppointmentSQLInterface delegate,
			Session session, ServiceLookup services) throws AbstractOXException {
		super(session, services);
		this.delegate = delegate;
		this.users = services.getService(UserService.class);
		loadContext();
	}

	public void setIncludePrivateAppointments(boolean include) {
		delegate.setIncludePrivateAppointments(include);
	}

	public boolean getIncludePrivateAppointments() {
		return delegate.getIncludePrivateAppointments();
	}

	public SearchIterator<Appointment> getAppointmentsBetweenInFolder(
			int folderId, int[] cols, Date start, Date end, int orderBy,
			Order order) throws OXException, SQLException {
		return delegate.getAppointmentsBetweenInFolder(folderId, cols, start,
				end, orderBy, order);
	}

	public SearchIterator<Appointment> getAppointmentsBetweenInFolder(
			int folderId, int[] cols, Date start, Date end, int from, int to,
			int orderBy, Order orderDir) throws OXException, SQLException {
		return delegate.getAppointmentsBetweenInFolder(folderId, cols, start,
				end, from, to, orderBy, orderDir);
	}

	public boolean[] hasAppointmentsBetween(Date start, Date end)
			throws OXException {
		return delegate.hasAppointmentsBetween(start, end);
	}

	public SearchIterator<Appointment> getModifiedAppointmentsInFolder(int fid,
			int[] cols, Date since) throws OXException {
		return delegate.getModifiedAppointmentsInFolder(fid, cols, since);
	}

	public SearchIterator<Appointment> getModifiedAppointmentsBetween(
			int userId, Date start, Date end, int[] cols, Date since,
			int orderBy, Order orderDir) throws OXException, SQLException {
		return delegate.getModifiedAppointmentsBetween(userId, start, end,
				cols, since, orderBy, orderDir);
	}

	public SearchIterator<Appointment> getModifiedAppointmentsInFolder(int fid,
			Date start, Date end, int[] cols, Date since) throws OXException,
			SQLException {
		return delegate.getModifiedAppointmentsInFolder(fid, start, end, cols,
				since);
	}

	public SearchIterator<Appointment> getDeletedAppointmentsInFolder(
			int folderId, int[] cols, Date since) throws OXException,
			SQLException {
		return delegate.getDeletedAppointmentsInFolder(folderId, cols, since);
	}

	public SearchIterator<Appointment> getAppointmentsByExtendedSearch(
			AppointmentSearchObject searchObject, int orderBy, Order orderDir,
			int[] cols) throws OXException, SQLException {
		return delegate.getAppointmentsByExtendedSearch(searchObject, orderBy,
				orderDir, cols);
	}

	public SearchIterator<Appointment> searchAppointments(
			AppointmentSearchObject searchObj, int orderBy, Order orderDir,
			int[] cols) throws OXException {
		return delegate.searchAppointments(searchObj, orderBy, orderDir, cols);
	}

	public CalendarDataObject getObjectById(int objectId) throws OXException,
			SQLException, OXObjectNotFoundException {
		return addOrganizer(delegate.getObjectById(objectId));
	}

	public CalendarDataObject getObjectById(int objectId, int inFolder)
			throws OXException, SQLException, OXObjectNotFoundException {
		return addOrganizer(delegate.getObjectById(objectId, inFolder));
	}

	private CalendarDataObject addOrganizer(CalendarDataObject objectById) throws OXException{
		if (objectById.getOrganizer() == null) {
			try {
				User u = users.getUser(objectById.getCreatedBy(), ctx);
				String mail = u.getMail();
				objectById.setOrganizer(mail.toLowerCase());
				objectById.setOrganizerId(u.getId());
			} catch (UserException e) {
				throw new OXException(e);
			}
		}
		return objectById;
	}

	public SearchIterator<Appointment> getObjectsById(
			int[][] objectIdAndInFolder, int[] cols) throws OXException {
		return delegate.getObjectsById(objectIdAndInFolder, cols);
	}

	public Appointment[] insertAppointmentObject(CalendarDataObject cdao)
			throws OXException {
		try {
			setOrganizer(cdao);
			setPrincipal(cdao);
		} catch (OXException x) {
			throw x;
		} catch (AbstractOXException e) {
			throw new OXException(e);
		}
		return delegate.insertAppointmentObject(cdao);
	}
	
	private void setPrincipal(CalendarDataObject cdao) throws AbstractOXException {
		loadContext();
		if (cdao.getPrincipal() == null) {
			int onBehalfOf = onBehalfOf(cdao.getParentFolderID());
			if (onBehalfOf > 0) {
				cdao.setPrincipal(users.getUser(onBehalfOf,ctx).getMail().toLowerCase());
				cdao.setPrincipalId(onBehalfOf);
			}
		} else {
			String principal = cdao.getPrincipal().toLowerCase();
    		if (principal.startsWith("mailto:")) {
    			principal = principal.substring(7);
    		}
    		try {
    			User result = users.searchUser(principal, ctx);
    			int uid = (result != null) ? result.getId() : 0;
    			cdao.setPrincipalId(uid);
    		} catch (UserException e) {
    		}
		}
	}

	private void setOrganizer(CalendarDataObject cdao) throws AbstractOXException {
        if (cdao.getOrganizer() == null) {
        	loadUser();
            cdao.setOrganizer(user.getMail().toLowerCase());
            cdao.setOrganizerId(user.getId());
        } else {
        	
        	String organizer = cdao.getOrganizer().toLowerCase();
    		if (organizer.startsWith("mailto:")) {
    			organizer = organizer.substring(7);
    		}
    		try {
    			User result = users.searchUser(organizer, ctx);
    			int uid = (result != null) ? result.getId() : 0;
    			cdao.setOrganizerId(uid);
    		} catch (UserException e) {
    		}
        }
		
	}



	public Appointment[] updateAppointmentObject(CalendarDataObject cdao,
			int inFolder, Date clientLastModified) throws OXException {
		return updateAppointmentObject(cdao, inFolder, clientLastModified, true);
	}

	public Appointment[] updateAppointmentObject(CalendarDataObject cdao,
			int inFolder, Date clientLastModified, boolean checkPermissions)
			throws OXException {

		ITipStrategy strategy = chooseStrategy(cdao);
		strategy.beforeUpdate(cdao);

		Appointment[] retval = delegate.updateAppointmentObject(cdao, inFolder,
				clientLastModified, checkPermissions);

		if (retval == null || retval.length == 0) {
			strategy.afterUpdate(cdao);
		}

		return retval;
	}

	public void deleteAppointmentObject(CalendarDataObject appointmentObject,
			int inFolder, Date clientLastModified) throws OXException {
		deleteAppointmentObject(appointmentObject, inFolder,
				clientLastModified, true);
	}

	public void deleteAppointmentObject(CalendarDataObject appointmentObject,
			int inFolder, Date clientLastModified, boolean checkPermissions)
			throws OXException {
		ITipStrategy strategy = chooseStrategy(appointmentObject);

		strategy.delete(appointmentObject, inFolder, clientLastModified,
				checkPermissions);
	}

	public void deleteAppointmentsInFolder(int inFolder) throws OXException,
			SQLException {
		delegate.deleteAppointmentsInFolder(inFolder);
	}

	public void deleteAppointmentsInFolder(int inFolder, Connection writeCon)
			throws OXException, SQLException {
		delegate.deleteAppointmentsInFolder(inFolder, writeCon);
	}

	public boolean checkIfFolderContainsForeignObjects(int user_id, int inFolder)
			throws OXException, SQLException {
		return delegate.checkIfFolderContainsForeignObjects(user_id, inFolder);
	}

	public boolean checkIfFolderContainsForeignObjects(int user_id,
			int inFolder, Connection readCon) throws OXException, SQLException {
		return delegate.checkIfFolderContainsForeignObjects(user_id, inFolder,
				readCon);
	}

	public boolean isFolderEmpty(int uid, int fid) throws OXException,
			SQLException {
		return delegate.isFolderEmpty(uid, fid);
	}

	public boolean isFolderEmpty(int uid, int fid, Connection readCon)
			throws OXException, SQLException {
		return delegate.isFolderEmpty(uid, fid, readCon);
	}

	public Date setUserConfirmation(int object_id, int folderId, int user_id,
			int confirm, String confirm_message) throws OXException {
		return delegate.setUserConfirmation(object_id, folderId, user_id,
				confirm, confirm_message);
	}

	public Date setExternalConfirmation(int oid, int folderId, String mail,
			int confirm, String message) throws OXException {
		return delegate.setExternalConfirmation(oid, folderId, mail, confirm,
				message);
	}

	public long attachmentAction(int objectId, int uid, int folderId,
			Session session, Context c, int numberOfAttachments)
			throws OXException {
		return delegate.attachmentAction(objectId, uid, folderId, session, c,
				numberOfAttachments);
	}

	public SearchIterator<Appointment> getFreeBusyInformation(int id, int type,
			Date start, Date end) throws OXException {
		return delegate.getFreeBusyInformation(id, type, start, end);
	}

	public SearchIterator<Appointment> getActiveAppointments(int user_uid,
			Date start, Date end, int[] cols) throws OXException {
		return delegate.getActiveAppointments(user_uid, start, end, cols);
	}

	public SearchIterator<Appointment> getAppointmentsBetween(int user_uid,
			Date start, Date end, int[] cols, int orderBy, Order order)
			throws OXException, SQLException {
		return delegate.getAppointmentsBetween(user_uid, start, end, cols,
				orderBy, order);
	}
	
	public SearchIterator<Appointment> getAppointmentsBetween(Date start, Date end, int cols[], int orderBy, Order order) throws OXException, SQLException {
	    return delegate.getAppointmentsBetween(start, end, cols, orderBy, order);
	}

	public int resolveUid(String uid) throws OXException {
		return delegate.resolveUid(uid);
	}

	public int getFolder(int objectId) throws OXException {
		return delegate.getFolder(objectId);
	}

	private void setOrganizerType(CalendarDataObject appointment)
			throws OXObjectNotFoundException, OXException {
		String organizer = appointment.getOrganizer();
		
		if (organizer == null) {
			Appointment loaded = null;
			try {
				loaded = getObjectById(appointment.getObjectID());
			} catch (SQLException e) {
				appointment.setExternalOrganizer(false);
				return;
			} catch (AbstractOXException x) {
				appointment.setExternalOrganizer(false);
				return;
			}
			organizer = loaded.getOrganizer();
			if (organizer == null || appointment.getOrganizerId() > 0 || appointment.getPrincipalId() > 0) {
				appointment.setExternalOrganizer(false);
				return;
			}
		}

		organizer = organizer.toLowerCase();
		if (organizer.startsWith("mailto:")) {
			organizer = organizer.substring(7);
		}
		int uid = -1;
		try {
			User result = users.searchUser(organizer, ctx);
			uid = (result != null) ? result.getId() : -1;
		} catch (UserException e) {
		}

		if (uid == -1) {
			appointment.setExternalOrganizer(true);
			return;
		}
		appointment.setExternalOrganizer(false);
		return;
	}

	private ITipStrategy chooseStrategy(CalendarDataObject appointment)
			throws OXObjectNotFoundException, OXException {
		setOrganizerType(appointment);
		if (appointment.isExternalOrganizer()) {
			return new ExternalOrganizerStrategy();
		}
		return new InternalOrganizerStrategy();
	}

}
