/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2010 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.tools.servlet;

import com.openexchange.groupware.AbstractOXException;
import com.openexchange.groupware.EnumComponent;

/**
 * Exception for problems in servlets.
 * 
 * @author <a href="mailto:marcus@open-xchange.org">Marcus Klein</a>
 */
public class AjaxException extends AbstractOXException {

    /**
     * For serialization.
     */
    private static final long serialVersionUID = -171563644613142249L;

    /**
     * Initializes a new {@link AjaxException}
     * 
     * @param cause The init cause
     */
    public AjaxException(final AbstractOXException cause) {
        super(cause);
    }

    /**
     * Initializes a new exception using the information provided by the code.
     * 
     * @param code code for the exception.
     * @param cause the cause of the exception.
     * @param messageArgs arguments that will be formatted into the message.
     */
    public AjaxException(final Code code, final Object... messageArgs) {
        this(code, null, messageArgs);
    }

    /**
     * Initializes a new exception using the information provided by the code.
     * 
     * @param code code for the exception.
     * @param cause the cause of the exception.
     * @param messageArgs arguments that will be formatted into the message.
     */
    public AjaxException(final Code code, final Throwable cause, final Object... messageArgs) {
        super(EnumComponent.SERVLET, code.category, code.number, null == code.message ? cause.getMessage() : code.message, cause);
        setMessageArgs(messageArgs);
    }

    /**
     * Error codes for servlet exceptions.
     * 
     * @author <a href="mailto:marcus@open-xchange.org">Marcus Klein</a>
     */
    public enum Code {
        /**
         * Unknown AJAX action: %s.
         */
        UnknownAction("Unknown AJAX action: %s.", Category.CODE_ERROR, 1),
        /**
         * Missing the following field: %s
         */
        MISSING_PARAMETER("Missing the following request parameter: %s", Category.CODE_ERROR, 2),
        /**
         * Missing upload image.
         */
        NoUploadImage("Missing upload image.", Category.CODE_ERROR, 3),
        /**
         * Invalid parameter: %s
         */
        InvalidParameter("Invalid parameter: %s", Category.CODE_ERROR, 4),
        /**
         * I/O error while writing to Writer object: %s
         */
        IOError("I/O error while writing to Writer object: %s", Category.INTERNAL_ERROR, 5),
        /**
         * Missing AJAX request handler for module %s
         */
        MISSING_REQUEST_HANDLER("Missing AJAX request handler for module %s", Category.CODE_ERROR, 6),
        /**
         * Unknown module: %s.
         */
        UNKNOWN_MODULE("Unknown module: %s.", Category.CODE_ERROR, 7),
        /**
         * A harmful attachment was detected.
         */
        HARMFUL_ATTACHMENT("A harmful attachment was detected.", Category.CODE_ERROR, 8),
        /**
         * JSON error: %s
         */
        JSONError("JSON error: %s", Category.CODE_ERROR, 9),
        /**
         * Invalid parameter "%1$s": %2$s
         */
        InvalidParameterValue("Invalid parameter \"%1$s\": %2$s", Category.CODE_ERROR, 10),
        /**
         * Unexpected error: %1$s
         */
        UnexpectedError("Unexpected error: %1$s", Category.CODE_ERROR, 11),
        /**
         * A parameter conflict occurred.
         */
        ParameterConflict("A parameter conflict occurred.", Category.CODE_ERROR, 12),
        /**
         * Parameter "%1$s" conflicts with parameter "%2$s".
         */
        EitherParameterConflict("Parameter \"%1$s\" conflicts with parameter \"%2$s\".", Category.CODE_ERROR, 13),
        /**
         * Action "%1$s" on request path "%2$s" is not permitted via a non-secure connection.
         */
        NonSecureDenied("Action \"%1$s\" on request path \"%2$s\" is not permitted via a non-secure connection.", Category.CODE_ERROR, 14);

        /**
         * Message of the exception.
         */
        private final String message;

        /**
         * Category of the exception.
         */
        private final Category category;

        /**
         * Detail number of the exception.
         */
        private final int number;

        /**
         * Default constructor.
         * 
         * @param message message.
         * @param category category.
         * @param detailNumber detail number.
         */
        private Code(final String message, final Category category, final int detailNumber) {
            this.message = message;
            this.category = category;
            number = detailNumber;
        }

        public Category getCategory() {
            return category;
        }

        public String getMessage() {
            return message;
        }

        public int getNumber() {
            return number;
        }
    }
}
