/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2010 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.admin.rmi.impl;

import java.io.BufferedReader;
import java.io.File;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.UnsupportedEncodingException;
import java.security.NoSuchAlgorithmException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashSet;
import org.apache.commons.io.FileUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.osgi.framework.Bundle;
import org.osgi.framework.BundleContext;
import org.osgi.framework.ServiceReference;
import com.openexchange.admin.daemons.AdminDaemon;
import com.openexchange.admin.daemons.ClientAdminThread;
import com.openexchange.admin.plugins.OXUserPluginInterface;
import com.openexchange.admin.plugins.PluginException;
import com.openexchange.admin.properties.AdminProperties;
import com.openexchange.admin.rmi.OXUserInterface;
import com.openexchange.admin.rmi.dataobjects.Context;
import com.openexchange.admin.rmi.dataobjects.Credentials;
import com.openexchange.admin.rmi.dataobjects.User;
import com.openexchange.admin.rmi.dataobjects.UserModuleAccess;
import com.openexchange.admin.rmi.exceptions.DatabaseUpdateException;
import com.openexchange.admin.rmi.exceptions.EnforceableDataObjectException;
import com.openexchange.admin.rmi.exceptions.InvalidCredentialsException;
import com.openexchange.admin.rmi.exceptions.InvalidDataException;
import com.openexchange.admin.rmi.exceptions.NoSuchContextException;
import com.openexchange.admin.rmi.exceptions.NoSuchUserException;
import com.openexchange.admin.rmi.exceptions.StorageException;
import com.openexchange.admin.storage.interfaces.OXUserStorageInterface;
import com.openexchange.admin.tools.AdminCache;
import com.openexchange.admin.tools.GenericChecks;
import com.openexchange.admin.tools.PropertyHandler;
import com.openexchange.admin.tools.SHACrypt;
import com.openexchange.admin.tools.UnixCrypt;
import com.openexchange.caching.Cache;
import com.openexchange.caching.CacheException;
import com.openexchange.caching.CacheKey;
import com.openexchange.caching.CacheService;
import com.openexchange.groupware.contexts.impl.ContextImpl;
import com.openexchange.groupware.userconfiguration.UserConfigurationException;
import com.openexchange.groupware.userconfiguration.UserConfigurationStorage;

/**
 * @author d7
 * @author cutmasta
 */
public class OXUser extends OXCommonImpl implements OXUserInterface {

    private final static Log log = LogFactory.getLog(OXUser.class);

    private static final String SYMBOLIC_NAME_CACHE = "com.openexchange.caching";

    private static final String NAME_OXCACHE = "oxcache";

    private final OXUserStorageInterface oxu;

    private final BasicAuthenticator basicauth;

    private final AdminCache cache;
    private final PropertyHandler prop;

    private BundleContext context = null;

    public OXUser(final BundleContext context) throws StorageException {
        super();
        this.context = context;
        this.cache = ClientAdminThread.cache;
        this.prop = this.cache.getProperties();
        if (log.isInfoEnabled()) {
            log.info("Class loaded: " + this.getClass().getName());
        }
        basicauth = new BasicAuthenticator();
        try {
            oxu = OXUserStorageInterface.getInstance();
        } catch (final StorageException e) {
            log.error(e.getMessage(), e);
            throw e;
        }
    }

    private boolean usernameIsChangeable(){
        return this.cache.getProperties().getUserProp(AdminProperties.User.USERNAME_CHANGEABLE, false);
    }

    public void change(final Context ctx, final User usrdata, final Credentials auth) throws StorageException, InvalidCredentialsException, NoSuchContextException,InvalidDataException, DatabaseUpdateException, NoSuchUserException {
        try {
            doNullCheck(usrdata);
        } catch (final InvalidDataException e2) {
            final InvalidDataException invalidDataException = new InvalidDataException("One of the given arguments for change is null");
            log.error(invalidDataException.getMessage(), invalidDataException);
            throw invalidDataException;
        }

        // SPECIAL USER AUTH CHECK FOR THIS METHOD!
        // check if credentials are from oxadmin or from an user
        Integer userid = null;
        try {
            contextcheck(ctx);

            checkContextAndSchema(ctx);

            setIdOrGetIDFromNameAndIdObject(ctx, usrdata);
            usrdata.testMandatoryCreateFieldsNull();
            userid = usrdata.getId();

            if (!ClientAdminThread.cache.contextAuthenticationDisabled()) {
                final int auth_user_id = tool.getUserIDByUsername(ctx, auth.getLogin());
                // check if given user is admin
                if (tool.isContextAdmin(ctx, auth_user_id)) {
                    basicauth.doAuthentication(auth, ctx);
                } else {
                    basicauth.doUserAuthentication(auth, ctx);
                    // now check if user which authed has the same id as the user he
                    // wants to change,else fail,
                    // cause then he/she wants to change not his own data!
                    if (userid.intValue() != auth_user_id) {
                        throw new InvalidCredentialsException("Permission denied");
                    }
                }
            }

            if (log.isDebugEnabled()) {
                log.debug(ctx + " - " + usrdata + " - " + auth);
            }

            if (!tool.existsUser(ctx, userid.intValue())) {
                final NoSuchUserException noSuchUserException = new NoSuchUserException("No such user " + userid + " in context " + ctx.getId());
                log.error(noSuchUserException.getMessage(), noSuchUserException);
                throw noSuchUserException;
            }
            final User[] dbuser = oxu.getData(ctx, new User[] { usrdata });

            checkChangeUserData(ctx, usrdata, dbuser[0], this.prop);

        } catch (final InvalidDataException e1) {
            log.error(e1.getMessage(), e1);
            throw e1;
        } catch (final StorageException e1) {
            log.error(e1.getMessage(), e1);
            throw e1;
        }

        final boolean isContextAdmin = tool.isContextAdmin(ctx, userid.intValue());

        oxu.change(ctx, usrdata);

        final ArrayList<Bundle> bundles = AdminDaemon.getBundlelist();
        for (final Bundle bundle : bundles) {
            final String bundlename = bundle.getSymbolicName();
            if (Bundle.ACTIVE==bundle.getState()) {
                final ServiceReference[] servicereferences = bundle.getRegisteredServices();
                if (null != servicereferences) {
                    for (final ServiceReference servicereference : servicereferences) {
                        final Object property = servicereference.getProperty("name");
                        if (null != property && property.toString().equalsIgnoreCase("oxuser")) {
                            final OXUserPluginInterface oxuser = (OXUserPluginInterface) this.context.getService(servicereference);
                            if (oxuser.canHandleContextAdmin() || (!oxuser.canHandleContextAdmin() && !isContextAdmin)) {
                                try {
                                    if (log.isDebugEnabled()) {
                                        log.debug("Calling change for plugin: " + bundlename);
                                    }
                                    oxuser.change(ctx, usrdata, auth);
                                } catch (final PluginException e) {
                                    log.error("Error while calling change for plugin: " + bundlename, e);
                                    throw new StorageException(e);
                                }
                            }
                        }
                    }
                }
            }
        }
        // change cached admin credentials if neccessary
        if (isContextAdmin && usrdata.getPassword() != null) {
            final Credentials cauth = ClientAdminThread.cache.getAdminCredentials(ctx);
            final String mech = ClientAdminThread.cache.getAdminAuthMech(ctx);
            if ("{CRYPT}".equals(mech)) {
                try {
                    cauth.setPassword(UnixCrypt.crypt(usrdata.getPassword()));
                } catch (final UnsupportedEncodingException e) {
                    log.error("Error encrypting password for credential cache ", e);
                    throw new StorageException(e);
                }
            } else if ("{SHA}".equals(mech)) {
                try {
                    cauth.setPassword(SHACrypt.makeSHAPasswd(usrdata.getPassword()));
                } catch (final NoSuchAlgorithmException e) {
                    log.error("Error encrypting password for credential cache ", e);
                    throw new StorageException(e);
                } catch (final UnsupportedEncodingException e) {
                    log.error("Error encrypting password for credential cache ", e);
                    throw new StorageException(e);
                }
            }
            ClientAdminThread.cache.setAdminCredentials(ctx,mech,cauth);
        }
        final CacheService cacheService = AdminDaemon.getService(SYMBOLIC_NAME_CACHE, NAME_OXCACHE, context,
				CacheService.class);
		if (null != cacheService) {
	        try {
	        	final CacheKey key = cacheService.newCacheKey(ctx.getId().intValue(), userid);
	        	Cache cache = cacheService.getCache("User");
	        	cache.remove(key);
	        	
	        	cache = cacheService.getCache("UserConfiguration");
	        	cache.remove(key);
	        	
	        	cache = cacheService.getCache("UserSettingMail");
	        	cache.remove(key);
	        } catch (final CacheException e) {
	            log.error(e.getMessage(), e);
	        } finally {
	        	AdminDaemon.ungetService(SYMBOLIC_NAME_CACHE, NAME_OXCACHE, context);
	        }
        }

    }

    public void changeModuleAccess(final Context ctx, final User user, final UserModuleAccess moduleAccess, final Credentials auth) throws StorageException, InvalidCredentialsException, NoSuchContextException,InvalidDataException, DatabaseUpdateException, NoSuchUserException {
        try {
            doNullCheck(user,moduleAccess);
        } catch (final InvalidDataException e1) {
            final InvalidDataException invalidDataException = new InvalidDataException("User or UserModuleAccess is null");
            log.error(invalidDataException.getMessage(), invalidDataException);
            throw invalidDataException;
        }

        if (log.isDebugEnabled()) {
            log.debug(ctx + " - " + user + " - "+ moduleAccess + " - " + auth);
        }

        try {
            basicauth.doAuthentication(auth, ctx);
            checkContextAndSchema(ctx);
            setIdOrGetIDFromNameAndIdObject(ctx, user);
            final int user_id = user.getId().intValue();
            if (!tool.existsUser(ctx, user_id)) {
                throw new NoSuchUserException("No such user " + user_id + " in context " + ctx.getId());
            }
            oxu.changeModuleAccess(ctx, user_id, moduleAccess);
        } catch (final StorageException e) {
            log.error(e.getMessage(), e);
            throw e;
        } catch (final InvalidDataException e) {
            log.error(e.getMessage(), e);
            throw e;
        } catch (final InvalidCredentialsException e) {
            log.error(e.getMessage(), e);
            throw e;
        } catch (final DatabaseUpdateException e) {
            log.error(e.getMessage(), e);
            throw e;
        } catch (final NoSuchContextException e) {
            log.error(e.getMessage(), e);
            throw e;
        } catch (final NoSuchUserException e) {
            log.error(e.getMessage(), e);
            throw e;
        }

//      JCS
        try {
            UserConfigurationStorage.getInstance().removeUserConfiguration(user.getId().intValue(), new ContextImpl(ctx.getId().intValue()));
        } catch (final UserConfigurationException e) {
            log.error("Error removing user "+user.getId()+" in context "+ctx.getId()+" from configuration storage",e);
        }
        // END OF JCS
    }

    public void changeModuleAccess(final Context ctx, final User user,final String access_combination_name, final Credentials auth)
			throws StorageException,InvalidCredentialsException, NoSuchContextException,InvalidDataException, DatabaseUpdateException, NoSuchUserException {
		
    	try {
            doNullCheck(user,access_combination_name);
            if (access_combination_name.trim().length() == 0) {
				throw new InvalidDataException("Invalid access combination name");
			}
        } catch (final InvalidDataException e1) {
            final InvalidDataException invalidDataException = new InvalidDataException("User or UserModuleAccess is null");
            log.error(invalidDataException.getMessage(), invalidDataException);
            throw invalidDataException;
        }

        if (log.isDebugEnabled()) {
            log.debug(ctx + " - " + user + " - " + access_combination_name
                + " - " + auth);
        }

        try {
            basicauth.doAuthentication(auth, ctx);
            checkContextAndSchema(ctx);
            setIdOrGetIDFromNameAndIdObject(ctx, user);
            final int user_id = user.getId().intValue();
            if (!tool.existsUser(ctx, user_id)) {
                throw new NoSuchUserException("No such user " + user_id + " in context " + ctx.getId());
            }

            final UserModuleAccess access = cache.getNamedAccessCombination(access_combination_name.trim());
            if(access==null){
            	// no such access combination name defined in configuration
            	// throw error!
            	throw new InvalidDataException("No such access combination name \""+access_combination_name.trim()+"\"");
            }
            oxu.changeModuleAccess(ctx, user_id, access);
        } catch (final StorageException e) {
            log.error(e.getMessage(), e);
            throw e;
        } catch (final InvalidDataException e) {
            log.error(e.getMessage(), e);
            throw e;
        } catch (final InvalidCredentialsException e) {
            log.error(e.getMessage(), e);
            throw e;
        } catch (final DatabaseUpdateException e) {
            log.error(e.getMessage(), e);
            throw e;
        } catch (final NoSuchContextException e) {
            log.error(e.getMessage(), e);
            throw e;
        } catch (final NoSuchUserException e) {
            log.error(e.getMessage(), e);
            throw e;
        }

//      JCS
        try {
            UserConfigurationStorage.getInstance().removeUserConfiguration(user.getId().intValue(), new ContextImpl(ctx.getId().intValue()));
        } catch (final UserConfigurationException e) {
            log.error("Error removing user "+user.getId()+" in context "+ctx.getId()+" from configuration storage",e);
        }
        // END OF JCS
    	
    	
    	
    	
    	
		
	}

    public User create(Context ctx, User usr, UserModuleAccess access, Credentials auth) throws StorageException, InvalidCredentialsException, NoSuchContextException, InvalidDataException, DatabaseUpdateException {
        // Call common create method directly because we already have out access module
    	return createUserCommon(ctx, usr, access, auth);
    }

    public User create(Context ctx, User usrdata, String access_combination_name, Credentials auth) throws StorageException,InvalidCredentialsException, NoSuchContextException,InvalidDataException, DatabaseUpdateException {

    	/*
		 * Resolve the access rights by the specified combination name. If
		 * combination name does not exists, throw error as it is described in
		 * the spec!
		 */

    	
    	try {
			doNullCheck(usrdata, access_combination_name);
			if (access_combination_name.trim().length() == 0) {
				throw new InvalidDataException("Invalid access combination name");
			}
		} catch (final InvalidDataException e3) {
			log.error("One of the given arguments for create is null", e3);
			throw e3;
		}

		if (log.isDebugEnabled()) {
			log.debug(ctx + " - " + usrdata + " - " + access_combination_name
			    + " - " + auth);
		}

		basicauth.doAuthentication(auth, ctx);

		
        final UserModuleAccess access = cache.getNamedAccessCombination(access_combination_name.trim());
        if(access==null){
        	// no such access combination name defined in configuration
        	// throw error!
        	throw new InvalidDataException("No such access combination name \""+access_combination_name.trim()+"\"");
        }

        // Call main create user method with resolved access rights
    	return createUserCommon(ctx, usrdata, access, auth);
    }


    public User create(Context ctx, User usrdata, Credentials auth)	throws StorageException,InvalidCredentialsException, NoSuchContextException,InvalidDataException, DatabaseUpdateException {

		/*
		 * Resolve current access rights from the specified context (admin) as
		 * it is described in the spec and then call the main create user method
		 * with the access rights!
		 */
    	
    	try {
			doNullCheck(usrdata);			
		} catch (final InvalidDataException e3) {
			log.error("One of the given arguments for create is null", e3);
			throw e3;
		}

		if (log.isDebugEnabled()) {
			log.debug(ctx + " - " + usrdata + " - "+ auth);
		}

		basicauth.doAuthentication(auth, ctx);
		
		/*
		 * Resolve admin user of specified context via tools and then get his current module access rights
		 */		
		
		final int admin_id = tool.getAdminForContext(ctx);
		final UserModuleAccess access = oxu.getModuleAccess(ctx, admin_id);
		
    	return createUserCommon(ctx, usrdata, access, auth);
    }

    /*
     * Main method to create a user. Which all inner create methods MUST use after resolving the access rights!
     */
    private User createUserCommon(Context ctx, User usr, UserModuleAccess access, Credentials auth) throws StorageException, InvalidCredentialsException, NoSuchContextException, InvalidDataException, DatabaseUpdateException {
    	
    	try {
            doNullCheck(usr,access);
        } catch (final InvalidDataException e3) {
            log.error("One of the given arguments for create is null", e3);
            throw e3;
        }

        if (log.isDebugEnabled()) {
            log.debug(ctx + " - " + usr + " - " + access + " - " + auth);
        }

        try {
            basicauth.doAuthentication(auth,ctx);

            checkContextAndSchema(ctx);

            tool.checkCreateUserData(ctx, usr);

            if (tool.existsUserName(ctx, usr.getName())) {
                throw new InvalidDataException("User " + usr.getName() + " already exists in this context");
            }

            // validate email adresss
            tool.primaryMailExists(ctx, usr.getPrimaryEmail());
        } catch (final InvalidDataException e2) {
            log.error(e2.getMessage(), e2);
            throw e2;
        } catch (final EnforceableDataObjectException e) {
            log.error(e.getMessage(), e);
            throw new InvalidDataException(e.getMessage());
        }

        final int retval = oxu.create(ctx, usr, access);
        usr.setId(Integer.valueOf(retval));
        final ArrayList<OXUserPluginInterface> interfacelist = new ArrayList<OXUserPluginInterface>();

        // homedirectory
        final String homedir = this.prop.getUserProp(AdminProperties.User.HOME_DIR_ROOT, "/home") + "/" + usr.getName();
        if (this.prop.getUserProp(AdminProperties.User.CREATE_HOMEDIRECTORY, false) && !tool.isContextAdmin(ctx, usr.getId().intValue())) {
            if (!new File(homedir).mkdir()) {
                log.error("unable to create directory: " + homedir);
            }
            final String CHOWN = "/bin/chown";
            final Process p;
            try {
                p = Runtime.getRuntime().exec(new String[] { CHOWN, usr.getName() + ":", homedir });
                p.waitFor();
                if (p.exitValue() != 0) {
                    log.error(CHOWN + " exited abnormally");
                    final BufferedReader prerr = new BufferedReader(new InputStreamReader(p.getErrorStream()));
                    String line = null;
                    while ((line = prerr.readLine()) != null) {
                        log.error(line);
                    }
                    log.error("Unable to chown homedirectory: " + homedir);
                }
            } catch (final IOException e) {
                log.error("Unable to chown homedirectory: " + homedir, e);
            } catch (final InterruptedException e) {
                log.error("Unable to chown homedirectory: " + homedir, e);
            }
        }

        final ArrayList<Bundle> bundles = AdminDaemon.getBundlelist();
        for (final Bundle bundle : bundles) {
            final String bundlename = bundle.getSymbolicName();
            if (Bundle.ACTIVE==bundle.getState()) {
                final ServiceReference[] servicereferences = bundle.getRegisteredServices();
                if (null != servicereferences) {
                    for (final ServiceReference servicereference : servicereferences) {
                        final Object property = servicereference.getProperty("name");
                        if (null != property && property.toString().equalsIgnoreCase("oxuser")) {
                            final OXUserPluginInterface oxuser = (OXUserPluginInterface) this.context.getService(servicereference);

                            if (oxuser.canHandleContextAdmin() || (!oxuser.canHandleContextAdmin() && !tool.isContextAdmin(ctx, usr.getId().intValue()))) {
                                try {
                                    if (log.isDebugEnabled()) {
                                        log.debug("Calling create for plugin: " + bundlename);
                                    }
                                    oxuser.create(ctx, usr, access, auth);
                                    interfacelist.add(oxuser);
                                } catch (final PluginException e) {
                                    log.error("Error while calling create for plugin: " + bundlename, e);
                                    log.info("Now doing rollback for everything until now...");
                                    for (final OXUserPluginInterface oxuserinterface : interfacelist) {
                                        try {
                                            oxuserinterface.delete(ctx, new User[] { usr }, auth);
                                        } catch (final PluginException e1) {
                                            log.error("Error doing rollback for plugin: " + bundlename, e1);
                                        }
                                    }
                                    try {
                                        oxu.delete(ctx, usr);
                                    } catch (final StorageException e1) {
                                        log.error("Error doing rollback for creating user in database", e1);
                                    }
                                    throw new StorageException(e);
                                }
                            }
                        }
                    }
                }
            }
        }
        return usr;
    }

    public void delete(Context ctx, User user, Credentials auth) throws StorageException, InvalidCredentialsException, NoSuchContextException, InvalidDataException, DatabaseUpdateException, NoSuchUserException {
        delete(ctx, new User[]{user}, auth);
    }

    public void delete(Context ctx, User[] users, Credentials auth) throws StorageException, InvalidCredentialsException, NoSuchContextException,InvalidDataException, DatabaseUpdateException, NoSuchUserException {
        try {
            doNullCheck((Object[])users);
        } catch (final InvalidDataException e1) {
            log.error("One of the given arguments for delete is null", e1);
            throw e1;
        }

        if (users.length == 0) {
            final InvalidDataException e = new InvalidDataException("User array is empty");
            log.error(e.getMessage(), e);
            throw e;
        }

        basicauth.doAuthentication(auth,ctx);

        if (log.isDebugEnabled()) {
            log.debug(ctx + " - " + Arrays.toString(users) + " - " + auth);
        }
        checkContextAndSchema(ctx);

        try {
            setUserIdInArrayOfUsers(ctx, users);
            // FIXME: Change function from int to user object
            if (!tool.existsUser(ctx, users)) {
                final NoSuchUserException noSuchUserException = new NoSuchUserException("No such user(s) " + getUserIdArrayFromUsersAsString(users) + " in context " + ctx.getId());
                log.error("No such user(s) " + Arrays.toString(users) + " in context " + ctx.getId(), noSuchUserException);
                throw noSuchUserException;
            }
            for (final User user : users) {
                if (tool.isContextAdmin(ctx, user.getId().intValue())) {
                    throw new InvalidDataException("Admin delete not supported");
                }
            }
        } catch (final InvalidDataException e1) {
            log.error(e1.getMessage(), e1);
            throw e1;
        } catch (final StorageException e1) {
            log.error(e1.getMessage(), e1);
            throw e1;
        }


        User[] retusers = oxu.getData(ctx, users);

        final ArrayList<OXUserPluginInterface> interfacelist = new ArrayList<OXUserPluginInterface>();

        // Here we define a list which takes all exceptions which occur during plugin-processing
        // By this we are able to throw all exceptions to the client while concurrently processing all plugins
        final ArrayList<Exception> exceptionlist = new ArrayList<Exception>();

        final ArrayList<Bundle> bundles = AdminDaemon.getBundlelist();
        final ArrayList<Bundle> revbundles = new ArrayList<Bundle>();
        for (int i = bundles.size() - 1; i >= 0; i--) {
            revbundles.add(bundles.get(i));
        }
        for (final Bundle bundle : revbundles) {
            final String bundlename = bundle.getSymbolicName();
            if (Bundle.ACTIVE==bundle.getState()) {
                final ServiceReference[] servicereferences = bundle.getRegisteredServices();
                if (null != servicereferences) {
                    for (final ServiceReference servicereference : servicereferences) {
                        final Object property = servicereference.getProperty("name");
                        if (null != property && property.toString().equalsIgnoreCase("oxuser")) {
                            final OXUserPluginInterface oxuser = (OXUserPluginInterface) this.context.getService(servicereference);
                            if (!oxuser.canHandleContextAdmin()) {
                                retusers = removeContextAdmin(ctx, retusers);
                                if (retusers.length > 0) {
                                    if (log.isDebugEnabled()) {
                                        log.debug("Calling delete for plugin: " + bundlename);
                                    }
                                    final Exception exception = callDeleteForPlugin(ctx, auth, retusers, interfacelist, bundlename, oxuser);
                                    if (null != exception) {
                                        exceptionlist.add(exception);
                                    }
                                }
                            } else {
                                if (log.isDebugEnabled()) {
                                    log.debug("Calling delete for plugin: " + bundlename);
                                }
                                final Exception exception = callDeleteForPlugin(ctx, auth, retusers, interfacelist, bundlename, oxuser);
                                if (null != exception) {
                                    exceptionlist.add(exception);
                                }
                            }
                        }
                    }
                }
            }
        }

        if (this.prop.getUserProp(AdminProperties.User.CREATE_HOMEDIRECTORY, false)) {
            for(final User usr : users) {
                // homedirectory
                String homedir = this.prop.getUserProp(AdminProperties.User.HOME_DIR_ROOT, "/home");
                homedir += "/" + usr.getName();
                // FIXME: if(! tool.isContextAdmin(ctx, usr.getId()) ) {} ??
                try {
                    FileUtils.deleteDirectory(new File(homedir));
                } catch (final IOException e) {
                    log.error("Could not delete homedir for user: " + usr);
                }
            }
        }

        oxu.delete(ctx, users);


        // JCS
        final CacheService cacheService = AdminDaemon.getService(SYMBOLIC_NAME_CACHE, NAME_OXCACHE, context,
				CacheService.class);
		if (null != cacheService) {
			try {
				final Cache usercCache = cacheService.getCache("User");
				final Cache ucCache = cacheService.getCache("UserConfiguration");
				final Cache usmCache = cacheService.getCache("UserSettingMail");
				for (final User user : users) {
					final CacheKey key = cacheService.newCacheKey(ctx.getId().intValue(), user.getId());
					usercCache.remove(key);
					ucCache.remove(key);
					usmCache.remove(key);
					try {
						UserConfigurationStorage.getInstance().removeUserConfiguration(user.getId().intValue(),
								new ContextImpl(ctx.getId().intValue()));
					} catch (final UserConfigurationException e) {
						log.error("Error removing user " + user.getId() + " in context " + ctx.getId()
								+ " from configuration storage", e);
					}
				}
			} catch (final CacheException e) {
				log.error(e.getMessage(), e);
			} finally {
				AdminDaemon.ungetService(SYMBOLIC_NAME_CACHE, NAME_OXCACHE, context);
			}
		}
        // END OF JCS

        if (!exceptionlist.isEmpty()) {
            final StringBuilder sb = new StringBuilder("The following exceptions occured in the plugins: ");
            for (final Exception e : exceptionlist) {
                sb.append(e.toString());
                sb.append('\n');
            }
            throw new StorageException(sb.toString());
        }
    }

    public User getData(Context ctx, User user, Credentials auth) throws StorageException, InvalidCredentialsException, NoSuchContextException, InvalidDataException, NoSuchUserException, DatabaseUpdateException {
        return getData(ctx, new User[]{user}, auth)[0];
    }

    public User[] getData(Context ctx, User[] users, Credentials auth) throws StorageException, InvalidCredentialsException, NoSuchContextException, InvalidDataException, NoSuchUserException, DatabaseUpdateException {
        try {
            doNullCheck((Object[])users);
        } catch (final InvalidDataException e1) {
            log.error("One of the given arguments for getData is null", e1);
            throw e1;
        }

        try {
            checkContext(ctx);

            if (users.length <= 0) {
                throw new InvalidDataException();
            }
        } catch (final InvalidDataException e) {
            log.error(e.getMessage(), e);
            throw e;
        }

        if (log.isDebugEnabled()) {
            log.debug(ctx + " - " + Arrays.toString(users) + " - " + auth);
        }

        try {
            // enable check who wants to get data if authentcaition is enabled
            if (!cache.contextAuthenticationDisabled()) {
                // ok here its possible that a user wants to get his own data
                // SPECIAL USER AUTH CHECK FOR THIS METHOD!
                // check if credentials are from oxadmin or from an user
                // check if given user is not admin, if he is admin, the
                final User authuser = new User();
                authuser.setName(auth.getLogin());
                if (!tool.isContextAdmin(ctx, authuser)) {
                    final InvalidCredentialsException invalidCredentialsException = new InvalidCredentialsException("Permission denied");
                    if (users.length == 1) {
                        final int auth_user_id = authuser.getId().intValue();
                        basicauth.doUserAuthentication(auth, ctx);
                        // its possible that he wants his own data
                        final Integer userid = users[0].getId();
                        if (userid != null) {
                            if (userid.intValue() != auth_user_id) {
                                throw invalidCredentialsException;
                            }
                        } else {
                            // id not set, try to resolv id by username and then
                            // check
                            // again
                            final String username = users[0].getName();
                            if (username != null) {
                                final int check_user_id = tool.getUserIDByUsername(ctx, username);
                                if (check_user_id != auth_user_id) {
                                    log.debug("user[0].getId() does not match id from Credentials.getLogin()");
                                    throw invalidCredentialsException;
                                }
                            } else {
                                log.debug("Cannot resolv user[0]`s internal id because the username is not set!");
                                throw new InvalidDataException("Username and userid missing.");
                            }
                        }
                    } else {
                        log.error("User sent " + users.length + " users to get data for. Only context admin is allowed to do that", invalidCredentialsException);
                        throw invalidCredentialsException;
                        // one user cannot edit more than his own data
                    }
                } else {
                    basicauth.doAuthentication(auth, ctx);
                }
            } else {
                basicauth.doAuthentication(auth, ctx);
            }

            checkContextAndSchema(ctx);

            for (final User usr : users) {
                final String username = usr.getName();
                final Integer userid = usr.getId();
                if (userid != null && !tool.existsUser(ctx, userid.intValue())) {
                    throw new NoSuchUserException("No such user "+userid+" in context "+ctx.getId());
                }
                if (username != null && !tool.existsUserName(ctx, username)) {
                    throw new NoSuchUserException("No such user " + username+" in context "+ctx.getId());
                }
                if (username == null && userid == null) {
                    throw new InvalidDataException("Username and userid missing.");
                }
				// ok , try to get the username by id or username
				if (username == null) {
				    usr.setName(tool.getUsernameByUserID(ctx, userid.intValue()));
				}

				if (userid == null) {
				    usr.setId(new Integer(tool.getUserIDByUsername(ctx, username)));
				}
            }
        } catch (final InvalidDataException e) {
            log.error(e.getMessage(), e);
            throw(e);
        } catch (final InvalidCredentialsException e) {
            log.error(e.getMessage(), e);
            throw(e);
        }

        User[] retusers = oxu.getData(ctx, users);

        final ArrayList<Bundle> bundles = AdminDaemon.getBundlelist();
        for (final Bundle bundle : bundles) {
            final String bundlename = bundle.getSymbolicName();
            if (Bundle.ACTIVE==bundle.getState()) {
                final ServiceReference[] servicereferences = bundle.getRegisteredServices();
                if (null != servicereferences) {
                    for (final ServiceReference servicereference : servicereferences) {
                        final Object property = servicereference.getProperty("name");
                        if (null != property && property.toString().equalsIgnoreCase("oxuser")) {
                            final OXUserPluginInterface oxuserplugin = (OXUserPluginInterface) this.context.getService(servicereference);
                            //TODO: Implement check for contextadmin here
                            if (log.isDebugEnabled()) {
                                log.debug("Calling getData for plugin: " + bundlename);
                            }
                            retusers = oxuserplugin.getData(ctx, retusers, auth);
                        }
                    }
                }
            }
        }
        log.debug(Arrays.toString(retusers));
        return retusers;
    }

    public UserModuleAccess getModuleAccess(Context ctx, User user, Credentials auth) throws StorageException, InvalidCredentialsException, NoSuchContextException,InvalidDataException, DatabaseUpdateException, NoSuchUserException {
        try {
            doNullCheck(user);
        } catch (final InvalidDataException e) {
            final InvalidDataException invalidDataException = new InvalidDataException("User object is null");
            log.error(invalidDataException.getMessage(), invalidDataException);
            throw invalidDataException;
        }
        if (log.isDebugEnabled()) {
            log.debug(ctx + " - " + user + " - " + auth);
        }
        try {
            basicauth.doAuthentication(auth, ctx);
            checkContextAndSchema(ctx);
            setIdOrGetIDFromNameAndIdObject(ctx, user);
            final int user_id = user.getId().intValue();
            if (!tool.existsUser(ctx, user_id)) {
                throw new NoSuchUserException("No such user " + user_id + " in context " + ctx.getId());
            }
            return oxu.getModuleAccess(ctx, user_id);
        } catch (final StorageException e) {
            log.error(e.getMessage(), e);
            throw e;
        } catch (final InvalidDataException e) {
            log.error(e.getMessage(), e);
            throw e;
        } catch (final InvalidCredentialsException e) {
            log.error(e.getMessage(), e);
            throw e;
        } catch (final DatabaseUpdateException e) {
            log.error(e.getMessage(), e);
            throw e;
        } catch (final NoSuchContextException e) {
            log.error(e.getMessage(), e);
            throw e;
        } catch (final NoSuchUserException e) {
            log.error(e.getMessage(), e);
            throw e;
        }
    }

    public String getAccessCombinationName(Context ctx, User user, Credentials auth) throws StorageException, InvalidCredentialsException, NoSuchContextException, InvalidDataException, DatabaseUpdateException, NoSuchUserException {
    	
    	try {
            doNullCheck(user);
        } catch (final InvalidDataException e) {
            final InvalidDataException invalidDataException = new InvalidDataException("User object is null");
            log.error(invalidDataException.getMessage(), invalidDataException);
            throw invalidDataException;
        }
        if (log.isDebugEnabled()) {
            log.debug(ctx + " - " + user + " - " + auth);
        }
        try {
            basicauth.doAuthentication(auth, ctx);
            checkContextAndSchema(ctx);
            setIdOrGetIDFromNameAndIdObject(ctx, user);
            final int user_id = user.getId().intValue();

            if (!tool.existsUser(ctx, user_id)) {
                throw new NoSuchUserException("No such user " + user_id + " in context " + ctx.getId());
            }

            return cache.getNameForAccessCombination(oxu.getModuleAccess(ctx, user_id));
        } catch (final StorageException e) {
            log.error(e.getMessage(), e);
            throw e;
        } catch (final InvalidDataException e) {
            log.error(e.getMessage(), e);
            throw e;
        } catch (final InvalidCredentialsException e) {
            log.error(e.getMessage(), e);
            throw e;
        } catch (final DatabaseUpdateException e) {
            log.error(e.getMessage(), e);
            throw e;
        } catch (final NoSuchContextException e) {
            log.error(e.getMessage(), e);
            throw e;
        } catch (final NoSuchUserException e) {
            log.error(e.getMessage(), e);
            throw e;
        }
    	
		
	}


	



    public boolean isContextAdmin(Context ctx, User user, Credentials auth) throws StorageException, InvalidCredentialsException, NoSuchContextException, InvalidDataException, NoSuchUserException, DatabaseUpdateException {
        try {
            doNullCheck(user);
        } catch (final InvalidDataException e1) {
            log.error("One of the given arguments is null", e1);
            throw e1;
        }

        basicauth.doUserAuthentication(auth,ctx);

        checkContextAndSchema(ctx);

        if (user.getId()!=null && !tool.existsUser(ctx, user.getId().intValue())) {
            final NoSuchUserException noSuchUserException = new NoSuchUserException("No such user " + user.getId().intValue() + " in context " + ctx.getId());
            log.error(noSuchUserException.getMessage(), noSuchUserException);
            throw noSuchUserException;
        }

        return tool.isContextAdmin(ctx, user.getId().intValue());
    }

    public User[] list(Context ctx, String search_pattern, Credentials auth) throws StorageException, InvalidCredentialsException, NoSuchContextException, InvalidDataException, DatabaseUpdateException {
        try {
            doNullCheck(ctx,search_pattern);
        } catch (final InvalidDataException e1) {
            log.error("One of the given arguments for list is null", e1);
            throw e1;
        }

        if (log.isDebugEnabled()) {
            log.debug(ctx + " - " + auth);
        }

        basicauth.doAuthentication(auth,ctx);

        checkContextAndSchema(ctx);

        final User[] retval =  oxu.list(ctx, search_pattern);

        return retval;
    }

    public User[] listAll(Context ctx, Credentials auth) throws StorageException, InvalidCredentialsException, NoSuchContextException, InvalidDataException, DatabaseUpdateException {
        return list(ctx, "*", auth);
    }

    private Exception callDeleteForPlugin(final Context ctx, final Credentials auth, final User[] retusers, final ArrayList<OXUserPluginInterface> interfacelist, final String bundlename, final OXUserPluginInterface oxuser) {
        try {
            if (log.isDebugEnabled()) {
                log.debug("Calling delete for plugin: " + bundlename);
            }
            oxuser.delete(ctx, retusers, auth);
            interfacelist.add(oxuser);
            return null;
        } catch (final PluginException e) {
            log.error("Error while calling delete for plugin: " + bundlename, e);
            return e;
        }
    }

    /**
     * checking for some requirements when changing exisiting user data
     *
     * @param ctx
     * @param newuser
     * @param dbuser
     * @param prop
     * @throws StorageException
     * @throws InvalidDataException
     */
    private void checkChangeUserData(Context ctx, User newuser, User dbuser, PropertyHandler prop) throws StorageException, InvalidDataException {
        if (newuser.getName() != null) {
            if (usernameIsChangeable()) {
                if (prop.getUserProp(AdminProperties.User.CHECK_NOT_ALLOWED_CHARS, true)) {
                    tool.validateUserName(newuser.getName());
                }
                if (prop.getUserProp(AdminProperties.User.AUTO_LOWERCASE, true)) {
                    newuser.setName(newuser.getName().toLowerCase());
                }
            }
            // must be loaded additionally because the user loading method gets the new user name passed and therefore does not load the
            // current one.
            final String currentName = tool.getUsernameByUserID(ctx, newuser.getId().intValue());
            if (!newuser.getName().equals(currentName)) {
                if (usernameIsChangeable()) {
                    if (tool.existsUserName(ctx, newuser.getName())) {
                        throw new InvalidDataException("User " + newuser.getName() + " already exists in this context");
                    }
                } else {
                    throw new InvalidDataException("Changing username is disabled!");
                }
            }
        }

        final String lang = newuser.getLanguage();
        if (lang != null && !lang.contains("_")) {
            throw new InvalidDataException("Language must contain an underscore, e.g. en_US.");
        }

        if (prop.getUserProp(AdminProperties.User.PRIMARY_MAIL_UNCHANGEABLE, true)) {
            if (newuser.getPrimaryEmail() != null && !newuser.getPrimaryEmail().equals(dbuser.getPrimaryEmail())) {
                throw new InvalidDataException("primary mail must not be changed");
            }
        }

        GenericChecks.checkChangeValidPasswordMech(newuser);

        // if no password mech supplied, use the old one as set in db
        if (newuser.getPasswordMech() == null) {
            newuser.setPasswordMech(dbuser.getPasswordMech());
        }

        if (!tool.isContextAdmin(ctx, newuser.getId().intValue())) {
            // checks below throw InvalidDataException
            tool.checkValidEmailsInUserObject(newuser);
            HashSet<String> useraliases = newuser.getAliases();
            if (useraliases == null) {
                useraliases = dbuser.getAliases();
            }

            final String defaultSenderAddress = newuser.getDefaultSenderAddress();
            final String primaryEmail = newuser.getPrimaryEmail();
            final String email1 = newuser.getEmail1();
            if (primaryEmail != null && email1 != null && !primaryEmail.equals(email1)) {
                // primary mail value must be same with email1
                throw new InvalidDataException("email1 not equal with primarymail!");
            }

            String check_primary_mail;
            String check_email1;
            String check_default_sender_address;
            if (primaryEmail != null) {
                check_primary_mail = primaryEmail;
                if (!primaryEmail.equals(dbuser.getPrimaryEmail())) {
                    tool.primaryMailExists(ctx, primaryEmail);
                }
            } else {
                check_primary_mail = dbuser.getPrimaryEmail();
            }

            if (email1 != null) {
                check_email1 = email1;
            } else {
                check_email1 = dbuser.getEmail1();
            }
            if (defaultSenderAddress != null) {
                check_default_sender_address = defaultSenderAddress;
            } else {
                check_default_sender_address = dbuser.getDefaultSenderAddress();
            }

            final boolean found_primary_mail = useraliases.contains(check_primary_mail);
            final boolean found_email1 = useraliases.contains(check_email1);
            final boolean found_default_sender_address = useraliases.contains(check_default_sender_address);

            if (!found_primary_mail || !found_email1 || !found_default_sender_address) {
                throw new InvalidDataException("primaryMail, Email1 and defaultSenderAddress must be present in set of aliases.");
            }
            // added "usrdata.getPrimaryEmail() != null" for this check, else we cannot update user data without mail data
            // which is not very good when just changing the displayname for example
            if (primaryEmail != null && email1 == null) {
                throw new InvalidDataException("email1 not sent but required!");

            }
        }

        
        // TODO mail checks
    }

    private void checkContext(final Context ctx) throws InvalidDataException {
        if (null == ctx.getId()) {
            throw new InvalidDataException("Context invalid");
        }
    }

    private String getUserIdArrayFromUsersAsString(final User[] users) throws InvalidDataException {
		if (null == users) {
			return null;
		} else if (users.length == 0) {
			return "";
		}
		final StringBuilder sb = new StringBuilder(users.length * 8);
		{
			final Integer id = users[0].getId();
			if (null == id) {
				throw new InvalidDataException("One user object has no id");
			}
			sb.append(id);
		}
		for (int i = 1; i < users.length; i++) {
			final Integer id = users[i].getId();
			if (null == id) {
				throw new InvalidDataException("One user object has no id");
			}
			sb.append(',');
			sb.append(id);
		}
		return sb.toString();
	}

    private User[] removeContextAdmin(final Context ctx, final User[] retusers) throws StorageException {
        final ArrayList<User> list = new ArrayList<User>(retusers.length);
        for (final User user : retusers) {
            if (!tool.isContextAdmin(ctx, user.getId().intValue())) {
                list.add(user);
            }
        }
        return list.toArray(new User[list.size()]);
    }
}
