/**
 * 
 * other original works), unless otherwise noted, is licensed under a Creative
 * Commons License.
 * 
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 * 
 * Copyright (C) 2004-2009 Open-Xchange, Inc. Mail: info@open-xchange.com
 * 
 * @author Matthias Biggeleben <matthias.biggeleben@open-xchange.com>
 * @ignore
 */

var calendarGridInstances = {};
var CG_ROW_HEIGHT = 50; // in px
var CG_PX2EM = 12;
var CG_HEAD_ZOOM = 75; // in % (show hours in header if > x%)
var CG_HOUR_ZOOM = 50; // in % (show hours in grid if > x%)

function getCalendarGrid(id) {
    // no instance for current view?
    if (!calendarGridInstances[id]) {
        // create new instance
        calendarGridInstances[id] = new CalendarGrid();
    }
    // return instance
    return calendarGridInstances[id];
}

function getCalendarGridIfExists(id, callback) {
    var grid = calendarGridInstances[id];
    if (grid) {
        callback(grid);
    }
}

// TODO:
// register for event "something" that is triggered if calendar data is
// changed. Might become unnecessary if this whole thing works with the cache.
// However, CalendarGrid's property "appointments" must be cleared and the grid
// must be repainted [e.g., grid.update()]. This should get it done.

var calendarGridHandlers = {
        
    addTeamMember: function() {
        getCalendarGridIfExists("teamview", function(grid) {
            if (!grid.participantsDialog) {
                // wrapper
                var wrapper = function(list) {
                    // this list is as good as a team
                    grid.addTeam(list);
                };
                grid.participantsDialog = new ParticipantsSmall(
                    null, wrapper, true, true, true, 
                    false, "Select Members", /* i18n */
                    "grid", false, false, true
                );
            }
            grid.participantsDialog.openAddParticipantsWindow();
        });
    },
    
    removeTeamMember: function() {
        getCalendarGridIfExists("teamview", function(grid) {
            grid.removeSelectedRows();
            grid.clearPartial();
            grid.update();
        });
    },
    
    removeAllTeamMembers: function() {
        getCalendarGridIfExists("teamview", function(grid) {
            grid.removeAll();
            grid.clearPartial();
            grid.update();
        });
    },
    
    sortTeamMembers: function() {
        getCalendarGridIfExists("teamview", function(grid) {
            grid.sort();
            grid.clearPartial();
            grid.update();
        });
    },
    
    appointmentSelected: function() {
        getCalendarGridIfExists("teamview", function(grid) {
            // set (magical) global vars
            selectedAppointment = grid.appointmentSelection.getSelectedItems();
            lastSelectedAppointment = selectedAppointment;
            lastUpdateOfCalendarTimestamp = 0;
            // trigger global event
            triggerEvent("Selected", selectedAppointment);
        });
    },
    
    refresh: function() {
        getCalendarGridIfExists("teamview", function(grid) {
            // clear appointments
            grid.invalidateAppointments();
            // update grid
            grid.clearPartial();
            grid.update();
        });
    },

    update: function() {
        getCalendarGridIfExists("teamview", function(grid) {
            // update grid
            grid.clearPartial();
            grid.update();
        });
    },

    clearSelection: function() {
        getCalendarGridIfExists("teamview", function(grid) {
            // clear selection
            grid.appointmentSelection.clear();
        });
    },
    
    gotoToday: function () {
        getCalendarGridIfExists("teamview", function(grid) {
            var now = new Date();
            // update mini calendar
            activeYear = now.getUTCFullYear();
            activeMonth = now.getUTCMonth();
            activeDay = now.getUTCDate();
            if (oMiniCalendar) oMiniCalendar.setSelectedByDate(activeYear, activeMonth, activeDay);
            // update grid
            grid.setGridInterval(activeYear, activeMonth, activeDay);
            grid.clearPartial();
            grid.update();
        });
    },
    
    gotoDay: function(year, month, day) {
        getCalendarGridIfExists("teamview", function(grid) {
            // update mini calendar
            activeYear = year;
            activeMonth = month;
            activeDay = day;
            if (oMiniCalendar) oMiniCalendar.setSelectedByDate(activeYear, activeMonth, activeDay);
            // update grid
            grid.setGridInterval(year, month, day);
            grid.clearPartial();
            grid.update();
        });
    },
    
    setColorLabel: function(labelNumber) {
        lastUpdateOfCalendarTimestamp = (new Date()).getTime();
        calendarAddTag(labelNumber, true);
        getCalendarGridIfExists("teamview", function(grid) {
            grid.invalidateAppointments();
        });
    },
    
    configurationChanged: function() {
        getCalendarGridIfExists("teamview", function(grid) {
            // update team
            if (grid.currentTeam != null) {
                grid.loadTeamByIndex(grid.currentTeam);
            }
            // refresh
            calendarGridHandlers.refresh();
        });
    },
    
    registerAll: function() {
        register("OX_Calendar_Teammember_Add", calendarGridHandlers.addTeamMember);
        register("OX_Calendar_Teammember_Remove", calendarGridHandlers.removeTeamMember);
        register("OX_Calendar_Teammember_Sort", calendarGridHandlers.sortTeamMembers);
        register("OX_Calendar_Teammember_RemoveAll", calendarGridHandlers.removeAllTeamMembers);
        register("OX_Calendar_Appointment_Selected", calendarGridHandlers.appointmentSelected);
        register('LanguageChanged', calendarGridHandlers.update);
        register('OX_Refresh', calendarGridHandlers.refresh);
        register('OX_After_Delete_Appointment', calendarGridHandlers.refresh);
        register('OX_After_Update_Appointment', calendarGridHandlers.refresh);
        register('OX_After_New_Appointment', calendarGridHandlers.refresh);
        register('OX_After_MoveCopy_Appointment', calendarGridHandlers.clearSelection);
        register('OX_Calendar_Today', calendarGridHandlers.gotoToday);
        register('OX_Mini_Calendar_Date_Picked', calendarGridHandlers.gotoDay);
        register("OX_Add_Flag", calendarGridHandlers.setColorLabel);
    },
    
    registerOnce: function() {
        // this one is registered once (and stays registered) to support
        // changing the teams in the config view and getting informed about that
        register("OX_Configuration_Changed", calendarGridHandlers.configurationChanged);
    },
    
    unregisterAll: function() {
        // unregister
        unregister("OX_Calendar_Teammember_Add", calendarGridHandlers.addTeamMember);
        unregister("OX_Calendar_Teammember_Remove", calendarGridHandlers.removeTeamMember);
        unregister("OX_Calendar_Teammember_Sort", calendarGridHandlers.sortTeamMembers);
        unregister("OX_Calendar_Teammember_RemoveAll", calendarGridHandlers.removeAllTeamMembers);
        unregister("OX_Calendar_Appointment_Selected", calendarGridHandlers.appointmentSelected);
        unregister('LanguageChanged', calendarGridHandlers.update);
        unregister('OX_Refresh', calendarGridHandlers.refresh);
        unregister('OX_After_Delete_Appointment', calendarGridHandlers.refresh);
        unregister('OX_After_Update_Appointment', calendarGridHandlers.refresh);
        unregister('OX_After_New_Appointment', calendarGridHandlers.refresh);
        unregister('OX_After_MoveCopy_Appointment', calendarGridHandlers.clearSelection);
        unregister('OX_Calendar_Today', calendarGridHandlers.gotoToday);
        unregister('OX_Mini_Calendar_Date_Picked', calendarGridHandlers.gotoDay);
        unregister("OX_Add_Flag", calendarGridHandlers.setColorLabel);
        // clear current selection
        getCalendarGridIfExists("teamview", function(grid) {
            grid.invalidateAppointments();
        });        
    }
};

calendarGridHandlers.registerOnce();

function calendarGrid_deleteSelectedAppointments() {
    getCalendarGridIfExists("teamview", function(grid) {
        // get selection
        var apps = grid.appointmentSelection.getSelectedItems();
        // define delete routine // user will be asked first (see below)
        var deleteRoutine = function() {
            // 0. clear appointments
            grid.invalidateAppointments();
            // the tricky part here is that we cannot directly delete the appointment
            // but remove the particular team member from the participants list. Thus:
            // 1. we prepare the given appointment data for a server request
            var multiple = [];
            for (var i = 0; i < apps.length; i++) {
                var request = {
                        "action": "get",
                        "module": "calendar",
                        "folder": apps[i].folder,
                        "id": apps[i].id
                };
                // add recurrence_position?
                if (apps[i].recurrence_position) {
                    request.recurrence_position = apps[i].recurrence_position;
                }
                multiple.push(request);
            }
            // 2. we ask the server for appointments details, i.e. the participants
            json.put(AjaxRoot + "/multiple?session=" + session, multiple, null, function(response) {
                // 3. now loop through the response to consolidate appointments.
                // we need to do this because the selection might refer to identical appointments
                // and we want to avoid that the last update overrides the previous ones
                var hash = {};
                var deleteHash = {};
                var timestamp = 0;
                for (var r = 0; r < response.length; r++) {
                    var app = response[r].data;
                    // tidy up
                    grid.tidyUpAppointment(app);
                    // create composite id (just the id & position; not the folder!)
                    var compId = [app.id, app.recurrence_position].join(".");
                    // update timestamp
                    timestamp = Math.max(timestamp, response[r].timestamp);
                    // add to hash
                    hash[compId] = app;
                }
                // now, hash is free of duplicates
                // 4. we loop again through the selected appointments
                for (var i = 0; i < apps.length; i++) {
                    // split up the selection id; format: folder.app_id.app_pos @ user_id:type
                    var keys = apps[i].selectionId.match(/^\w+\.(\w+\.\w+)\@(\w+)\:(\w+)$/);
                    // get the appointment composite key, the participant id, and the type
                    var compId = keys[1];
                    var id = keys[2];
                    var type = keys[3];
                    // found one?
                    if (typeof(hash[compId]) != "undefined") {
                        // 5a. do we delete the owner?
                        if (id == app.created_by) {
                            deleteHash[compId] = hash[compId];
                        } else {
                            // 5b. we look for this participant by looping through the list
                            var app = hash[compId];
                            for (var p = 0; p < app.participants.length; p++) { // fifth element (see columns)
                                var part = app.participants[p];
                                // match?
                                if (part.type == type && part.id == id) {
                                    // 6. delete it
                                    app.participants.splice(p, 1);
                                    break;
                                }
                            }
                        }
                    }
                }
                // clean up "drop list" (these appointments do not require an update, since they will be deleted)
                for (var compId in deleteHash) {
                    delete hash[compId];
                }
                // 7. throw the new data back to the server (delete participants)
                var multipleDrop = [];
                var localTimestamp = timestamp;
                for (var compId in hash) {
                    var app = {
                            "module": "calendar",
                            "action": "update",
                            "folder": hash[compId].folder_id,
                            "id": hash[compId].id,
                            "data": {
                                "participants": hash[compId].participants
                            },
                            "timestamp": localTimestamp
                    };
                    // add an hour to the timestamp (workaround) to prevent update collision errors
                    localTimestamp += 1000 * 60 * 60;
                    // add recurrence_id?
                    if (hash[compId].recurrence_id)
                        app.data.recurrence_id = hash[compId].recurrence_id;
                    // add recurrence_position?
                    if (hash[compId].recurrence_position)
                        app.data.recurrence_position = hash[compId].recurrence_position;
                    multipleDrop.push(app);
                }
                // send first request
                json.put(AjaxRoot + "/multiple?session=" + session + "&continue=true", multipleDrop, null, function(response) {
                    // update timestamp (only available if appointments have been updated)
                    for (var i = 0; i < response.length; i++) {
                        timestamp = Math.max(timestamp, response[i].timestamp || 0);
                    }
                    // prepare 2nd request
                    var multipleDelete = [];
                    for (var compId in deleteHash) {
                        var app = {
                                "module": "calendar",
                                "action": "delete",
                                "data": {
                                    "folder": deleteHash[compId].folder_id,
                                    "id": deleteHash[compId].id,
                                    "pos": deleteHash[compId].recurrence_position
                                },
                                "timestamp": timestamp
                        };
                        multipleDelete.push(app);
                    }
                    // send 2nd request
                    json.put(AjaxRoot + "/multiple?session=" + session + "&continue=true", multipleDelete, null, function(response) {
                        // finally...
                        calendarGridHandlers.refresh();
                    });
                });
            });
        } // end deleteRoutine
        
        // ask first
        if (apps.length == 1) {
            newConfirm(_("Delete Appointment"),_("Are you sure you want to delete the selected item?"),AlertPopup.YESNO,null,null,deleteRoutine,null,null); /* i18n */
        } else {
            newConfirm(_("Delete Appointment"),_("Are you sure you want to delete the selected items?"),AlertPopup.YESNO,null,null,deleteRoutine,null,null); /* i18n */
        }
    });
}

// util
function getCumulativeOffset(elem) {
    var top = 0, left = 0;
    while (elem) {
      top += elem.offsetTop || 0;
      left += elem.offsetLeft || 0;
      elem = elem.offsetParent;
    }
    return { "top": top, "left": left };
}

function CalendarGrid() {
   
   this.domNode = null;
   this.statusValid = false;
   
   // rows
   this.rows = [];
   this.rowIndex = {};
   this.lastSelectedRowIndex = 0;
   this.autoSortEnabled = false;
   this.sortAsc = true;
   this.isSorted = false;
   
   // appointments
   this.appointments = {};
   // create/get selection for appointments
   this.appointmentSelection = getSimpleSelection("gridCalendarAppointments");
   this.appointmentSelection.setChangedEvent("OX_Calendar_Appointment_Selected");
   
   // grid & day interval
   this.gridIntervalStart = 0;
   this.gridIntervalEnd = 0;
   this.gridIntervalLength = 28;
   this.dayIntervalStart = 0;
   this.dayIntervalEnd = 23;
   this.workdayStart = configGetKey("gui.calendar.starttime") || 8;
   this.workdayEnd = configGetKey("gui.calendar.endtime") || 17;
   this.ONE_HOUR = 1000 * 60 * 60;
   this.ONE_DAY = this.ONE_HOUR * 24;
   this.ONE_WEEK = this.ONE_DAY * 7;
   // initialize date
   this.setDayInterval();
   this.setGridInterval();
   this.setGridIntervalLength(28); // 4 weeks
   
   // zoom level
   this.zoom = 100;
   
   // title bar, container, controlBar, autoComplete
   this.titleBar = newnode("div", {
       position: "absolute",
       top: "0px", right: "0px", left: "0px", height: "30px"
   }, { className: "titleBar" });
   this.container = newnode("div", {
       position: "absolute", 
       top: "30px", right: "0px", bottom: "30px", left: "0px" 
   });
   this.dataGrid = null;
   this.controlBar = newnode("div", {
       position: "absolute", 
       right: "0px", bottom: "0px", left: "0px", height: "30px",
       lineHeight: "30px"
   });
   this.autoCompleteContainer = newnode("div", {
       position: "absolute", top: "0px", left: "0px", width: "600px", height: "70px", zIndex: 1000,
       visibility: "hidden", backgroundColor: "white", border: "1px solid #aaa", fontSize: "8pt",
       overflow: "auto", overflowX: "hidden", overflowY: "auto", MozBoxShadow: "#333 5px 5px 8px"
   });
   document.body.appendChild(this.autoCompleteContainer);
   
   // participants dialog
   this.participantsDialog = null;
   this.currentTeam = null;
   
   // selection
   this.rowSelection = {};
   
   // events
   this.selectionChangedEvent = null;
   this.teamMemberChangedEvent = null;
   this.zoomLevelChangedEvent = null;
   
   // status information
   this.firstRun = true;
   this.manualScroll = false;
   this.showDetails = true;
}

/* imports current team of team week (if exists) or loads the default team */
CalendarGrid.prototype.importTeam = function() {
    // cooperate with week view - any teams out there?
    if (hasTeamMembers()) {
        var Self = this, previousTeam = this.currentTeam;
        this.loadTeam(getTeamArray(), function() {
            // reset current team
            Self.currentTeam = previousTeam;
        });
    } else {
        this.loadDefaultTeam();
    }
}

CalendarGrid.prototype.setDayInterval = function(hourStart, hourEnd) {
    if (!isNaN(hourStart) && !isNaN(hourEnd) && hourStart < hourEnd) {
        this.dayIntervalStart = hourStart;
        this.dayIntervalEnd = hourEnd;
    } else {
        this.dayIntervalStart = 0
        this.dayIntervalEnd = 23;
    }
}

CalendarGrid.prototype.scroll2Date = function(date, roundTo) {
    // adjust parameter
    var roundTo = typeof(roundTo) == "undefined" ? this.ONE_HOUR : roundTo;
    // inside time interval?
    var t = Math.floor(date.getTime()/roundTo)*roundTo;
    if (this.insideGridInterval(t)) {
        var x = this.time2pixel(t);
        this.dataGridContainer.scrollLeft = x;
        this.gridHeader.scrollLeft = x;
        this.lastScrollLeft = x;
    }
}

CalendarGrid.prototype.insideGridInterval = function(timestamp) {
    return timestamp >= this.gridIntervalStart.getTime() && timestamp <= this.gridIntervalEnd.getTime();
}

CalendarGrid.prototype.setGridIntervalToWeek = function(timestamp, intervalLength) {
    // timestamp?
    timestamp = timestamp || this.gridIntervalStart.getTime() || (new Date()).getTime();
    // round down to days
    var t = Math.floor(timestamp/this.ONE_DAY)*this.ONE_DAY;
    // check week day
    var w = (new Date(t)).getUTCDay() - 1; // sub 1 since it starts with Sunday (=0)
    // subtract days (so that t points at Monday morning) & create date
    var d = new Date(t - w * this.ONE_DAY);
    // set interval (default length = 4 weeks)
    this.setGridInterval(d.getUTCFullYear(), d.getUTCMonth(), d.getUTCDate(), intervalLength || this.gridIntervalLength);
}

CalendarGrid.prototype.setGridIntervalLength = function(intervalLength) {
    // new value?
    if (this.gridIntervalLength != intervalLength) {
        this.gridIntervalLength = intervalLength;
        if (this.gridIntervalStart) {
            var d = this.gridIntervalStart;
            this.setGridInterval(d.getUTCFullYear(), d.getUTCMonth(), d.getUTCDate());
        }
    }
}

CalendarGrid.prototype.setGridInterval = function(UTCyear, UTCmonth, UTCday, intervalLength) {
    // no date given?
    var year = UTCyear == null || isNaN(UTCyear) ? activeYear : UTCyear;
    var month = UTCmonth == null || isNaN(UTCmonth) ? activeMonth : UTCmonth;
    var day = UTCday == null || isNaN(UTCday) ? activeDay : UTCday;
    // set date
    var utc_start = Date.UTC(year, month, day, this.dayIntervalStart, 0, 0);
    var maxDay = intervalLength || this.gridIntervalLength || this.getMonthLength(year, month);
    this.gridIntervalStart = new Date(utc_start);
    var utc_end = utc_start + Math.max(0, maxDay - 1) * this.ONE_DAY + (this.dayIntervalEnd-this.dayIntervalStart+1) * this.ONE_HOUR - 1;
    this.gridIntervalEnd = new Date(utc_end);
    // invalidate scrolling positions
    this.invalidScrollPositions = true;
    // invalidate appointments
    this.invalidateAppointments();
}

CalendarGrid.prototype.invalidateAppointments = function() {
    // clear selection
    this.appointmentSelection.clear();
    // clear appointments
    this.appointments = {};
};

CalendarGrid.prototype.getAppointments = function(callback) {
    
    // create multiple requests
    var multiple = { requests: [], objects: [] };
    var skipRequest = true;
    
    // get timestamps
    var start = this.gridIntervalStart.getTime();
    var end = this.gridIntervalEnd.getTime();
    
    // loop members
    for (var i = 0; i < this.rows.length; i++) {
        // get row data
        var data = this.rows[i].data;
        // prepare request object
        var req = {};
        req.module = "calendar";
        req.action = "freebusy";
        req.id = data.id;
        req.type = data.type; // 1 = user; 3 = resource
        req.start = start;
        req.end = end;
        // add to request list
        multiple.requests.push(req);
        // appointments already loaded?
        var rowId = data.id + ":" + data.type;
        skipRequest = skipRequest && typeof(this.appointments[rowId]) != "undefined";
    }
    
    // send server request
    if (this.rows.length > 0 && !skipRequest) {
        var Self = this;
        json.put(AjaxRoot + "/multiple?session=" + session, multiple.requests, null, function(responses) {
        	var tFolders = { };
        	// to prevent an empty folder array later we always add the users default calendar
        	tFolders[configGetKey("folder.calendar")] = configGetKey("folder.calendar");
            if (responses && responses.length) {
                // loop responses
                for (var r = 0; r < responses.length; r++) {
                    var response = responses[r].data
                    var row = Self.rows[r];
                    var id = row.rowId;
                    // not yet loaded?
                    if (!Self.appointments[id]) {
                        // create new list for this id
                        Self.appointments[id] = [];
                        // loop data
                        for (var i = 0; i < response.length; i++) {
                            var app = response[i];
                            // tidy up recurrence appointments
                            Self.tidyUpAppointment(app);
                            // add
                            Self.appointments[id].push(app);
                            // store folder_id in hash to check permissions later
                            if (app.folder_id && !tFolders[app.folder_id]) {
                            	tFolders[app.folder_id] = app.folder_id;
                            }
                            // folder_id vs. folder
                            app.folder = app.folder_id;
                        }
                    }
                }
                // now we have the relevant appointments for each row.
                // we can get the appointments of a row by looking up its id
                // and then using this id to find the appointments in the
                // variable "appointments"
            }
            
            // convert folder hash to array
            var folders = [];
            for (var i in tFolders) {
            	folders.push(tFolders[i]);
            }
            // fetch folder permissions
            // @todo: check the response! it looks like the last folder in
            // the array is always undefined! likely a folder cache bug as the
            // server response seems  to be ok.
            oMainFolderTree.fetchAndGetFolders(folders, 
            		function(result) {
		            	// callback?
		                if (callback) callback(result);
            		} 
            );
        });
    } else {
        // callback?
        if (callback) callback();
    }
}

CalendarGrid.prototype.tidyUpAppointment = function(app) {
    // position?
    if (typeof(app.recurrence_position) == "undefined") {
        app.recurrence_position = 0;
    }
    // recurrence?
    if (typeof(app.recurrence_id) == "undefined") {
        app.recurrence_id = app.id;
    }
    // add alias of folder_id
    if (typeof(app.folder_id) == "undefined") {
        app.folder = app.folder_id;
    }
}

CalendarGrid.prototype.triggerTeamMemberChanged = function() {
    // event defined?
    if (this.teamMemberChangedEvent) {
        // patch: cooperate with week view
        importTeam(this.getRowHash());
        // trigger (selected or all)
        triggerEvent(this.teamMemberChangedEvent, this.getSelectedRowsOrAll());
    }
}

CalendarGrid.prototype.add = function(row) {
    if (!this.hasRow(row)) {
        // add to list
        this.rowIndex[row.rowId] = row.index = this.rows.length;
        this.rows.push(row);
        // sort rows
        this.autoSort();
        // invalidate current team
        this.currentTeam = null;
    }
    // trigger event
    this.triggerTeamMemberChanged();
}

CalendarGrid.prototype.insertAt = function(row, index) {
    if (!this.hasRow(row)) {
        // adjust
        index = Math.max(0, Math.min(index, this.rows.length));
        // add to list and update index
        this.rows.splice(index, 0, row);
        this.updateRowIndex();
        // sort rows
        this.autoSort();
        // invalidate current team
        this.currentTeam = null;
    }
    // trigger event
    this.triggerTeamMemberChanged();
}

CalendarGrid.prototype.hasRow = function(row) {
    return typeof(this.rowIndex[row.rowId]) != "undefined";
}

CalendarGrid.prototype.getRowHash = function() {
    var hash = {};
    for (var i = 0; i < this.numRows(); i++) {
        var row = this.rows[i];
        hash[row.rowId] = row;
    }
    return hash;
}

CalendarGrid.prototype.makeFunky = function(hash) {
    var list = [];
    for (var id in hash) {
        var row = hash[id];
        // some storage style? don't ask me
        list.push([0, 0, row.display_name, 0, row.id, 0, 0, row.type]);
    }
    return list;
}

CalendarGrid.prototype.createAppointment = function(startDate, endDate, members) {
    // is all day?
    var allDay = startDate.getTime() < endDate.getTime() && startDate.getUTCHours() == 0 && endDate.getUTCHours() == 0;
    // create date (activefolder is global)
    calendar_createNewAppointment(startDate, activefolder, allDay, {
        "data": this.makeFunky(members), "module": "contacts"
    }, endDate);
}

CalendarGrid.prototype.createAppointmentForSelectedMembers = function(startDate, endDate) {
    // get selected members (or all)
    var rows = this.numSelectedRows() > 0 ? this.getSelectedHash() : this.getRowHash();
    this.createAppointment(startDate, endDate, rows);
}

CalendarGrid.prototype.autoSort = function() {
    if (this.autoSortEnabled) {
        this.sort();
    }
}

CalendarGrid.prototype.sort = function() {
    // sort rows
    var Self = this;
    this.rows.sort(function() {
        // bind to "this"
        return CalendarGrid.prototype.rowSorter.apply(Self, arguments);
    });
    this.isSorted = true;
    // update row index (required after sort)
    this.updateRowIndex();
    // trigger event
    this.triggerTeamMemberChanged();
}

CalendarGrid.prototype.updateRowIndex = function() {
    for (var i = 0; i < this.numRows(); i++) {
        var row = this.rows[i];
        this.rowIndex[row.rowId] = row.index = i;
    }
}

CalendarGrid.prototype.addMultiple = function(list) {
    // loop through list
    for (var i = 0; i < list.length; i++) {
        var row = list[i];
        if (!this.hasRow(row)) {
            // add to list
            this.rows.push(row);
            // invalidate current team
            this.currentTeam = null;
        }
    }
    // sort rows
    this.autoSort();
    // trigger event
    this.triggerTeamMemberChanged();
}

CalendarGrid.prototype.rowSorter = function(a, b) {
    var v = false;
    if (a.type < b.type) {
        v = false;
    } else if (a.type > b.type) {
        v = true;
    } else {
        v = a.display_name.toLowerCase() >= b.display_name.toLowerCase();
    }
    return (this.sortAsc ? v : !v) ? 1 : -1; // Safari needs -1!
}

CalendarGrid.prototype.loadDefaultTeam = function() {
    // exist in configuration?
    if (configContainsKey("gui.calendar.teams")) {
        var teams = configGetKey("gui.calendar.teams");
        // loop teams
        for (var i = 0; i < teams.length; i++) {
            // marked as default?
            if (teams[i].team_default) {
                // load team
                var Self = this;
                this.loadTeam(teams[i].children, function() {
                    Self.currentTeam = i;
                });
                break;
            }
        }
    }
}

CalendarGrid.prototype.loadTeamByIndex = function(index, callback) {
    // exist in configuration?
    if (configContainsKey("gui.calendar.teams")) {
        var teams = configGetKey("gui.calendar.teams");
        // loop teams
        for (var i = 0; i < teams.length; i++) {
            if (i == index) {
                // load team
                var Self = this;
                this.loadTeam(teams[i].children, function() {
                    Self.currentTeam = i;
                    if (callback) callback();
                });
                break;
            }
        }
    }
}

CalendarGrid.prototype.loadTeam = function(team, callback) {
    // remove all
    this.removeAll();
    // load/add team
    this.addTeam(team, callback);
}

CalendarGrid.prototype.addTeam = function(team, callback) {
    
    var Self = this;
    
    // clear
    Self.clearPartial();
    
    // get team members
    var groups = [];
    for (var i = 0; i < team.length; i++) {
        var member = team[i];
        // handle different types
        switch (member.type) {
            case 1: // user
            case 3: // resource
                var row = new CalendarGridRow(member.type, member.id, member.display_name);
                row.data = member;
                this.add(row);
                break;
            case 2: // group
                groups.push(member);
        }
    }

    // local function
    var CG_localCallback = function() {
        // sort rows
        Self.autoSort();
        // lazy update
        setTimeout(function() { Self.update(); }, 50);
        // trigger event
        Self.triggerTeamMemberChanged();
        // callback?
        if (callback) callback();
    };

    var CG_resolvedGroups = function(groups) {
        // "groups" is a hash, so...
        var ids = [];
        for (var id in groups) {
            var group = groups[id];
            // loop through members (also a hash)
            for (var memberId in group.members) {
                ids.push(group.members[memberId]);
            }
        }
        if (ids.length > 0) {
            // ask cache about these users (async)
            internalCache.getUsers(ids, CG_resolvedUsers);
        } else {
            CG_localCallback();
        }
    };
    
    var CG_resolvedUsers = function(users) {
        var rows = [];
        for (var i in users) {
            var user = users[i];
            // add row
            user.type = 1;
            var row = new CalendarGridRow(user.type, user.id, user.display_name);
            row.data = user;
            rows.push(row);
        }
        Self.addMultiple(rows);
        CG_localCallback();
    }
    
    // groups to resolve?
    if (groups.length > 0) {
        // ask cache about these groups (async)
        internalCache.getObjects(groups, CG_resolvedGroups);
    } else {
        CG_localCallback();
    }
}

CalendarGrid.prototype.removeAll = function() {
    // delete rows
    this.rows = [];
    // clear index
    this.rowIndex = {};
    // clear selection
    this.rowSelection = {};
    // invalidate current team
    this.currentTeam = null;
    // trigger selection event?
    if (this.selectionChangedEvent) {
        triggerEvent(this.selectionChangedEvent, this.numSelectedRows());
    }
    // trigger changed event
    this.triggerTeamMemberChanged();
}

CalendarGrid.prototype.removeById = function(id) {
    // find row
    for (var i = 0; i < this.rows.length; i++) {
        if (this.rows[i].rowId == id) {
            this.removeAt(i);
            break;
        }
    }
}

CalendarGrid.prototype.removeAt = function(index) {
    // get row
    var row = this.rows[index];
    // delete from list
    delete this.rowIndex[row.rowId];
    this.rows.splice(index, 1);
    // invalidate current team
    this.currentTeam = null;
    // delete from selection
    for (var id in this.rowSelection[row.rowId]) {
        if (id == row.rowId) {
            delete this.rowSelection[id];
            // trigger event?
            if (this.selectionChangedEvent) {
                triggerEvent(this.selectionChangedEvent, this.numSelectedRows());
            }
        }
    }
    // trigger changed event
    this.triggerTeamMemberChanged();
}

CalendarGrid.prototype.removeSelectedRows = function(index) {
    var i = 0;
    while (i < this.rows.length) {
        var row = this.rows[i];
        // is selected?
        if (this.rowSelection[row.rowId]) {
            this.rows.splice(i, 1);
            delete this.rowIndex[row.rowId];
            delete this.rowSelection[row.rowId];
            // invalidate current team
            this.currentTeam = null;
            continue;
        }
        i++;
    }
    // trigger event?
    if (this.selectionChangedEvent) {
        triggerEvent(this.selectionChangedEvent, this.numSelectedRows());
    }
    // trigger changed event
    this.triggerTeamMemberChanged();
}

CalendarGrid.prototype.clickRow = function(index, e) {
    // event given?
    e = e || { shiftKey: false, ctrlKey: false };
    // shift?
    if (e.shiftKey && this.numSelectedRows() > 0) {
    	// holds shift!
    	this.selectNone();
    	this.selectRows(this.lastSelectedRowIndex, index);
    } else if (this.rows[index]) {
    	// multiple?
        if (!e.ctrlKey) {
            this.selectNone();
        }
    	if (this.rowSelection[this.rows[index].rowId]) {
	    	// <ctrl> and click on selected row
	    	var row = this.rows[index];
	    	row.node.className = "member";
	        delete this.rowSelection[row.rowId];
	    } else {
	    	this.selectRows(index, index);
	    }
    	this.lastSelectedRowIndex = index;
    }
}

CalendarGrid.prototype.selectRows = function(start, end) {
    if (start > end) {
        var tmp = end; end = start; start = tmp;
    }
	for (var i = start; i <= end; i++) {
		var row = this.rows[i];
		row.node.className = "member rowSelected";
		this.rowSelection[row.rowId] = row;
	}
	
	// trigger event?
    if (this.selectionChangedEvent) {
        triggerEvent(this.selectionChangedEvent, this.numSelectedRows());
    }
    // trigger changed event
    this.triggerTeamMemberChanged();
}

CalendarGrid.prototype.selectNone = function() {
    for (var id in this.rowSelection) {
        this.rowSelection[id].node.className = "member";
        delete this.rowSelection[id];
    }
    // trigger changed event
    this.triggerTeamMemberChanged();
}

CalendarGrid.prototype.numSelectedRows = function(row) {
    var count = 0;
    for (var id in this.rowSelection) count++;
    return count;
}

CalendarGrid.prototype.setSelectionChangedEvent = function(eventName) {
    this.selectionChangedEvent = eventName;
}

CalendarGrid.prototype.setTeamMemberChangedEvent = function(eventName) {
    var oldEvent = this.teamMemberChangedEvent;
    this.teamMemberChangedEvent = eventName;
    // fire now?
    if (oldEvent != eventName) {
        this.triggerTeamMemberChanged();
    }
}

CalendarGrid.prototype.numRows = function(row) {
    return this.rows.length;
}

CalendarGrid.prototype.getSelectedHash = function() {
    return this.rowSelection;
}

CalendarGrid.prototype.getSelectedRows = function() {
    var selection = [];
    for (var id in this.rowSelection) {
        selection.push(this.rowSelection[id]);
    }
    return selection;
}

CalendarGrid.prototype.getSelectedRowsOrAll = function() {
    return this.numSelectedRows() > 0 ? this.getSelectedRows() : this.rows;
}

CalendarGrid.prototype.getSelectedIndexes = function() {
    var selection = [];
    for (var id in this.rowSelection) {
        selection.push(this.rowIndex[id]);
    }
    return selection;
}

CalendarGrid.prototype.getSelectedIDs = function() {
    var selection = [];
    for (var id in this.rowSelection) {
        selection.push(id);
    }
    return selection;
}

CalendarGrid.prototype.setParentDOMNode = function(node) {
    // set node
    this.domNode = node;
    // set class
    this.domNode.className += " calendarGrid";
    // add children
    node.appendChild(this.titleBar);
    node.appendChild(this.container);
    node.appendChild(this.controlBar);
    // prevent selection
    node.onselectstart = function() { return false; }
    node.style.MozUserSelect = "none";
}

CalendarGrid.prototype.validate = function() {
    if (!this.statusValid && this.domNode != null) {
        this.draw();
        this.statusValid = true;
    }
}

CalendarGrid.prototype.update = function() {
	if (this.domNode != null) {
		this.drawPartial();
	}
}

CalendarGrid.prototype.validateOrUpdate = function() {
	if (this.domNode != null) {
		if (!this.statusValid) {
			this.validate();
		} else {
			this.update();
		}
	}
}

CalendarGrid.prototype.getMonthLength = function(year, month) {
    var d = Date.UTC(year, month+1, 1);
    d -= this.ONE_DAY;
    d = new Date(d);
    return d.getUTCDate();
}

CalendarGrid.prototype.moveInterval = function(type, steps) {
    // get current interval start
    var start = new Date(this.gridIntervalStart);
    var year = start.getUTCFullYear();
    var month = start.getUTCMonth();
    var day = start.getUTCDate();
    // get sign
    var sign = steps/Math.abs(steps);
    // iterate
    for (var i = 0; i < Math.abs(steps); i++) {
        // consider type
        switch (type) {
            case "MONTH":
                month += sign;
                break;
            case "WEEK":
                day += sign * 7;
                break;
            case "DAY":
                day += sign;
                break;
        }
    }
    // update grid interval
    this.setGridInterval(year, month, day);
    // update grid
    this.clearPartial();
    this.update();
}

CalendarGrid.prototype.draw = function() {

    var Self = this;
    
    // title
    var arrowLeft = newnode("span", { cursor: "pointer" }, null, [
        newtext(" "),
        newnode("img", { 
            width: "4px", height: "7px", margin: "0px 5px 0px 20px"
        }, {
            src: getFullImgSrc("img/arrows/arrow_darkgrey_left.gif"), alt: ""
        }),
        newtext(" ")
    ]);
    addDOMEvent(arrowLeft, "click", function(e) {
        // change interval
        Self.moveInterval("DAY", -Self.gridIntervalLength);
        // update calendar
        activeYear = Self.gridIntervalStart.getUTCFullYear();
        activeMonth = Self.gridIntervalStart.getUTCMonth();
        activeDay = Self.gridIntervalStart.getUTCDate();
        if (oMiniCalendar) oMiniCalendar.setSelectedByDate(activeYear, activeMonth, activeDay);
    });
    this.titleBar.appendChild(arrowLeft);
    this.titleDates = newtext("");
    this.drawDateInterval();
    this.titleBar.appendChild(this.titleDates);
    // right arrow
    var arrowRight = newnode("span", { cursor: "pointer" }, null, [
        newtext(" "),
        newnode("img", { 
            width: "4px", height: "7px", margin: "0px 20px 0px 5px"
        }, {
            src: getFullImgSrc("img/arrows/arrow_darkgrey_right.gif"), alt: ""
        }),
        newtext(" ")
    ]);
    addDOMEvent(arrowRight, "click", function(e) {
        // change interval
        Self.moveInterval("DAY", +Self.gridIntervalLength);
        // update calendar
        activeYear = Self.gridIntervalStart.getUTCFullYear();
        activeMonth = Self.gridIntervalStart.getUTCMonth();
        activeDay = Self.gridIntervalStart.getUTCDate();
        if (oMiniCalendar) oMiniCalendar.setSelectedByDate(activeYear, activeMonth, activeDay);
    });
    this.titleBar.appendChild(arrowRight);

    // --------------------------------------------------------------------------
    
    // container
    this.drawPartial();
    
    // --------------------------------------------------------------------------
    
    // controls
    // 1. slider
    this.zoomSlider = newnode("div", null,
            { className: "zoomSlider" }
    );
    this.zoomLevel = newtext(Self.zoom + " %");
    var zoomLevelBox = newnode("span", {
        fontSize: "8pt", color: "white"
        }, null //, [this.zoomLevel]
    );
    this.zoomSliderBox = newnode("div", null,
            { className: "zoomSliderBox" }, [this.zoomSlider, zoomLevelBox]
    );
    // 2. combo box
    this.zoomCombo = newnode("select", null, { size: 1 });
    // add options
    var zoomLevels = ["","10","50","75","100","200","400","700","1000"];
    for (var i = 0; i < zoomLevels.length; i++) {
        var text = newtext(zoomLevels[i] ? zoomLevels[i] + " %" : "");
        var option = newnode("option", null, { value: zoomLevels[i] }, [text]);
        this.zoomCombo.appendChild(option);
    }
    var zoomComboBox = newnode("div", null,
            { className: "zoomComboBox" }, [this.zoomCombo]
    );
    // add listener to combo box
    addDOMEvent(this.zoomCombo, "mousedown", function() {
        Self.zoomCombo.firstChild.innerHTML = "";
        Self.zoomCombo.firstChild.value = "";
    });
    addDOMEvent(this.zoomCombo, "change", function() {
        var v = Self.zoomCombo.value;
        if (v) {
            Self.zoomCombo.firstChild.innerHTML = "";
            Self.zoomCombo.firstChild.value = "";
            // initialize scroll/zoom fix
            //Self.scrollPositionFix.style.left = Self.roundEm(Self.dataGridContainer.scrollLeft/Self.zoom) + 1 + "em";
            // change zoom level
            Self.changeZoomLevel(v);
        }
    });
    // add controls
    this.controlBar.appendChild(this.zoomSliderBox);
    this.controlBar.appendChild(zoomComboBox);
    
    // update zoom level
    this.updateZoomControls(this.zoom);
    
    // add listener
    this.zooming = false;
    this.zoomSliderLeft = 0;
    this.zoomTimeout = null;
    this.lastZoom = this.zoom;
    addDOMEvent(this.zoomSlider, "mousedown", CalendarGrid_startSlider);
    addDOMEvent(this.zoomSliderBox, "mousedown", CalendarGrid_startSlider);
    function CalendarGrid_startSlider(e) {
        // hide appointments
        try {
            Self.dataGrid.removeChild(Self.appContainer);
        } catch(e) { }
        // remove scroll handler
        removeDOMEvent(Self.dataGridContainer, "scroll", Self.scrollHandler);
        // start zooming
        Self.zooming = true;
        Self.zoomSliderLeft = getCumulativeOffset(Self.zoomSliderBox).left + 10;
        // make first move
        addDOMEvent(document, "mousemove", CalendarGrid_moveSlider);
        CalendarGrid_moveSlider(e);
    };
    addDOMEvent(document, "mouseup", function() {
        if (Self.zooming) {
            // stop zooming
            Self.zooming = false;
            window.setTimeout(function() {
                try {
                    // show appointments
                    Self.dataGrid.appendChild(Self.appContainer);
                    removeDOMEvent(document, "mousemove", CalendarGrid_moveSlider);
                    // add scroll handler
                    addDOMEvent(Self.dataGridContainer, "scroll", Self.scrollHandler);
                } catch(e) { }
            }, 100);
        }
    });
    function CalendarGrid_moveSlider(e) {
        if (Self.zooming) {
            var x = e.clientX - Self.zoomSliderLeft;
            x = Math.max(0, Math.min(180, x)) + 10;
            // snap to grid (10 pixel)
            x = Math.round(x/10) * 10;
            // get zoom level
            var z = x <= 100 ? x : 100 + (x-100) * 10;
            if (!isNaN(z) && z != Self.lastZoom) {
                Self.changeZoomLevel(z);
            }
        }
    }
    
    // 3. "Details" checkbox
    var showDetailsBox = newnode("input", { border: "0px none" }, {
        type: "checkbox", value: "1", defaultChecked: "checked", checked: "checked", id: "calendarGridShowDetails"
    });
    var showDetailsLabel = newnode("label", 0, 0, [newtext(_("Details"))]);
    showDetailsLabel.setAttribute("for", "calendarGridShowDetails");
    var showDetailsDiv = newnode("div", {
        position: "absolute", left: "200px", top: "2px"
    }, 0, [showDetailsBox, showDetailsLabel]);
    // add handler
    if (IE) {
		// IE checkbox fix
		var detailHandlerIE = function() {
			Self.showDetails = showDetailsBox.checked;
			Self.clearPartial();
			Self.update();
		};
		addDOMEvent(showDetailsBox, "click", detailHandlerIE);
	} else {
		var detailHandler = function() {
	    	Self.showDetails = showDetailsBox.checked;
	    	if (!Self.showDetails) {
	    		Self.dataGridContainer.className += " bar";
	    	} else {
	    		Self.dataGridContainer.className = Self.dataGridContainer.className.replace(/\s*bar/g, '');
	    	}
	    };
		addDOMEvent(showDetailsBox, "change", detailHandler);
	}
    this.controlBar.appendChild(showDetailsDiv);
}

CalendarGrid.prototype.changeZoomLevel = function(z) {
    // check value
    z = Math.max(10, Math.min(1000, z));
    // set
    this.zoom = z;
    this.lastZoom = z;
    var Self = this;
    // speed!
    window.setTimeout(function() { Self.updateZoomControls(Self.zoom); }, 0);
    window.clearTimeout(this.zoomTimeout);
    this.zoomTimeout = window.setTimeout(function() {
        // new font size
        var fs = Self.zoom + "px";
        // set new zoom level
        Self.dataGrid.style.fontSize = fs;
        Self.gridHeader.style.fontSize = fs;
        // hide/show hours?
        Self.gridHeader.className = Self.zoom >= CG_HEAD_ZOOM ? "gridHeader zoom1" : "gridHeader zoom0";
        // restore correct scroll position
        Self.dataGridContainer.scrollLeft = Self.scrollPositionFix.offsetLeft;
        Self.gridHeader.scrollLeft = Self.scrollPositionFix.offsetLeft;
        // trigger event?
        if (Self.zoomLevelChangedEvent) {
            triggerEvent(Self.zoomLevelChangedEvent, Self.zoom);
        }
    }, 250);
}

CalendarGrid.prototype.updateZoomControls = function(zoomLevel) {
    // get x
    var x = zoomLevel <= 100 ?
            zoomLevel : Math.round(zoomLevel/10) + 90;
    // update slider
    if (this.zoomSlider) {
        this.zoomSlider.style.left = (x-10) + "px";
    }
    // update combo box
    if (this.zoomCombo) {
        this.zoomCombo.firstChild.innerHTML = "";
        this.zoomCombo.firstChild.value = "";
        this.zoomCombo.value = this.zoom;
        // check
        if (this.zoomCombo.value != this.zoom) {
            this.zoomCombo.firstChild.innerHTML = this.zoom + " %";
            this.zoomCombo.firstChild.value = this.zoom;
            this.zoomCombo.value = this.zoom;
        }
    }
}

CalendarGrid.prototype.setZoomLevelChangedEvent = function(eventName) {
    this.zoomLevelChangedEvent = eventName;
}

CalendarGrid.prototype.snapX2grid = function(x, factor) {
    // snap  x position to grid (pixel; half hours / quarter hours)
    var factor = factor ? factor : (this.zoom <= 400 ? 2 : 4);
    var snapper = 100 * CG_PX2EM * this.zoom / 100 / 24 / factor;
    return Math.round(Math.round(x/snapper)*snapper);
}

CalendarGrid.prototype.time2pixel = function(t) {
    // convert into time (~1200 Pixel = 1 Day if zoom = 100%)
    var x = Math.round((t - this.gridIntervalStart.getTime()) * this.zoom * 100 * CG_PX2EM / this.ONE_DAY / 100 );
    return x;
}

CalendarGrid.prototype.pixel2time = function(x, snap2Grid) {
    // convert into time (1200 Pixel = 1 Day if zoom = 100%)
    var t = this.gridIntervalStart.getTime() + Math.round(this.ONE_DAY * x * 100 / this.zoom / 100 / CG_PX2EM);
    // snap to grid?
    if (snap2Grid) {
        // half hour / quarter hour
        var snapper = this.zoom <= 400 ? this.ONE_HOUR/2 : this.ONE_HOUR/4;
        t = Math.round(t/snapper)*snapper;
    }
    return t;
}

CalendarGrid.prototype.roundEm = function(em) {
    return Math.round(em*1000)/1000;
}

CalendarGrid.prototype.drawDateInterval = function() {
    var startDate = formatDate(this.gridIntervalStart, "date");
    var endDate = formatDate(this.gridIntervalEnd, "date");
    this.titleDates.nodeValue = "" +
        // start date
        startDate + " " +
        // start calendar week
        "(" + _("CW") + " " + formatDateTime("w", new Date(this.gridIntervalStart)) + ")" + /* i18n */
        // different?
        (startDate != endDate ?
            // dash
            " - " + 
            // end date
            endDate + " " +
            // end calendar week
            "(" + _("CW") + " " + formatDateTime("w", new Date(this.gridIntervalEnd)) + ")" /* i18n */
            :
            ""
        );
}

CalendarGrid.prototype.drawPartial = function() {
    
    var Self = this;
    
    // scroll position fix
    if (this.invalidScrollPositions) {
        this.scrollPositionFix = null;
        this.manualScroll = false;
    }
    
    // -------------------------------------------------------------------------
    
    // update title
    this.drawDateInterval();
    
    // -------------------------------------------------------------------------
    
    // current zoom -> pixel
    var fs = Math.round(this.zoom) + "px";
    
    // container
    this.dataGrid = newnode("div", { fontSize: fs }, { "className": "zoom0" });
    this.dataGridContainer = newnode("div", null, { 
        "className": "dataGrid" }, [this.dataGrid]
    );

    var maxRows = this.numRows();
    var maxCols = Math.ceil((this.gridIntervalEnd.getTime() - this.gridIntervalStart.getTime())/this.ONE_DAY);
    var hourStart = this.dayIntervalStart;
    var hourStop = this.dayIntervalEnd;
    var numHours = hourStop - hourStart + 1;
    var hourWidth = CG_PX2EM/numHours;
    
    // set data grid width
    this.dataGrid.style.width = (maxCols * numHours * hourWidth) + "em";
    
    // hours (vertical; one extra row for new members)
    var wdStart = this.workdayStart - 1, wdEnd = this.workdayEnd - 1;
    var hourZ = 0;
    for (var i = 0; i < maxCols; i++) {
        for (var h = hourStart; h <= hourStop; h++) {
            var hourClassName = "hour ";
            hourClassName += h < wdStart || h > wdEnd ? "dark " : "bright ";
            if (i != 0 || h != 0) {
                hourClassName += h == 0 ? "borderColorMidnight " : h < wdStart || h > wdEnd ? "borderColorDark " : "borderColorBright ";
                hourClassName += h == 0 ? "borderThick" : "borderThin";
            }
            var hour = newnode("div", {
                left: this.roundEm((i * CG_PX2EM) + ((h-hourStart) * hourWidth)) + "em",
                width: this.roundEm(hourWidth) + "em",
                height: ((maxRows+1) * CG_ROW_HEIGHT) + "px"
            }, { className: hourClassName }, [newnode("div", { fontSize: "10px" })]);
            this.dataGrid.appendChild(hour);
        }
    }
    // shadows (horizontal; one extra row for new members)
    for (var r = 0; r <= maxRows; r++) {
        if (r % 2 == 1) { // IE ||
            var shadowsPerRow = IE ? Math.ceil(maxCols/2) : 1; // IE opacity fix
            for (var i = 0; i < shadowsPerRow; i++) {
                var shadow = newnode("div", {
                    top: (r * CG_ROW_HEIGHT - 1) + "px",
                    left: this.roundEm(i * maxCols * CG_PX2EM / shadowsPerRow) + "em",
                    width: this.roundEm(maxCols * CG_PX2EM / shadowsPerRow) + "em",
                    backgroundColor: IE && false ? "transparent" : "black"
                }, { className: "rowShadow" }, [newnode("div", { fontSize: "10px" })]);
                this.dataGrid.appendChild(shadow);
            }
        }
    }
    var lastRow = newnode("div", {
        top: ((maxRows+1) * CG_ROW_HEIGHT - 1) + "px",
        width: (maxCols * CG_PX2EM) + "em"
    }, { className: "lastRow" }, [newnode("div", { fontSize: "10px" })]);
    this.dataGrid.appendChild(lastRow);
    
    //--------------------------------------------------------------------------
    
    // add scroll/zoom position fix    
    // this one is required to keep the horizontal scroll position during zoom
    var top = "0px", left = "0px";
    if (this.scrollPositionFix) {
        top = this.scrollPositionFix.style.top || "0px";
        left = this.scrollPositionFix.style.left || "0px";
    }
    this.scrollPositionFix = newnode("div", {
        position: "absolute", "top": top, height: "1px", "left": left,
        width: "1px", zIndex: 0
    });
    this.dataGrid.appendChild(this.scrollPositionFix);
    
    //--------------------------------------------------------------------------
    
    // draw header
    var leftHeaderText = newtext(_("Team Members")); /* i18n */
    var src = !this.isSorted ? "img/dummy.gif" :
        this.sortAsc ? "img/arrows/sort_up.gif" : "img/arrows/sort_down.gif";
    var leftHeaderSort = newnode("img",
        { position: "absolute", right: "2px", top: "8px", width: "16px", height: "16px", border: "0px none" },
        { "src": getFullImgSrc(src), alt: "" }
    );
    var src = this.numRows() > 0 ? "img/menu/remove_teammember.gif" : "img/menu/remove_teammember_d.gif";
    var leftHeaderRemove = newnode("img",
            { position: "absolute", left: "10px", top: "8px", width: "16px", height: "16px", border: "0px none" },
            { "src": getFullImgSrc(src), alt: "" }
        );
    var leftHeader = newnode("div", {
        position: "absolute", top: "0px", left: "0px", width: "163px", height: "30px",
        lineHeight: "30px", fontSize: "9pt", textAlign: "left", paddingLeft: "36px",
        borderRight: "1px solid #ccc", cursor: "pointer"
    }, null, [leftHeaderRemove, leftHeaderText, leftHeaderSort]);
    // add event (sort)
    addDOMEvent(leftHeader, "click", function() {
        // sort
        Self.sort();
        // change sort direction
        Self.sortAsc = !Self.sortAsc;
        // update
        Self.clearPartial();
        Self.update();
    });
    // add event (remove)
    addDOMEvent(leftHeaderRemove, "click", function(e) {
        // remove all members
        Self.removeAll();
        // update
        Self.clearPartial();
        Self.update();
        stopEvent(e);
    });
    this.gridHeader = newnode("div", {
        position: "absolute", top: "0px", left: "200px", right: "0px", height: "30px",
        fontSize: fs, overflow: "hidden", cursor: "ew-resize"
    }, { className: "gridHeader" });
    var header = newnode("div", {
        position: "absolute", top: "0px", right: "0px", left: "0px", height: "28px",
        backgroundColor: "#eee", borderBottom: "1px solid #ccc", borderTop: "1px solid #ccc"
    }, null, [leftHeader, this.gridHeader]);
    
    //--------------------------------------------------------------------------
    
    // add days & hours to time grid header
    var baseTime = this.gridIntervalStart.getTime();
    for (var i = 0, hoursTotal = 0; i < maxCols; i++) {
        // create day
        var dayText = formatDate(baseTime+i*this.ONE_DAY, "dateshortday") + " " +
        	"(" + _("CW") + " " + formatDateTime("w", new Date(baseTime+i*this.ONE_DAY)) + ")"; /* i18n */
    	var day = newnode("div", {
            left: (i * CG_PX2EM) + "em",
            width: CG_PX2EM + "em"
        }, { className: "dayContainer" },
        	[ newnode("div", 0, {className : "day" }, [newtext(dayText)]) ]);
        this.gridHeader.appendChild(day);
        // add hours
        for (var h = hourStart; h <= hourStop; h++) {
            // create hour
            var left = (i * CG_PX2EM) + ((h-hourStart) * hourWidth) + "em";
            var hour = newnode("div", {
                "left": left, "width": hourWidth + "em"
            }, { className: "hourContainer", title: dayText },
            		[newnode("div", 0, {className: "hour"}, [newtext(formatDate(baseTime+i*this.ONE_DAY+h*this.ONE_HOUR, "time"))])]);
            this.gridHeader.appendChild(hour);
            hoursTotal++;
            // repeat current day (for higher zoom levels)
            if (h > hourStart + 3) {
            	var smallday = newnode("div", {
                    "left": left, width: hourWidth + "em"
                }, { className: "smallDayContainer" },
                	[newnode("div", 0, {className : "day"}, [newtext(formatDate(baseTime+i*this.ONE_DAY, "shortdate")) ])]);
                this.gridHeader.appendChild(smallday);
            }
        }
    }
    
    // add extra space on the very right to sync scrolling
    this.gridHeaderScrollBarFix = newnode("div", {
        position: "absolute", left: (hoursTotal * CG_PX2EM) + "em",
        width: "15px", height: "30px"
    });
    this.gridHeader.appendChild(this.gridHeaderScrollBarFix);
    
    // --------------------------------------------------------------------------
    
    // add "drag-scroll" (with "flick" gesture)
    this.gridHeaderScrolling = {
            on: false,
            start: 0,
            x: 0,
            lastX: 0,
            decel: 0, wait4decel: false,
            time: 0
    };
    addDOMEvent(this.gridHeader, "mousedown", function(e) {
        var ghs = Self.gridHeaderScrolling;
        ghs.x = Self.gridHeader.scrollLeft;
        ghs.start = e.clientX;
        ghs.decel = 0;
        ghs.wait4decel = false;
        ghs.on = true;
        ghs.time = (new Date()).getTime();
        // hide hours (since they make this terribly slow)
        Self.gridHeader.className = "gridHeader zoom0";
    });
    addDOMEvent(document, "mouseup", function(e) {
        var ghs = Self.gridHeaderScrolling;
        // turn scrolling off
        ghs.on = false;
        // get last scroll duration
        var duration = (new Date()).getTime() - ghs.time;
        // gesture?
        if (duration < 750 && ghs.wait4decel) {
            ghs.decel = 20 * (e.clientX - ghs.start > 0 ? +1 : -1);
            ghs.x = Self.dataGridContainer.scrollLeft;
            GridCalendarDecelerate();
        } else {
            // show hours
            Self.gridHeader.className = Self.zoom >= CG_HEAD_ZOOM ? "gridHeader zoom1" : "gridHeader zoom0";
        }
    });
    addDOMEvent(this.gridHeader, "mousemove", function(e) {
        var ghs = Self.gridHeaderScrolling;
        if (ghs.on) {
            var x = e.clientX - ghs.start;
            Self.dataGridContainer.scrollLeft = ghs.x - x;
            Self.gridHeader.scrollLeft = ghs.x - x;
            Self.manualScroll = true;
            ghs.wait4decel = true;
        }
    });
    function GridCalendarDecelerate() {
        var ghs = Self.gridHeaderScrolling;
        if (ghs.decel != 0) {
            ghs.x = Math.round(ghs.x-ghs.decel);
            Self.dataGridContainer.scrollLeft = ghs.x;
            Self.gridHeader.scrollLeft = ghs.x;
            ghs.decel += ghs.decel > 0 ? -0.25 : +0.25;
            setTimeout(function() { GridCalendarDecelerate(); }, 5);
        } else {
            // show hours
            Self.gridHeader.className = Self.zoom >= CG_HEAD_ZOOM ? "gridHeader zoom1" : "gridHeader zoom0";
        }
    }
    
    // --------------------------------------------------------------------------
    
    // draw left column container
    this.leftColumn = newnode("div", null, { className: "leftColumn" });
    this.leftColumnScrollBarFix = newnode("div", {
        position: "absolute", top: ((maxRows+1) * CG_ROW_HEIGHT) + "px", left: "0px", 
        width: "199px", height: "15px",
        zIndex: 0
    });
    this.leftColumn.appendChild(this.leftColumnScrollBarFix);
    
    // sync scrolling
    this.scrollSyncTimeout = null;
    this.scrollXAdjusted = false;
    this.scrollYAdjusted = false;
    // longer, "complicated" code for better scroll performance:
    this.scrollHandler = function() {

        // prevent double scrolling // following zooming
        if (Self.gridHeaderScrolling.on || Self.zooming) return;
        
        // update
        setTimeout(function() { Self.gridHeader.scrollLeft = Self.dataGridContainer.scrollLeft; }, 0);
        setTimeout(function() { Self.leftColumn.scrollTop = Self.dataGridContainer.scrollTop; }, 1);
        setTimeout(function() {
            Self.scrollPositionFix.style.left = Self.roundEm(Self.dataGridContainer.scrollLeft/Self.zoom) + "em";
            Self.scrollPositionFix.style.top = Self.dataGridContainer.scrollTop + "px";
        }, 2);
        
        // unimportant things
        setTimeout(function() {
            // remember if even scrolled
            Self.manualScroll = true;
            // adjustments
            if (!Self.scrollXAdjusted) {
                var clientHeightDiff = Self.leftColumn.clientHeight - Self.dataGridContainer.clientHeight;
                Self.gridHeaderScrollBarFix.style.width = clientHeightDiff + "px";
                Self.scrollXAdjusted = true;
            }
            if (!Self.scrollYAdjusted) {
                var clientHeightDiff = Self.leftColumn.clientHeight - Self.dataGridContainer.clientHeight;
                Self.leftColumnScrollBarFix.style.height = clientHeightDiff + "px";
                Self.scrollYAdjusted = true;
            }
        }, 10);
    };
    addDOMEvent(this.dataGridContainer, "scroll", this.scrollHandler);
    
    // --------------------------------------------------------------------------
    
    // draw members
    Self.memberDrag = {};
    for (var r = 0; r < maxRows; r++) {
        var row = this.rows[r];
        var className = "member" + (this.rowSelection[row.rowId] ? " rowSelected" : "");
        var div = newnode("div", 
            { top: (r * CG_ROW_HEIGHT) + "px" },
            { "className": className }
        );
        this.drawLeftColumn(row, r, div);
        this.leftColumn.appendChild(div);
        // remember node
        row.node = div;
        // add listener
        var wrapper = (function(Self, index) {
            return function(e) {
                Self.clickRow(index, e); 
            };
        })(Self, r);
        addDOMEvent(div, "mousedown", wrapper);
    }
    
    // --------------------------------------------------------------------------
    
    // clear
    this.container.innerHTML = "";
    // add
    this.container.appendChild(this.dataGridContainer);
    this.container.appendChild(this.leftColumn);
    this.container.appendChild(header);
    // restore zoom level
    this.gridHeader.className = this.zoom >= CG_HEAD_ZOOM ? "gridHeader zoom1" : "gridHeader zoom0";
    // restore scroll position
    this.dataGridContainer.scrollTop = Self.scrollPositionFix.offsetTop;
    this.dataGridContainer.scrollLeft = Self.scrollPositionFix.offsetLeft;
    this.gridHeader.scrollLeft = Self.scrollPositionFix.offsetLeft;
    this.invalidScrollPositions = false;
    
    //--------------------------------------------------------------------------
    
    // add "new" row (it might set the focus)
    var div = newnode("div", 
        { top: (maxRows * CG_ROW_HEIGHT) + "px" },
        { "className": "member new" }
    );
    this.leftColumn.appendChild(div);
    this.drawLeftNewColumn(div);
    addDOMEvent(div, "click", function() {
        Self.selectNone();
    });
    
    // add "drop zone" for members
    this.memberDropZone = newnode("div", 0, { className: "dropzone" }); 
    this.leftColumn.appendChild(this.memberDropZone);
    
    //--------------------------------------------------------------------------
    
    // add "new appointment" dragger
    this.newAppDragger = newnode("div", {
        position: "absolute", top: "0px", height: ((maxRows+1) * CG_ROW_HEIGHT) + "px",
        backgroundColor: "#555", opacity: "0.5", filter: "alpha(opacity=50)", zIndex: 20,
        visibility: "hidden"
    });
    this.dataGrid.appendChild(this.newAppDragger);
    
    // dragging logic
    this.newAppDrag = 0;
    this.newAppDragX = 0;
    this.newAppDragBase = 0;
    this.newAppDragTimeout = 0;
    addDOMEvent(this.dataGrid, "mousedown", function(e) {
        // ignore context menu
        if (e.button != 2) {
            // potential drag
            Self.newAppDrag = 1;
            // remember position
            Self.newAppDragBase = getCumulativeOffset(Self.dataGridContainer).left;
            Self.newAppDragX = Self.snapX2grid(e.clientX + Self.dataGridContainer.scrollLeft - Self.newAppDragBase);
        }
    });
    addDOMEvent(this.dataGrid, "mouseup", function(e) {
        // create appointment?
        if (Self.newAppDrag == 2) {
            // get positions
            var x1 = Self.newAppDragX;
            var x2 = Self.snapX2grid(e.clientX + Self.dataGridContainer.scrollLeft - Self.newAppDragBase);
            // swap?
            if (x1 > x2) {
            	var tmp = x1; x1 = x2; x2 = tmp;
            }
            if (x2 != x1) {
                var startDate = new Date(Self.pixel2time(x1, true));
                var endDate = new Date(Self.pixel2time(x2, true));
                Self.createAppointmentForSelectedMembers(startDate, endDate);
            }
        }
        // stop dragging
        Self.newAppDrag = 0;
        Self.newAppDragger.style.visibility = "hidden";
        Self.newAppDragger.style.left = "0px";
    });
    addDOMEvent(this.dataGrid, "mousemove", function(e) {
        if (Self.newAppDrag == 1) {
            // show new appointment dragger
            Self.newAppDragger.style.visibility = "visible";
            Self.newAppDrag = 2;
        }
        if (Self.newAppDrag == 2) {
            var x = Self.snapX2grid(e.clientX + Self.dataGridContainer.scrollLeft - Self.newAppDragBase);
            // set width
            var width = x-Self.newAppDragX;
            if (width > 0) {
            	Self.newAppDragger.style.left = Self.newAppDragX + "px";
            	Self.newAppDragger.style.width = Math.max(1, width) + "px";
            } else {
            	Self.newAppDragger.style.left = (Self.newAppDragX+width) + "px";
            	Self.newAppDragger.style.width = Math.max(1, -width) + "px";
            }
        }
    });
    
    
    
    // add mousewheel support
    // get proper event name for mousewheel
    var eventName = navigator.userAgent.indexOf('Gecko') > -1 && 
        navigator.userAgent.indexOf('KHTML') == -1 ? "DOMMouseScroll" : "mousewheel";
    // zoom via mouse wheel
    addDOMEvent(this.dataGridContainer, eventName, function(e) {
        e.delta = 0;
        e.deltaSign = 0;
        if (e.wheelDelta) {
            e.delta = e.wheelDelta;
        } else if (e.detail) {
            e.delta = -e.detail;
        }
        // signed delta
        if (e.delta != 0) {
            e.deltaSign = e.delta/Math.abs(e.delta);
        }
        // set new zoom level
        if (e.shiftKey) {
            Self.changeZoomLevel(Self.zoom + e.deltaSign * 20);
            stopEvent(e);
        } else {
            // always scroll horizontally (vertical scroll via mouse wheel is less important)
            var x = Self.snapX2grid(Self.dataGridContainer.scrollLeft, 1);
            x += Math.round(-e.deltaSign * 100 * CG_PX2EM * Self.zoom / 100 / 24);
            Self.dataGridContainer.scrollLeft = x;
            Self.gridHeader.scrollLeft = x;
            Self.manualScroll = true;
            stopEvent(e);
        }
    });
    
    // -------------------------------------------------------------------------
    
    // "double click to create appointment"
    addDOMEvent(this.dataGridContainer, "dblclick", function(e) {
        // get position
        var x = e.clientX + Self.dataGridContainer.scrollLeft - getCumulativeOffset(Self.dataGridContainer).left;
        var y = e.clientY + Self.dataGridContainer.scrollTop - getCumulativeOffset(Self.dataGridContainer).top;
        // convert pixel into time
        var t = Self.pixel2time(x, true);
        // get row index to get the member
        var r = Math.floor(y / CG_ROW_HEIGHT);
        // valid index?
        if (Self.rows[r]) {
            var row = Self.rows[r];
            // one member only
            var member = {}; member[row.rowId] = row;
            // adjust member selection
            Self.clickRow(r);
            // open "create appointment" dialog (duration: one hour)
            var now = (new Date(t)).getTime();
            var start = new Date(Math.floor(now/Self.ONE_HOUR)*Self.ONE_HOUR);
            var end = new Date(Math.ceil(now/Self.ONE_HOUR)*Self.ONE_HOUR);
            Self.createAppointment(start, end, member);
        }
    });
    
    // -------------------------------------------------------------------------
    
    // update controls
    this.updateZoomControls(this.zoom);
    
    // -------------------------------------------------------------------------
    
    // add appointment container
    this.appContainer = newnode("div", {
        position: "absolute", top: "0px", left: "0px",
        width: (maxCols * CG_PX2EM) + "em",
        height: (maxRows * CG_ROW_HEIGHT) + "px",
        zIndex: 10000
    });
    this.dataGrid.appendChild(this.appContainer);
    
    /* initialize hover */
    if (this.firstRun) {
        var hover = new Hover($("calendar_new_query"), OXAppointmentHover.getContent().node);
        hover.setSize(OXAppointmentHover.contentobject.node);
        /* event got triggered on each mouse move */
        hover.getTarget = function (node) {
            try {
                while (node) {
                    if (node.getAttribute("ox_object_id")) {
                        return node.parentNode ? node : null;
                    }
                    node = node.parentNode;
                }
            } catch (e) { /*see default implementation*/ }
        };
        /* event fired when getTarget found a node with valid attributes */
        hover.onShow = function (node) {
            OXAppointmentHover.actualHover = this;
            var ref       = node.getAttribute("ox_rowId");
            var folder_id = node.getAttribute("ox_folder_id");
            var object_id = node.getAttribute("ox_object_id");
            var rec_pos   = node.getAttribute("ox_rec_pos");
            if (object_id != undefined && folder_id != undefined) {
                // we have a folder and a object id so we can use the regular hover
                OXAppointmentHover.refillContent(object_id, folder_id, rec_pos);
            } else if (ref != undefined && Self.appointments && Self.appointments[ref]) {
                // we don't have read permission so we just show content we are allowed to
                var obj = { };
                for (var i in Self.appointments[ref]) {
                    if (object_id == Self.appointments[ref][i].id) {
                        obj = clone(Self.appointments[ref][i]);
                        break;
                    }
                }
                // disable tabs we don't need
                OXAppointmentHover.slider.getTabById("participants").show(false);
                OXAppointmentHover.slider.getTabById("attachments").show(false);
                OXAppointmentHover.slider.getTabById("others").show(false);
                OXAppointmentHover.refillContentByObject(obj);
            }
        }
        hover.onHide = function (node) {
          // re-enable tabs for later use
          OXAppointmentHover.slider.getTabById("participants").show(true);
          OXAppointmentHover.slider.getTabById("attachments").show(true);
          OXAppointmentHover.slider.getTabById("others").show(true);
        }
        calendarhovers["teamday"] = hover;
        // must be fired once here to enable it, later a change view does it for us
        calendarhovers["teamday"].enable();
        this.hoverInitialized = true;
    }
    
    // required to get the right class names for different appointment types
    var shownAs2Class = { "1": "reserved", "2": "temporary", "3": "absent", "4": "free" };
    
    // get and draw appointments (async.; should be the last thing to do here)
    this.getAppointments(function() { try {
        // get microseconds of start date
        var gridStart = Self.gridIntervalStart.getTime();
        var gridEnd = Self.gridIntervalEnd.getTime();
        // loop rows
        for (var r = 0; r < Self.numRows(); r++) {
            var row = Self.rows[r];
            // look up appointments
            var appointments = Self.appointments[row.rowId];
            var userId = row.id;
            // loop appointments
            for (var i = 0; i < appointments.length; i++) {
                var app = appointments[i];
                var canRead = typeof(app.folder_id) != "undefined";
                var canWrite = false;
                // check object write permissions
                if (canRead && oMainFolderTree.cache.find_folder(app.folder_id)) {
                	var permission = oMainFolderTree.cache.find_folder(app.folder_id).oxfolder.data.own_rights;
                	canWrite = computePerm(permission, 0) >= 2;
                	// @todo: i'm not sure if it's enough to check the own permissions. i assume it fails
                	// as soon as you are member of a group. in this case the permissions array has to 
                	// be checked -> similar to function 'permissionsToViewObjects'.
                }
                var innerDiv = null;
                if (!Self.showDetails) {
                    // bars only
                    var className = "appointment bar " + shownAs2Class[app.shown_as];
                    innerDiv = newnode("div", null, { "className": className });
                } else {
                    // show details
                    var wholeday = !!app.full_time;
                    // create inner div
                    var className = "appointment " + shownAs2Class[app.shown_as] +
                        // read permission?
                        (canWrite ? "" : " norights") +
                        // all day?
                        (wholeday ? " wholeday": "") +
                        // color label?
                        (app.color_label ? " hasColorLabel colorLabel"+app.color_label : "");
                    var appTitle = app.title || "\u2014"; // aka &mdash;
                    // is owner?
                    if (app.created_by == userId) {
                        appTitle += " \u2022"; // aka &bull;
                    }
                    innerDiv = newnode("div", null, { "className": className }, [
                        // add title
                        newnode("b", 0, 0, [newtext(appTitle)]),
                        // add time interval / wholeday
                        newnode("br"),
                        newtext(wholeday ? _("All day") : /* i18n */
                            formatDate(app.start_date, "time") + " - " +
                            formatDate(app.end_date, "time")
                        ),
                        newtext(", "), //newnode("br"),
                        // start date
                        newtext(formatDate(app.start_date, "dateday") +
                            // add end date?
                            (app.end_date-app.start_date > Self.ONE_DAY ? 
                            " - " + formatDate(app.end_date, "dateday") : "")
                        ),
                        // add blank line
                        newnode("br"),
                        newtext("\u00A0") // aka &nbsp;
                    ]);
                }
                // set required attributes to enable hovers
                if (canRead) {
                	innerDiv.setAttribute("ox_folder_id", app.folder_id);
                }
                innerDiv.setAttribute("ox_object_id", app.id);
                if (app.recurrence_position) {
                	innerDiv.setAttribute("ox_rec_pos", app.recurrence_position);
                }
                innerDiv.setAttribute("ox_rowId", row.rowId);
                // now convert microseconds to "em"!
                var top = (r * CG_ROW_HEIGHT + 5 + (wholeday ? 0 : -3));
                var left = ((app.start_date - gridStart) * CG_PX2EM / Self.ONE_DAY);
                var width = (((app.end_date < gridEnd ? app.end_date : gridEnd) - app.start_date) * CG_PX2EM / Self.ONE_DAY);
                var height =  (CG_ROW_HEIGHT - 10 - 1 + (wholeday ? 0 : 6));
                // create outer div
                var div = newnode("div", {
                    "position": "absolute", "top": top + "px", "left": Self.roundEm(left) + "em",
                    "width": Self.roundEm(width) + "em", "height": height + "px",
                    zIndex: (wholeday ? 10001 : 10002)
                }, null, [innerDiv]);
                // add to dom
                Self.appContainer.appendChild(div);
                // add listener
                if (canWrite) {
                    // observe by selection
                    var id = app.folder_id+"."+app.id+"."+app.recurrence_position+"@"+row.rowId;
                    Self.appointmentSelection.observe(innerDiv, id, app);
                    // add double click edit
                    var wrapper = (function(Self, app) {
                        return function(e) {
                            // copy
                            selectedAppointment = app;
                            stopEvent(e);
                            triggerEvent("OX_Calendar_Edit");
                        };
                    })(Self, app);
                    addDOMEvent(innerDiv, "dblclick", wrapper);
                    
                    // add context menu
                    addDOMEvent(innerDiv, "contextmenu", function(e) {
                        globalContextMenus.teamview.display(
                           e.clientX, e.clientY, Self.appointmentSelection.fetch()
                        );
                    });
                }
            }
        }
        
        // "deselect all" click
        addDOMEvent(Self.dataGridContainer, "click", function(e) {
            Self.appointmentSelection.clear();
            stopEvent(e);
        });
        
        // scroll to "now"
        if (!Self.manualScroll) {
            Self.scroll2Date(new Date(), Self.ONE_HOUR);
        }
        
        Self.firstRun = false;
        
    } catch(e) { if (console && console.error) console.error("Unexpected", e); }
    });
}

CalendarGrid.prototype.clearPartial = function() {
    // clear
    if (this.dataGrid && this.appContainer && this.leftColumn) {
        // hide first (responsiveness)
        this.dataGrid.style.visibility = "hidden";
        this.appContainer.style.visibility = "hidden";
        this.leftColumn.style.visibility = "hidden";
        // clear
        this.dataGrid.innerHTML = "";
        this.appContainer.innerHTML = "";
        this.leftColumn.innerHTML = "";
        // show
        this.dataGrid.style.visibility = "visible";
        this.appContainer.style.visibility = "visible";
        this.leftColumn.style.visibility = "visible";
    }
}

CalendarGrid.prototype.drawLeftColumn = function(row, index, parentNode) {
    
    var Self = this;
    
    // icon
    var src = "";
    switch (row.data.type) {
        default: /* no break */
        case 1: src = "img/calendar/user.gif"; break;
        case 3: src = "img/calendar/ressourcen.gif"; break;
    }
    var icon = newnode("img", 0, {
        src: getFullImgSrc(src), alt: ""
    });
    parentNode.appendChild(icon);
    
    // remove member "X"
    var remove = newnode("div", 0, { className: "memberX" }, [newtext("\u00d7")]); // &times;
    parentNode.appendChild(remove);
    addDOMEvent(remove, "click", function(e) {
        // trigger event
        // this row is already selected since the selection routine waits for mousedown
        triggerEvent('OX_Calendar_Teammember_Remove');
    });
    
    // text
    // remove surrounding quoting marks
    row.display_name = row.display_name.replace(/(^"|"$)/g, '');
    // add node
    parentNode.appendChild(newtext(row.display_name));
    
    // add dragger
    addDOMEvent(parentNode, "mousedown", function(e) {
        if (!Self.memberDrag.on) {
            // stand by
            Self.memberDrag.standBy = true;
            Self.memberDrag.startY = e.clientY - parentNode.offsetTop;
            // define listener
            Self.memberDrag.mover = function(e) {
                if (!Self.memberDrag.on && Self.memberDrag.standBy) {
                    Self.memberDrag.on = true;
                    Self.memberDrag.standBy = false;
                    // make clone
                    Self.memberDrag.clone = parentNode.cloneNode(false);
                    // adjust node design
                    parentNode.className += " drag";
                    // add clone
                    parentNode.parentNode.appendChild(Self.memberDrag.clone);
                    // adjust clone design
                    Self.memberDrag.clone.className += " clone";
                    // show drop zone
                    Self.memberDropZone.style.visibility = "visible";
                }
                if (Self.memberDrag.on) {
                    // move node
                    var y = Math.max(0, e.clientY - Self.memberDrag.startY);
                    parentNode.style.top = y + "px";
                    // move drop zone
                    var index = Math.min(Math.round(y/CG_ROW_HEIGHT), Self.numRows());
                    Self.memberDropZone.style.top = (index * CG_ROW_HEIGHT-2) + "px";
                }
            };
            // add event
            Self.memberDrag.stopper = handler;
            addDOMEvent(document, "mouseup", Self.memberDrag.stopper);
            addDOMEvent(document, "mousemove", Self.memberDrag.mover);
        }
    });
    var handler = function(e) {
        // get state
        var wasOn = Self.memberDrag.on;
        // remove event
        removeDOMEvent(document, "mouseup", Self.memberDrag.stopper);
        removeDOMEvent(document, "mousemove", Self.memberDrag.mover);
        // stop
        Self.memberDrag.on = false;
        Self.memberDrag.standBy = false;
        // hide drop zone
        Self.memberDropZone.style.visibility = "hidden";
        // get last position
        var y = Math.max(0, e.clientY - Self.memberDrag.startY);
        if (wasOn) {
            // determine indexes
            var oldIndex = row.index;
            var index = Math.min(Math.round(y/CG_ROW_HEIGHT), Self.numRows()+1);
            // anything to do?
            if (Math.abs(oldIndex-index) >= 1) {
                // remove row
                Self.removeById(row.rowId);
                // adjust index
                index = oldIndex < index ? index-1 : index;
                // insert row
                Self.insertAt(row, index);
            }
            // update
            Self.selectNone();
            Self.clearPartial();
            Self.update();
        }
    };
}

CalendarGrid.prototype.drawLeftNewColumn = function(parentNode) {
    
    var Self = this;
    
    // icon
    var icon = newnode("img", 0, {
        src: getFullImgSrc("img/menu/add_member_participant.gif"), alt: ""
    });
    parentNode.appendChild(icon);
    // open dialog when clicking the icon
    addDOMEvent(icon, "click", function(e) {
        triggerEvent('OX_Calendar_Teammember_Add');
    });
    
    // text
    var inputField = newnode("input",
        { width: "150px" }, { value: "" }
    );
    inputField.type = "text";
    parentNode.appendChild(inputField);
    
    // set focus if visible
    if (this.isShowing()) {
        inputField.focus();
        inputField.select();
        // better safe then sorry (IE)
        Self.leftColumn.scrollTop = Self.dataGridContainer.scrollTop;
        setTimeout(function() {
            Self.leftColumn.scrollTop = Self.dataGridContainer.scrollTop;
        }, 50); // yes, twice!
    }
    
    // add auto complete feature (in place editing)
    var autoCom = new AutoComplete(inputField, this.autoCompleteContainer, 25, configGetKey("minimumSearchCharacters") || 2, 100);
    autoCom.autoAdjustHeight = false;
    autoCom.showDisplayNamesOnly = true;
    autoCom.showSystemUsersOnly = true;
    autoCom.includeResources = true;
    // override onKeyEnter
    autoCom.onSelect = function (elem, st) {
        // get name & id
        var display_name = elem.getAttribute("oval");
        var member = autoCom.nameIndex[display_name];
        if (member) {
            // create member
            var row = new CalendarGridRow(member.type, member.id, member.display_name);
            row.data = member;
            Self.add(row);
            // update grid
            Self.clearPartial();
            Self.update();
        }
    };
}

CalendarGrid.prototype.isShowing = function() {
    var node = this.dataGrid;
    do {
        if (node.style.visibility == "hidden" || node.style.display == "none") {
            return false;
        }
        // climb up
        node = node.parentNode;
    } while (node != document.body);
    return true;
}

function CalendarGridRow(type, id, name, data) {
    this.type = type;
    this.id = id
    // sequenced "id:type" to allow cooperation with team week:
    this.rowId = id + ":" + type;
    this.display_name = name || "New member";
    // keep original data (might be useful in future)
    this.data = data || {};
    // reference to DOM node / makes selection of rows much easier
    this.node = null;
    // current index
    this.index = 0;
}

//------------------------------------------------------------------------------

var simpleSelectionHash = {};

function getSimpleSelection(id) {
    // exists?
    if (typeof(simpleSelectionHash[id]) == "undefined") {
        simpleSelectionHash[id] = new SimpleSelection();
    }
    return simpleSelectionHash[id];
}

function SimpleSelection() {
    this.multiple = true;
    this.domNodes = {};
    this.selectedItems = {};
    this.observedItems = {};
    this.changedEvent = null;
    this.lastSelectedItem = null;
}

SimpleSelection.prototype.setMultiple = function(flag) {
    this.multiple = !!flag;
};

SimpleSelection.prototype.setChangedEvent = function(eventName) {
    this.changedEvent = eventName;
    this.triggerChangedEvent();
};

SimpleSelection.prototype.triggerChangedEvent = function() {
    if (this.changedEvent) {
        triggerEvent(this.changedEvent, this.getSelectedItems());
    }
};

SimpleSelection.prototype.observe = function(DOMNode, id, item) {
    if (DOMNode) {
        // add listener
        var wrapper = (function(Self, id) {
             return function(e) {
                 Self.click(id, e.ctrlKey, e.shiftKey);
             };
        })(this, id);
        addDOMEvent(DOMNode, "click", wrapper);
        addDOMEvent(DOMNode, "click", stopEvent);
        // add second listener (context menu)
        var contextWrapper = (function(Self, id) {
            return function(e) {
                if (!Self.isSelected(id)) {
                    Self.click(id, false, false);
                }
            };
        })(this, id);
        addDOMEvent(DOMNode, "contextmenu", contextWrapper);
        // add to observation list
        this.observedItems[id] = item;
        // add to dom node list
        this.domNodes[id] = DOMNode;
    }
};

SimpleSelection.prototype.click = function(id, multiple, range) {
    // multiple?
    if (!this.multiple || !multiple) {
        // clear selection
        this.clear();
    }
    // range?
    if (range && this.lastSelectedItem) {
        // select range
        this.selectRange(this.lastSelectedItem, id);
    } else {
        // add single item to selection
        this.toggle(id);
        // remember last item
        this.lastSelectedItem = id;
    }
    // trigger event
    this.triggerChangedEvent();
};

SimpleSelection.prototype.selectRange = function(lastId, newId) {
    var Self = this;
    // get grid
    getCalendarGridIfExists("teamview", function(grid) {
        // get user id
        var fromUser = lastId.match(/@(\d+:\d+)$/);
        var toUser = newId.match(/@(\d+:\d+)$/);
        if (fromUser.length && toUser.length) {
            var members = [];
            // get indexes
            var fromIndex = grid.rowIndex[fromUser[1]];
            var toIndex = grid.rowIndex[toUser[1]];
            // swap?
            if (fromIndex > toIndex) {
                var tmp = fromIndex; fromIndex = toIndex; toIndex = tmp;
            }
            // loop through rows
            for (var i = fromIndex; i <= toIndex; i++) {
                members.push(grid.rows[i].rowId);
            }
            // create regex
            var regex = new RegExp("@(" + members.join("|") + ")$");
            // get start time
            var fromDate = Self.observedItems[lastId].start_date;
            var toDate = Self.observedItems[newId].start_date;
            // swap?
            if (fromDate > toDate) {
                var tmp = fromDate; fromDate = toDate; toDate = tmp;
            }
            // now loop through appointments
            for (var id in Self.observedItems) {
                var item = Self.observedItems[id];
                // check item
                if (item.start_date >= fromDate && item.start_date <= toDate && regex.test(id) ) {
                    Self.select(id);
                }
            }
        }
    });
};

SimpleSelection.prototype.select = function(id) {
    this.selectedItems[id] = this.observedItems[id];
    this.selectedItems[id].selectionId = id;
    var node = this.domNodes[id];
    if (node) {
        node.className += " isSelected";
    }
};

SimpleSelection.prototype.deselect = function(id) {
    delete this.selectedItems[id];
    var node = this.domNodes[id];
    if (node) {
        node.className = node.className.replace(/\s?isSelected/, "");
    }
};

SimpleSelection.prototype.isSelected = function(id) {
    return typeof(this.selectedItems[id]) != "undefined";
};

SimpleSelection.prototype.toggle = function(id) {
    if (this.isSelected(id)) {
        this.deselect(id);
    } else {
        this.select(id);
    }
};

SimpleSelection.prototype.clear = function() {
    // deselect items
    for (var id in this.selectedItems) {
        this.deselect(id);
    }
    // trigger event
    this.triggerChangedEvent();
};

SimpleSelection.prototype.numSelected = function() {
    var count = 0;
    for (var id in this.selectedItems) count++;
    return count;
};

SimpleSelection.prototype.getSelectedItems = function() {
    var list = [];
    for (var id in this.selectedItems) {
        list.push(this.selectedItems[id]);
    }
    return list;
};

SimpleSelection.prototype.getSelection = function() {
    var list = [];
    for (var id in this.selectedItems) {
        list.push(id);
    }
    return list;
};

/* to work with context menus */
SimpleSelection.prototype.fetch = function() {
    return this;
};

//------------------------------------------------------------------------------
