/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2006 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.mailaccount;

import java.sql.Connection;

/**
 * {@link UnifiedINBOXManagement} - Management for Unified INBOX accounts.
 * 
 * @author <a href="mailto:thorben.betten@open-xchange.com">Thorben Betten</a>
 */
public interface UnifiedINBOXManagement {

    /**
     * The Unified INBOX protocol name.
     */
    public static final String PROTOCOL_UNIFIED_INBOX = "unifiedinbox";

    /**
     * The Unified INBOX name.
     */
    public static final String NAME_UNIFIED_INBOX = "Unified INBOX";

    /**
     * Creates the Unified INBOX account for given user in specified context.
     * 
     * @param userId The user ID
     * @param contextId The context ID
     * @throws MailAccountException If creating the Unified INBOX account fails for given user in specified context
     */
    public void createUnifiedINBOX(int userId, int contextId) throws MailAccountException;

    /**
     * Creates the Unified INBOX account for given user in specified context.
     * 
     * @param userId The user ID
     * @param contextId The context ID
     * @param con The connection to use
     * @throws MailAccountException If creating the Unified INBOX account fails for given user in specified context
     */
    public void createUnifiedINBOX(int userId, int contextId, Connection con) throws MailAccountException;

    /**
     * Deletes the Unified INBOX account for given user in specified context.
     * 
     * @param userId The user ID
     * @param contextId The context ID
     * @throws MailAccountException If deleting the Unified INBOX account fails for given user in specified context
     */
    public void deleteUnifiedINBOX(int userId, int contextId) throws MailAccountException;

    /**
     * Deletes the Unified INBOX account for given user in specified context.
     * 
     * @param userId The user ID
     * @param contextId The context ID
     * @param con The connection to use
     * @throws MailAccountException If deleting the Unified INBOX account fails for given user in specified context
     */
    public void deleteUnifiedINBOX(int userId, int contextId, Connection con) throws MailAccountException;

    /**
     * Checks if the Unified INBOX account is enabled for given user in specified context.
     * <p>
     * The Unified INBOX account is considered to be enabled if at least one account indicates its subscription to Unified INBOX.
     * 
     * @param userId The user ID
     * @param contextId The context ID
     * @return <code>true</code> if the Unified INBOX account is enabled for given user in specified context; otherwise <code>false</code>
     * @throws MailAccountException If checking Unified INBOX account's enabled status fails
     */
    public boolean isEnabled(int userId, int contextId) throws MailAccountException;

    /**
     * Checks if the Unified INBOX account is enabled for given user in specified context.
     * <p>
     * The Unified INBOX account is considered to be enabled if at least one account indicates its subscription to Unified INBOX.
     * 
     * @param userId The user ID
     * @param contextId The context ID
     * @param con The connection to use
     * @return <code>true</code> if the Unified INBOX account is enabled for given user in specified context; otherwise <code>false</code>
     * @throws MailAccountException If checking Unified INBOX account's enabled status fails
     */
    public boolean isEnabled(int userId, int contextId, Connection con) throws MailAccountException;

    /**
     * Gets the ID of the mail account denoting the Unified INBOX account.
     * 
     * @param userId The user ID
     * @param contextId The context ID
     * @return The ID of the Unified INBOX account or <code>-1</code> if none found
     * @throws MailAccountException If detecting ID of the Unified INBOX account fails
     */
    public int getUnifiedINBOXAccountID(int userId, int contextId) throws MailAccountException;

    /**
     * Gets the ID of the mail account denoting the Unified INBOX account.
     * 
     * @param userId The user ID
     * @param contextId The context ID
     * @param con The connection to use
     * @return The ID of the Unified INBOX account or <code>-1</code> if none found
     * @throws MailAccountException If detecting ID of the Unified INBOX account fails
     */
    public int getUnifiedINBOXAccountID(int userId, int contextId, Connection con) throws MailAccountException;

}
