/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2006 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.database.internal;

import org.apache.commons.logging.LogFactory;
import org.apache.commons.logging.Log;
import com.openexchange.caching.CacheService;
import com.openexchange.config.ConfigurationService;
import com.openexchange.database.DBPoolingException;
import com.openexchange.management.ManagementService;
import com.openexchange.timer.TimerService;

/**
 * {@link Initialization}
 *
 * @author <a href="mailto:marcus.klein@open-xchange.com">Marcus Klein</a>
 */
public final class Initialization {

    private static final Log LOG = LogFactory.getLog(Initialization.class);

    private static final Initialization SINGLETON = new Initialization();

    private ManagementService managementService;

    private CacheService cacheService;

    private Configuration configuration;

    private Pools pools;

    private AssignmentStorage assignmentStorage;

    private ConfigDatabaseServiceImpl configDatabaseService;

    /**
     * Prevent instantiation.
     */
    private Initialization() {
        super();
    }

    public static final Initialization getInstance() {
        return SINGLETON;
    }

    public boolean isStarted() {
        return null != configuration;
    }

    public void start(ConfigurationService configurationService, TimerService timerService) throws DBPoolingException {
        configuration = new Configuration();
        configuration.readConfiguration(configurationService);
        // Setting up database connection pools.
        pools = new Pools(timerService);
        pools.start(configuration);
        if (null != managementService) {
            pools.setManagementService(managementService);
        }
        // Setting up assignment storage.
        assignmentStorage = new AssignmentStorage();
        if (null != cacheService) {
            assignmentStorage.setCacheService(cacheService);
        }
        // Initialize service for connections to config database.
        configDatabaseService = new ConfigDatabaseServiceImpl();
        ConfigDatabaseServiceImpl.setForceWrite(ConnectionPool.DEFAULT_CONFIG.forceWriteOnly); // FIXME: This is most certainly not correct.
        ConfigDatabaseServiceImpl.setPools(pools);
        pools.setConnectionDataStorage(new ConnectionDataStorage(configDatabaseService));
        ConfigDatabaseServiceImpl.setAssignmentStorage(assignmentStorage);
        assignmentStorage.setConfigDatabaseService(configDatabaseService);

        Server.setConfigDatabaseService(configDatabaseService);
        Server.start(configurationService);
        try {
            LOG.info("Resolved server name \"" + Server.getServerName() + "\" to identifier " + Server.getServerId());
        } catch (DBPoolingException e) {
            LOG.warn("Resolving server name to an identifier failed. This is normal until a server has been registered.", e);
        }

        DatabaseServiceImpl.setConfigDatabaseService(configDatabaseService);
        DatabaseServiceImpl.setPools(pools);
        DatabaseServiceImpl.setAssignmentStorage(assignmentStorage);
        DatabaseServiceImpl.setForceWrite(ConnectionPool.DEFAULT_CONFIG.forceWriteOnly); // FIXME: This is most certainly not correct.
    }

    public void stop() {
        DatabaseServiceImpl.setAssignmentStorage(null);
        assignmentStorage.stop();
        assignmentStorage = null;
        DatabaseServiceImpl.setPools(null);
        pools.stop();
        pools = null;
        configuration.clear();
        configuration = null;
    }

    public void setManagementService(ManagementService service) {
        this.managementService = service;
        if (null != pools) {
            pools.setManagementService(service);
        }
    }

    public void removeManagementService() {
        this.managementService = null;
        if (null != pools) {
            pools.removeManagementService();
        }
    }

    public void setCacheService(CacheService service) {
        this.cacheService = service;
        if (null != pools) {
            assignmentStorage.setCacheService(service);
        }
    }

    public void removeCacheService() {
        this.cacheService = null;
        if (null != pools) {
            assignmentStorage.removeCacheService();
        }
    }
}
