/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.usm.session.storage.sql.impl;

import java.sql.Clob;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.List;
import java.util.Set;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import com.openexchange.usm.api.database.CachedDBEntry;
import com.openexchange.usm.api.database.DatabaseAccess;
import com.openexchange.usm.api.database.EncapsulatedConnection;
import com.openexchange.usm.api.database.StorageAccessException;
import com.openexchange.usm.api.exceptions.USMStorageException;
import com.openexchange.usm.api.session.Session;
import com.openexchange.usm.api.session.storage.PersistentSyncStateStorage;
import com.openexchange.usm.session.impl.USMSessionManagementErrorCodes;
import com.openexchange.usm.util.Toolkit;

/**
 * {@link PersistentSyncStateStorageSQL}
 * 
 * @author <a href="mailto:ioannis.chouklis@open-xchange.com">Ioannis Chouklis</a>
 */
public class PersistentSyncStateStorageSQL implements PersistentSyncStateStorage {

    private static final Log LOG = LogFactory.getLog(PersistentSyncStateStorageSQL.class);

    private final DatabaseAccess dbAccess;

    private static final int MAX_RETRIES_SYNC_STATE_INSERT_IN_DB = 10;

    /**
     * Initializes a new {@link PersistentSyncStateStorageSQL}.
     * 
     * @param dba
     */
    public PersistentSyncStateStorageSQL(DatabaseAccess dba) {
        dbAccess = dba;
    }

    /*
     * (non-Javadoc)
     * @see com.openexchange.usm.api.session.PersistentSyncStateStorage#store(com.openexchange.usm.api.session.Session, java.lang.String, java.lang.String, long, long, long)
     */
    @Override
    public long store(Session session, String objectID, String data, long oldTimestamp, long newTimestamp, long timestampToKeep) throws StorageAccessException, USMStorageException {
        boolean tooManyTimestamps = oldTimestamp != timestampToKeep;
        EncapsulatedConnection con = null;
        PreparedStatement statement = null;
        try {
            con = dbAccess.getWritable(session.getContextId());
            // make delete/insert
            String deleteStatement = tooManyTimestamps ? SyncStateStorageSQLStatements.COMPLEX_SYNCSTATE_DELETE : SyncStateStorageSQLStatements.SIMPLE_SYNCSTATE_DELETE;
            statement = con.prepareStatement(deleteStatement);
            statement.setInt(1, session.getContextId());
            statement.setInt(2, session.getSessionId());
            statement.setString(3, objectID);
            statement.setLong(4, oldTimestamp);
            if (tooManyTimestamps)
                statement.setLong(5, timestampToKeep);
            statement.executeUpdate();
            statement.close();
            statement = null;
            int retries = 0;
            int executeUpdateAffectedRows = 0;
            do {
                statement = con.prepareStatement(dbAccess.supportsInsertIgnoreStatement() ? SyncStateStorageSQLStatements.FULL_INSERT_IGNORE : SyncStateStorageSQLStatements.FULL_INSERT);
                statement.setInt(1, session.getContextId());
                statement.setInt(2, session.getSessionId());
                statement.setString(3, objectID);
                statement.setLong(4, newTimestamp);
                statement.setString(5, data);
                executeUpdateAffectedRows = statement.executeUpdate();
                if (executeUpdateAffectedRows == 0) {
                    if (!dbAccess.supportsInsertIgnoreStatement())
                        throw new SQLException("Insert did not produce any results");
                    retries++;
                    newTimestamp = newTimestamp + 1L;
                    if (LOG.isDebugEnabled())
                        LOG.debug("Retry sync state insert for timestamp: " + newTimestamp + " sessionId: " + session.getSessionId() + " in context: " + session.getContextId());
                }
                statement.close();
                statement = null;
            } while (retries < MAX_RETRIES_SYNC_STATE_INSERT_IN_DB && executeUpdateAffectedRows == 0);

            if (executeUpdateAffectedRows == 0) // throw exception if after 10 retries still no insert was possible
                throw new SQLException("Insert did not produce any results");

        } catch (SQLException e) {
            throw new USMStorageException(USMSessionManagementErrorCodes.DATA_STORAGE_ERROR_NUMBER1,session + " Error storing cache data for " + objectID, e);
        } finally {
            Toolkit.close(statement);
            Toolkit.close(con);
        }
        return newTimestamp;
    }

    /*
     * (non-Javadoc)
     * @see com.openexchange.usm.api.session.storage.PersistentSyncStateStorage#get(com.openexchange.usm.api.session.Session, java.lang.String, int)
     */
    @Override
    public List<CachedDBEntry> get(Session session, String objectID, int maxSyncStatesInDB) throws StorageAccessException, USMStorageException {
        List<CachedDBEntry> dbList = new ArrayList<CachedDBEntry>();
        EncapsulatedConnection con = null;
        PreparedStatement statement = null;
        try {
            con = dbAccess.getReadOnly(session.getContextId());
            statement = con.prepareStatement(SyncStateStorageSQLStatements.SELECT_CACHED_DATA + SyncStateStorageSQLStatements.UNIQUE_OBJECT_IDENTIFIER + SyncStateStorageSQLStatements.SELECT_CACHED_DATA_ORDER_BY);
            statement.setInt(1, session.getContextId());
            statement.setInt(2, session.getSessionId());
            statement.setString(3, objectID);
            statement.setInt(4, maxSyncStatesInDB);
            ResultSet result = statement.executeQuery();
            while (result.next()) {
                long timestamp = result.getLong(1);
                Clob element = result.getClob(2);
                dbList.add(new CachedDBEntry(timestamp, element.getSubString(1L, (int) element.length())));
            }
        } catch (SQLException e) {
            String errorMessage = session + " Error retrieving cached data for " + objectID;
            LOG.error(errorMessage, e);
            throw new USMStorageException(USMSessionManagementErrorCodes.DATA_STORAGE_ERROR_NUMBER4, errorMessage, e);
        } finally {
            Toolkit.close(statement);
            Toolkit.close(con);
        }
        return dbList;
    }

    /*
     * (non-Javadoc)
     * @see com.openexchange.usm.api.session.PersistentSyncStateStorage#remove(com.openexchange.usm.api.session.Session, java.lang.String[])
     */
    @Override
    public void remove(Session session, String... objectIDs) throws USMStorageException {
        executeUpdate(session,
            "",
            "Error while deleting cached information for folders",
            completeComplexStatement(
                SyncStateStorageSQLStatements.DELETE + SyncStateStorageSQLStatements.UNIQUE_SESSION_IDENTIFIER + " and ObjectID IN (",
                objectIDs.length),
            objectIDs);
    }

    /*
     * (non-Javadoc)
     * @see com.openexchange.usm.api.session.PersistentSyncStateStorage#retain(com.openexchange.usm.api.session.Session, java.util.Set)
     */
    @Override
    public List<String> getRemainingObjectIDs(Session session, Set<String> objectIDSet) throws USMStorageException {
        EncapsulatedConnection con = null;
        PreparedStatement statement = null;
        List<String> toDelete = new ArrayList<String>();
        try {
            con = dbAccess.getReadOnly(session.getContextId());
            statement = con.prepareStatement(SyncStateStorageSQLStatements.SELECT_OBJECT_IDS);
            statement.setInt(1, session.getContextId());
            statement.setInt(2, session.getSessionId());
            ResultSet result = statement.executeQuery();
            while (result.next()) {
                String id = result.getString(1);
                if (!objectIDSet.contains(id))
                    toDelete.add(id);
            }
        } catch (StorageAccessException e) {
            String errorMessage = session + " Error while deleting cached information for unused folders";
            LOG.error(errorMessage, e);
            throw new USMStorageException(USMSessionManagementErrorCodes.DATA_STORAGE_ERROR_NUMBER2, errorMessage, e);
        } catch (SQLException e) {
            String errorMessage = session + " Error retrieving list of folders stored in DB cache";
            LOG.error(errorMessage, e);
            throw new USMStorageException(USMSessionManagementErrorCodes.DATA_STORAGE_ERROR_NUMBER6, errorMessage, e);
        } finally {
            Toolkit.close(statement);
            Toolkit.close(con);
        }
        return toDelete;
    }

    /*
     * (non-Javadoc)
     * @see com.openexchange.usm.api.session.PersistentSyncStateStorage#remapStates(com.openexchange.usm.api.session.Session, java.lang.String, java.lang.String)
     */
    @Override
    public void remapStates(Session session, String oldObjectID, String newObjectID) throws StorageAccessException, USMStorageException {
        EncapsulatedConnection con = null;
        PreparedStatement statement = null;
        try {
            String updateStatement = SyncStateStorageSQLStatements.UPDATE;
            con = dbAccess.getWritable(session.getContextId());
            statement = con.prepareStatement(updateStatement);
            statement.setString(1, newObjectID);
            statement.setString(2, oldObjectID);
            statement.setInt(3, session.getContextId());
            statement.setInt(4, session.getSessionId());
            if (statement.executeUpdate() == 0) {
                if (LOG.isDebugEnabled()) {
                    LOG.debug(session + " No states were remapped for object:  " + oldObjectID);
                }
            }
        } catch (SQLException e) {
            throw new USMStorageException(
                USMSessionManagementErrorCodes.DATA_STORAGE_ERROR_NUMBER5,
                session + " Error re-storing cache data for " + oldObjectID,
                e);
        } finally {
            Toolkit.close(statement);
            Toolkit.close(con);
        }
    }

    /*
     * (non-Javadoc)
     * @see com.openexchange.usm.api.session.PersistentSyncStateStorage#purgeSession(com.openexchange.usm.api.session.Session)
     */
    @Override
    public void purgeSession(Session session) throws USMStorageException {
        executeUpdate(session, "", "Error while deleting cached information for one session", SyncStateStorageSQLStatements.DELETE + SyncStateStorageSQLStatements.UNIQUE_SESSION_IDENTIFIER);
    }

    /**
     * Execute a SQL statement, the first 2 parameters must be user and session 
     * and doesn't need be specified as parameters to this method. The remaining parameters 
     * are filled from the provided values.
     * 
     * @param warnMessage
     * @param errorMessage
     * @param command
     * @param parameters
     * @throws USMStorageException
     */
    private void executeUpdate(Session session, String warnMessage, String errorMessage, String command, String... parameters) throws USMStorageException {
        EncapsulatedConnection con = null;
        try {
            con = dbAccess.getWritable(session.getContextId());
            executeUpdate(session, con, warnMessage, errorMessage, command, parameters);
        } catch (StorageAccessException e) {
            LOG.error(String.valueOf(session) + ' ' + errorMessage, e);
        } finally {
            Toolkit.close(con);
        }
    }

    /**
     * @param session
     * @param con
     * @param warnMessage
     * @param errorMessage
     * @param command
     * @param parameters
     * @throws USMStorageException
     */
    private static void executeUpdate(Session session, EncapsulatedConnection con, String warnMessage, String errorMessage, String command, String... parameters) throws USMStorageException {
        PreparedStatement statement = null;
        try {
            statement = con.prepareStatement(command);
            statement.setInt(1, session.getContextId());
            statement.setInt(2, session.getSessionId());
            int i = 3;
            for (String param : parameters)
                statement.setString(i++, param);
            if (statement.executeUpdate() == 0 && warnMessage.length() > 0)
                LOG.warn(String.valueOf(session) + ' ' + warnMessage);
        } catch (SQLException e) {
            LOG.error(String.valueOf(session) + ' ' + errorMessage, e);
            throw new USMStorageException(USMSessionManagementErrorCodes.DATA_STORAGE_ERROR_NUMBER3, session + ": " + errorMessage, e);
        } finally {
            Toolkit.close(statement);
        }
    }

    private static String completeComplexStatement(String prefix, int length) {
        StringBuilder sb = new StringBuilder(prefix);
        for (int i = 0; i < length; i++) {
            if (i > 0)
                sb.append(',');
            sb.append('?');
        }
        return sb.append(')').toString();
    }
}
