/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.ant.data;

import java.io.File;
import java.io.IOException;
import java.util.Collections;
import java.util.Enumeration;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.Stack;
import java.util.jar.JarEntry;
import java.util.jar.JarFile;
import org.apache.tools.ant.BuildException;

/**
 * Abstract super class for all supported types of presentation of a bundle. Contains the general methods for handling imports and exports
 * and for generating the dependencies and the required class path.
 *
 * @author <a href="mailto:marcus.klein@open-xchange.com">Marcus Klein</a>
 */
public abstract class AbstractModule {

    protected File file;
    protected String name;
    protected OSGIManifest osgiManifest;
    private String fragmentHostName;
    protected AbstractModule fragmentHost;
    protected final Set<AbstractModule> dependencies = new HashSet<AbstractModule>();
    private Set<String> deepClasspath;
    private Set<String> deepRuntimeClasspath;

    protected AbstractModule(File file) {
        super();
        this.file = file;
        name = file.getName();
    }

    public Set<String> getExportedPackages() {
        final Set<String> retval;
        if (null == osgiManifest) {
            retval = Collections.emptySet();
        } else {
            retval = osgiManifest.getExports();
        }
        return retval;
    }

    /**
     * Computes the dependencies for this module. This method should only be called, if every necessary module is parsed completely. So the
     * whole process is to parse all modules, build the structures for resolving dependencies and give those to this method and the this
     * method uses these structures to determine the module dependencies.
     * @param modulesByName a map having the modules name as key and the corresponding module as value.
     * @param modulesByPackage a map having the exported package as key and a set of exporting modules as value.
     * @param modulesByFilename a map having the directory or file of the module as key and the module as value.
     * @param strict <code>true</code> to throw a BuildException if dependencies can not be resolved from the given data structures. Can be
     * given as <code>false</code> only for source packaging; for compiling and binary build this must be <code>true</code>.
     */
    public void computeDependencies(Map<String, AbstractModule> modulesByName, Map<String, Set<AbstractModule>> modulesByPackage, Map<String, AbstractModule> modulesByFilename, boolean strict) throws BuildException {
        if (osgiManifest != null) {
            for (final BundleImport imported : osgiManifest.getImports()) {
                final Set<AbstractModule> exportingModules = modulesByPackage.get(imported.getPackageName());
                if (exportingModules != null) {
                    for (final AbstractModule module : exportingModules) {
                        if (module != this) {
                            dependencies.add(module);
                        }
                    }
                } else if (!JDK.exports(imported.getPackageName()) && !imported.isOptional() && strict) {
                    throw new BuildException("Can not find bundle that exports \"" + imported.getPackageName() + "\" to resolve import of bundle \"" + name + "\".");
                }
            }
            for (final RequiredBundle requiredBundle : osgiManifest.getRequiredBundles()) {
                final AbstractModule requiredModule = modulesByName.get(requiredBundle.getPackageName());
                if (null == requiredModule) {
                    if (strict) {
                        throw new BuildException("Can not find bundle to resolve require bundle \"" + requiredBundle.getPackageName() + "\".");
                    }
                } else {
                    if (!this.equals(requiredModule)) {
                        dependencies.add(requiredModule);
                    }
                }
            }
            fragmentHostName = osgiManifest.getEntry(OSGIManifest.FRAGMENT_HOST);
            if (fragmentHostName != null && !"system.bundle".equals(fragmentHostName)) {
                fragmentHost = modulesByName.get(fragmentHostName);
                if (null == fragmentHost) {
                    if (strict) {
                        throw new BuildException("Can not find bundle for fragment host \"" + fragmentHostName + "\".");
                    }
                } else {
                    dependencies.add(fragmentHost);
                }
            }
        }
    }

    public void computeDependenciesForFragments() {
        if (fragmentHost != null) {
            dependencies.addAll(fragmentHost.getDependencies());
            dependencies.remove(this); // just in case the fragment host "requires" the fragment
        }
    }

    public Set<AbstractModule> getDependencies() {
        return dependencies;
    }

    public Set<String> getRequiredClasspath() {
        final Set<String> retval = new HashSet<String>();
        for (final AbstractModule dependency : dependencies) {
            retval.addAll(dependency.getExportedClasspath());
        }
        return Collections.unmodifiableSet(retval);
    }

    public Set<String> getDeepRequiredClasspath() {
        if (null == deepClasspath) {
            final Stack<AbstractModule> seenModules = new Stack<AbstractModule>();
            deepClasspath = getDeepRequiredClasspath(seenModules, this);
        }
        return deepClasspath;
    }

    public Set<String> getDeepRuntimeClasspath() {
        if (null == deepRuntimeClasspath) {
            Stack<AbstractModule> seenModules = new Stack<AbstractModule>();
            deepRuntimeClasspath = getDeepRuntimeClasspath(seenModules, this);
        }
        return deepRuntimeClasspath;
    }

    public boolean isFragment() {
        if (osgiManifest != null) {
            fragmentHostName = osgiManifest.getEntry(OSGIManifest.FRAGMENT_HOST);
        }
        return null != fragmentHostName;
    }

    private Set<String> getDeepRequiredClasspathUnseen(Stack<AbstractModule> seenModules) {
        if (null == deepClasspath) {
            deepClasspath = getDeepRequiredClasspath(seenModules, this);
        }
        return deepClasspath;
    }

    private Set<String> getDeepRuntimeClasspathUnseen(Stack<AbstractModule> seenModules) {
        if (null == deepClasspath) {
            deepClasspath = getDeepRuntimeClasspath(seenModules, this);
        }
        return deepClasspath;
    }

    private static Set<String> getDeepRequiredClasspath(final Stack<AbstractModule> seenModules, final AbstractModule module) {
        seenModules.push(module);
        Set<String> retval = new HashSet<String>();
        for (final AbstractModule dependency : module.getDependencies()) {
            if (!seenModules.contains(dependency)) {
                retval.addAll(dependency.getDeepRequiredClasspathUnseen(seenModules));
            }
            retval.addAll(dependency.getExportedClasspath());
        }
        seenModules.pop();
        return retval;
    }

    private static Set<String> getDeepRuntimeClasspath(Stack<AbstractModule> seenModules, AbstractModule module) {
        seenModules.push(module);
        Set<String> retval = new HashSet<String>();
        for (AbstractModule dependency : module.getDependencies()) {
            if (!seenModules.contains(dependency)) {
                retval.addAll(dependency.getDeepRuntimeClasspathUnseen(seenModules));
            }
            retval.addAll(dependency.getRuntimeClasspath());
        }
        seenModules.pop();
        return retval;
    }

    protected final boolean isExported(final File classpathEntry) {
        if (!classpathEntry.exists() || classpathEntry.isDirectory()) {
            return false;
        }
        try {
            final JarFile jarFile = new JarFile(classpathEntry);
            try {
                for (final String exportedPackage : getExportedPackages()) {
                    String expected = exportedPackage.replace('.', '/') + '/';
                    Enumeration<JarEntry> entries = jarFile.entries();
                    while (entries.hasMoreElements()) {
                        JarEntry entry = entries.nextElement();
                        if (entry.getName().startsWith(expected)) {
                            return true;
                        }
                    }
                }
            } finally {
                jarFile.close();
            }
        } catch (IOException e) {
            throw new BuildException(e);
        }
        return false;
    }

    @Override
    public boolean equals(final Object o) {
        if (o == null) {
            return false;
        }
        if (o == this) {
            return true;
        }
        if (o.getClass() != this.getClass()) {
            return false;
        }
        final AbstractModule other = (AbstractModule) o;
        return this.name.equals(other.name);
    }

    @Override
    public int hashCode() {
        return name.hashCode();
    }

    @Override
    public String toString() {
        return name;
    }

    public String getName() {
        return name;
    }

    /**
     * JARs are only added to returned set if some Java package of that JAR is exported in the bundle manifest.
     * Beware, this must not be sufficient to create a classpath capable to execute some code.
     * @return
     */
    protected abstract Set<String> getExportedClasspath();

    /**
     * The returned set contains all referenced JARs of this bundle. Referenced means, the JAR is mentioned in the bundle manifest. It must
     * not be exported compared to {@link #getExportedClasspath()}. If the JAR is mentioned in the .classpath file it does not need to be
     * exported there, too.
     * @return
     */
    protected abstract Set<String> getRuntimeClasspath();

    public Set<String> getSourceDirs() {
        return Collections.emptySet();
    }

    public Set<String> getTestSourceDirs() {
        return Collections.emptySet();
    }

    public File getFile() {
        return file;
    }

    /**
     * Extends the local classpath of a bundle to a set of absolute file names. Beware, the returned Set can be modified, so you need to
     * protect it, if nobody should change the set.
     * @param bundleDir bundle file.
     * @param classpath set of local classpath entries of the bundle.
     * @return a set of absolute file names for the given classpath
     */
    protected static Set<String> addBundlePath(File bundleDir, Set<String> classpath) {
        final Set<String> retval = new HashSet<String>();
        for (final String classpathEntry : classpath) {
            retval.add(new File(bundleDir, classpathEntry).getAbsolutePath());
        }
        return retval;
    }
}
