/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.obs.api;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import javax.net.ssl.SSLHandshakeException;
import javax.xml.xpath.XPathExpressionException;
import org.apache.http.HttpEntity;
import org.apache.http.HttpEntityEnclosingRequest;
import org.apache.http.HttpException;
import org.apache.http.HttpHost;
import org.apache.http.HttpRequest;
import org.apache.http.HttpResponse;
import org.apache.http.HttpStatus;
import org.apache.http.NameValuePair;
import org.apache.http.NoHttpResponseException;
import org.apache.http.auth.AuthScope;
import org.apache.http.auth.UsernamePasswordCredentials;
import org.apache.http.client.AuthCache;
import org.apache.http.client.ClientProtocolException;
import org.apache.http.client.HttpRequestRetryHandler;
import org.apache.http.client.entity.UrlEncodedFormEntity;
import org.apache.http.client.methods.HttpDelete;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpPut;
import org.apache.http.client.methods.HttpRequestBase;
import org.apache.http.client.protocol.ClientContext;
import org.apache.http.entity.InputStreamEntity;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.auth.BasicScheme;
import org.apache.http.impl.client.BasicAuthCache;
import org.apache.http.impl.client.DefaultHttpClient;
import org.apache.http.message.BasicNameValuePair;
import org.apache.http.protocol.BasicHttpContext;
import org.apache.http.protocol.ExecutionContext;
import org.apache.http.protocol.HttpContext;
import org.apache.http.util.EntityUtils;

/**
 * @author choeger
 */
public class BuildServiceClient {

    private static final int MAX_RETRIES = 5;

    private final String login;

    private DefaultHttpClient httpclient;

    BasicHttpContext localcontext;
    HttpHost targetHost;

    /**
     * requires the account name, password and url of the open build service host
     */
    public BuildServiceClient(final String url, final String login, final String password) throws MalformedURLException {
        super();

        this.login = login;

        URL tmpurl = new URL(url);
        targetHost = new HttpHost(tmpurl.getHost(), tmpurl.getPort(), tmpurl.getProtocol());

        this.httpclient = new DefaultHttpClient();
        this.httpclient.getCredentialsProvider().setCredentials(
                new AuthScope(targetHost.getHostName(), targetHost.getPort()),
                new UsernamePasswordCredentials(login, password));

        HttpRequestRetryHandler myRetryHandler = new HttpRequestRetryHandler() {

            public boolean retryRequest(
                    IOException exception,
                    int executionCount,
                    HttpContext context) {
                if (executionCount >= 5) {
                    // Do not retry if over max retry count
                    return false;
                }
                if (exception instanceof NoHttpResponseException) {
                    // Retry if the server dropped connection on us
                    return true;
                }
                if (exception instanceof SSLHandshakeException) {
                    // Do not retry on SSL handshake exception
                    return false;
                }
                HttpRequest request = (HttpRequest) context.getAttribute(ExecutionContext.HTTP_REQUEST);
                boolean idempotent = !(request instanceof HttpEntityEnclosingRequest);
                if (idempotent) {
                    // Retry if the request is considered idempotent
                    return true;
                }
                return false;
            }

        };

        httpclient.setHttpRequestRetryHandler(myRetryHandler);

        // Create AuthCache instance
        AuthCache authCache = new BasicAuthCache();
        // Generate BASIC scheme object and add it to the local auth cache
        BasicScheme basicAuth = new BasicScheme();
        authCache.put(targetHost, basicAuth);

        // Add AuthCache to the execution context
        localcontext = new BasicHttpContext();
        localcontext.setAttribute(ClientContext.AUTH_CACHE, authCache);
    }

    /**
     * Executes a given method, checks if the http status of the response is OK and returns the response entity.
     * 
     * @param method the method to execute
     * @return the response entity
     * @throws ClientProtocolException
     * @throws IOException
     * @throws BuildServiceException
     */
    private HttpEntity getEntity(HttpRequestBase method) throws ClientProtocolException, IOException, BuildServiceException {
        HttpResponse response = httpclient.execute(targetHost, method, localcontext);
        checkIfHttpStatusOk(response);
        return response.getEntity();
    }

    /**
     * Executes given {@link GetMethod}, checks the response for HTTP status OK and returns an {@link InputStream} with the response.
     * @param method method to execute.
     * @return the input stream of the response.
     * @throws IOException
     */
    private InputStream getResponseStream(HttpRequestBase method) throws BuildServiceException, IOException {
        HttpEntity entity = getEntity(method);
        return entity.getContent();
    }

    /**
     * Return {@link InputStream} to binary package content
     *
     * @param projectName
     * @param repositoryName
     * @param arch the architecture. i586 or x86_64
     * @param packageName
     * @param filename
     * @return
     * @throws IOException
     */
    public InputStream getProjectBinaryPackageByName(String projectName, String repositoryName, Architecture arch, String packageName, String filename) throws BuildServiceException, IOException {
        HttpGet method = new HttpGet("/build/" + projectName + '/' + repositoryName + '/' + arch.toString() + '/' + packageName + '/' + filename);
        return getResponseStream(method);
    }

    /**
     * Return String of log message
     *
     * @param project
     * @param repository
     * @param arch the architecture. i586 or x86_64
     * @param pkgName
     * @return string of log message
     *
     * @throws IOException
     */
    public String getPackageLogByName(String project, String repository, Architecture arch, String pkgName) throws BuildServiceException, IOException {
        HttpGet method = new HttpGet("/build/" + project + '/' + repository + '/' + arch.toString() + '/' + pkgName + "/_log");
        return EntityUtils.toString(getEntity(method));
    }

    /**
     * create empty package on obs
     *
     * @param project
     * @param pkgName
     * @param disabledRepositories
     * @throws IOException
     */
    private void uploadPackageMeta(final String project, final String pkgName, String[] disabledRepositories) throws BuildServiceException, IOException {
        String pkgmeta = getPackageMeta(project, pkgName, disabledRepositories);
        final HttpPut pmethod = new HttpPut("/source/" + project + "/" + pkgName + "/_meta");
        pmethod.setEntity(new StringEntity(pkgmeta, "text/plain", "utf-8"));
        HttpResponse response = httpclient.execute(targetHost, pmethod, localcontext);
        consumeEntity(response);
        checkIfHttpStatusOk(response);
    }

    /**
     * Deletes package on OBS.
     * @param project Name of the project from that the package should be deleted.
     * @param pkgName Name of the package to be deleted.
     * @throws IOException If deleting the package fails.
     */
    public void deletePackage(String project, String pkgName) throws BuildServiceException {
        HttpDelete method = new HttpDelete("/source/" + project + "/" + pkgName);
        try {
            HttpResponse response = httpclient.execute(targetHost, method, localcontext);
            checkIfHttpStatusOk(response);
            consumeEntity(response);
        } catch (IOException e) {
            throw new BuildServiceException(e.getMessage(), e);
        }
    }

    /**
     * upload specified file into obs
     *
     * @param project
     * @param pkgName
     * @param file
     * @throws IOException
     */
    private void uploadSource(final String project, final String pkgName, final File file) throws BuildServiceException, IOException {
        final HttpPut pmethod = new HttpPut("/source/" + project + "/" + pkgName + "/" + file.getName() + "?rev=upload" );
        FileInputStream fis = new FileInputStream(file);
        HttpResponse response;
        try {
            pmethod.setEntity(new InputStreamEntity(fis, -1));
            response = httpclient.execute(targetHost, pmethod, localcontext);
        } finally {
            fis.close();
        }
        consumeEntity(response);
        checkIfHttpStatusOk(response);
    }


    /**
     * commit on obs
     *
     * @param project
     * @param pkgName
     * @param file
     * @throws IOException
     */
    private void commit(final String project, final String pkgName) throws BuildServiceException, IOException {
        final HttpPost pmethod = new HttpPost("/source/" + project + "/" + pkgName + "?cmd=commit");
        HttpResponse response = httpclient.execute(targetHost, pmethod, localcontext);
        consumeEntity(response);
        checkIfHttpStatusOk(response);
    }

    /**
     * delete specified source file
     *
     * @param projectName
     * @param pkgName
     * @param fileName
     * @throws IOException
     */
    private void deleteSource(final String projectName, final String pkgName, final String fileName) throws BuildServiceException, IOException {
        final HttpDelete dmethod = new HttpDelete("/source/" + projectName + "/" + pkgName + "/" + fileName);
        HttpResponse response = httpclient.execute(targetHost, dmethod, localcontext);
        consumeEntity(response);
        checkIfHttpStatusOk(response);
    }

    /**
     * generate package meta xml data
     *
     * @param project
     * @param pkgname
     * @param disabledRepositories
     * @return
     * @throws BuildServiceException if some of the disabled repositories can not be parsed.
     */
    private String getPackageMeta(String project, String pkgname, String[] disabledRepositories) throws BuildServiceException {
        StringBuilder sb = new StringBuilder();
        sb.append("<package project=\"" + project + "\" name=\"" + pkgname + "\">\n" +
                "<title>" + pkgname + "</title>\n" +
                "<description/>\n" +
                "<person role=\"maintainer\" userid=\"" + this.login + "\"/>\n");

        if (disabledRepositories != null){
            if (disabledRepositories.length > 0){
                sb.append("<build>\n");
                for (String s : disabledRepositories){
                    sb.append("<disable repository=\"" + DistroNaming.parse(s).getOBSName() + "\"/>\n");
                }
                sb.append("</build>\n");
            }
        }

        sb.append("</package>");
        return sb.toString();
    }

    /**
     * upload all source files from within directory path
     *
     * @param project
     * @param pkgname
     * @param path
     * @throws HttpException
     * @throws IOException
     * @throws BuildServiceException
     * @throws XPathExpressionException
     */
    public void uploadSourcePackage(String project, String pkgname, String path,String[] disabledRepositories) throws IOException, BuildServiceException {
        final File sdir = new File(path);
        if (!sdir.isDirectory()) {
            throw new BuildServiceException("path \"" + sdir.getAbsolutePath() + "\" is not a directory");
        }
        uploadSourcePackage(project, pkgname, sdir.listFiles(), disabledRepositories);
    }

    public void uploadSourcePackage(final String project, final String pkgName, final File[] files, String[] disabledRepositories) throws IOException, BuildServiceException {
        if (!existsProject(project)) {
            throw new BuildServiceException("Project " + project + " does not exist on build service.");
        }
        uploadPackageMeta(project, pkgName, disabledRepositories);

        SourceFile[] sourceFiles = getPackageSourceNames(project, pkgName);

        HashMap<String, SourceFile> currentSourceNamesHash = new HashMap<String, SourceFile>();
        for (int i = 0; i < sourceFiles.length;i++){
            currentSourceNamesHash.put(sourceFiles[i].getName(),sourceFiles[i]);
        }

        for (final File f : files) {
            if (currentSourceNamesHash.containsKey(f.getName())){
                System.out.println("Uploading " + f.getPath() + " to " + project);
                uploadSource(project, pkgName, f);
                currentSourceNamesHash.remove(f.getName());

                /*
                 * md 5 approach. Will work if directories keep their modification date after copying
                 *
                FileInputStream fis = new FileInputStream(f);
                String md5 = org.apache.commons.codec.digest.DigestUtils.md5Hex(fis);
                if (md5 != currentSourceNamesHash.get(f.getName()).getMd5()){
                    System.out.println("Removing " + f.getName() + " from " + project + " with md5: " + currentSourceNamesHash.get(f.getName()).getMd5());
                    deleteSource(project, pkgName, f.getName());
                    currentSourceNamesHash.remove(f.getName());
                    System.out.println("Uploading " + f.getPath() + " to " + project + " with md5: " + md5);
                    uploadSource(project, pkgName, f);
                } else {
                    System.out.println("Not Uploading " + f.getPath() + " to " + project + " because of same md5");
                }
                */

            } else {
                System.out.println("Uploading " + f.getPath() + " to " + project);
                uploadSource(project, pkgName, f);
            }


        }

        Collection<SourceFile> notNeededSourceFiles = currentSourceNamesHash.values();
        for (final SourceFile s : notNeededSourceFiles) {
            System.out.println("Removing " + s.getName() + " from " + project);
            deleteSource(project, pkgName, s.getName());
        }

        //Automatic Build will start after this commit
        commit(project,pkgName);

    }

    /**
     * get the list of binaries generated by a certain package.
     *
     * @param projectName
     * @param repositoryName
     * @param arch the architecture. i586 or x86_64
     * @param packageName
     * @return
     * @throws IOException
     * @throws BuildServiceException
     */
    public Binary[] getProjectBinaryPackageNames(String projectName, String repositoryName, Architecture arch, String packageName) throws IOException, BuildServiceException {
        HttpGet method = new HttpGet("/build/" + projectName + '/' + repositoryName + '/' + arch.toString() + '/' + packageName);
        final InputStream is = getResponseStream(method);
        try {
            return BinaryParser.parse(is);
        } finally {
            is.close();
        }
    }

    /**
     * @param project
     * @param pkgname
     * @return
     * @throws BuildServiceException
     */
    private SourceFile[] getPackageSourceNames(final String project, final String pkgname) throws BuildServiceException {
        final HttpGet method = new HttpGet("/source/" + project + "/" + pkgname);
        int retries = MAX_RETRIES;
        SourceFile[] sourceFiles = null;
        do {
            BuildServiceException exception = null;
            retries--;
            InputStream is = null;
            try {
                is = getResponseStream(method);
                sourceFiles = SourceFileParser.parse(is);
            } catch (final IOException e) {
                exception = new BuildServiceException(e.getMessage(), e);
            } catch (final BuildServiceException e) {
                exception = e;
            } finally {
                if (null != is) {
                    try {
                        is.close();
                    } catch (IOException e) {
                        if (null == exception) {
                            exception = new BuildServiceException(e.getMessage(), e);
                        }
                    }
                }
            }
            if (null != exception && retries == 0) {
                throw exception;
            }
        } while (null == sourceFiles);
        return sourceFiles;

    }

    public ObsPackage[] getPackageList(final String project) throws BuildServiceException {
        final HttpGet method = new HttpGet("/source/" + project);
        int retries = MAX_RETRIES;
        ObsPackage[] results = null;
        do {
            BuildServiceException exception = null;
            retries--;
            try {
                InputStream is = getResponseStream(method);
                try {
                    results = ObsPackageParser.parse(is);
                } finally {
                    is.close();
                }
            } catch (final IOException e) {
                exception = new BuildServiceException(e.getMessage(), e);
            } catch (final BuildServiceException e) {
                exception = e;
            }
            if (null != exception && retries == 0) {
                throw exception;
            }
        } while (null == results);
        return results;
    }

    /**
    * Executes {@code GET /build/<project>/_result} on OBS
    * @param project the project name
    * @param packageNames Collection of package Names, is allowed to be <code>null</code>
    * @return results array
    */
    public Result[] getProjectStatus(final String project, Collection<String> packageNames) throws BuildServiceException {
        StringBuilder sb = new StringBuilder();
        if (packageNames != null && packageNames.size() > 0) {
            sb.append("?");
            for (final String packageName : packageNames) {
                sb.append("package=");
                sb.append(packageName);
                sb.append('&');
            }
            sb.setLength(sb.length() - 1);
        }
        final HttpGet method = new HttpGet("/build/" + project + "/_result" + sb.toString());
        int retries = MAX_RETRIES;
        Result[] results = null;
        do {
            BuildServiceException exception = null;
            retries--;
            InputStream is = null;
            try {
                is = getResponseStream(method);
                results = ResultParser.parse(is);
            } catch (final IOException e) {
                exception = new BuildServiceException(e.getMessage(), e);
            } catch (final BuildServiceException e) {
                exception = e;
            } finally {
                if (null != is) {
                    try {
                        is.close();
                    } catch (IOException e) {
                        e.printStackTrace();
                    }
                }
            }
            if (null != exception && retries == 0) {
                throw exception;
            }
        } while (null == results);
        return results;
    }

    /**
     * Check if a project already exists on the buildservice (based on the project's name).
     * 
     * @param project The project to check
     * @return true if the {@link Project} already exists, else false
     * @throws BuildServiceException
     */
    public boolean existsProject(final Project project) throws BuildServiceException {
        final HttpGet method = new HttpGet("/source/" + project.getName() + "/_meta");
        final InputStream is;
        try {
            HttpResponse response;
            response = httpclient.execute(targetHost, method, localcontext);
            if (HttpStatus.SC_NOT_FOUND == response.getStatusLine().getStatusCode()){
                consumeEntity(response);
                return false;
            }
            checkIfHttpStatusOk(response);
            HttpEntity entity = response.getEntity();
            is = entity.getContent();
            try {
                final Project other = ProjectParser.parse(is);
                return other.equals(project);
            } finally {
                try {
                    is.close();
                } catch (IOException e) {
                    throw new BuildServiceException(e.getMessage(), e);
                }
            }
        } catch (final IOException e) {
            throw new BuildServiceException(e.getMessage(), e);
        }
    }

    /**
     * Check if a project already exists on the buildservice (based on the project's name).
     * @param project The project name to check
     * @return true if a project with the given name already exists, else false
     * @throws BuildServiceException
     */
    public boolean existsProject(String project) throws BuildServiceException {
        final HttpGet method = new HttpGet("/source/" + project + "/_meta");
        try {
            HttpResponse response;
            response = httpclient.execute(targetHost, method, localcontext);
            if (HttpStatus.SC_NOT_FOUND == response.getStatusLine().getStatusCode()){
                consumeEntity(response);
                return false;
            } else if (HttpStatus.SC_OK == response.getStatusLine().getStatusCode()){
                consumeEntity(response);
                return true;
            }
            checkIfHttpStatusOk(response);
            consumeEntity(response);
            return false;
        } catch (final IOException e) {
            throw new BuildServiceException(e.getMessage(), e);
        }
    }

    private void writeProjectMeta(final String project, final String data) throws BuildServiceException {
        final HttpPut method1 = new HttpPut("/source/" + project + "/_meta");
        try {
            method1.setEntity(new StringEntity(data, "text/xml", "UTF-8"));
            final HttpResponse response = httpclient.execute(targetHost, method1, localcontext);
            checkIfHttpStatusOk(response);
            consumeEntity(response);
        } catch (final IOException e) {
            throw new BuildServiceException(e.getMessage(), e);
        }
    }

    public void createProject(final Project project) throws BuildServiceException {
        final String xml = Tools.transform(ProjectWriter.write(project));
        writeProjectMeta(project.getName(), xml);
        final HttpPut method2 = new HttpPut("/source/" + project.getName() + "/_config");
        try {
            method2.setEntity(new StringEntity(project.getBuildConfiguration()));
            final HttpResponse response = httpclient.execute(targetHost, method2, localcontext);
            checkIfHttpStatusOk(response);
            consumeEntity(response);
        } catch (final IOException e) {
            throw new BuildServiceException(e.getMessage(), e);
        }
    }

    public void copyProject(String project, String OldProject) throws BuildServiceException {
        final HttpPost method = new HttpPost("/source/" + project);
        List<NameValuePair> nameValuePairs = new ArrayList<NameValuePair>(1);
        nameValuePairs.add(new BasicNameValuePair("cmd", "copy"));
        nameValuePairs.add(new BasicNameValuePair("oproject", OldProject));
        nameValuePairs.add(new BasicNameValuePair("withhistory", "1"));
        nameValuePairs.add(new BasicNameValuePair("withbinaries", "1"));
        nameValuePairs.add(new BasicNameValuePair("resign", "1"));
        nameValuePairs.add(new BasicNameValuePair("nodelay", "1"));
        try {
            method.setEntity(new UrlEncodedFormEntity(nameValuePairs));
            HttpResponse response = httpclient.execute(targetHost, method, localcontext);
            checkIfHttpStatusOk(response);
            consumeEntity(response);
        } catch (IOException e) {
            throw new BuildServiceException(e.getMessage(), e);
        }
    }

    public void deleteProject(String projectName) throws BuildServiceException {
        HttpDelete request = new HttpDelete("/source/" + projectName);
        try {
            HttpResponse response = httpclient.execute(targetHost, request, localcontext);
            checkIfHttpStatusOk(response);
            consumeEntity(response);
        } catch (IOException e) {
            throw new BuildServiceException(e.getMessage(), e);
        }
    }

    public void disableProject(Project project) throws BuildServiceException {
        disableProject(project.getName());
    }

    public void disableProject(String project) throws BuildServiceException {
        setFlagForProject(project, Flag.BUILD, Status.DISABLE);
        setFlagForProject(project, Flag.PUBLISH, Status.DISABLE);
    }

    public void setFlagForProject(String project, Flag flag, Status status) throws BuildServiceException {
        final HttpPost method = new HttpPost("/source/" + project);
        List<NameValuePair> nameValuePairs = new ArrayList<NameValuePair>(1);
        nameValuePairs.add(new BasicNameValuePair("cmd", "set_flag"));
        nameValuePairs.add(new BasicNameValuePair("flag", flag.name().toLowerCase()));
        nameValuePairs.add(new BasicNameValuePair("status", status.name().toLowerCase()));
        nameValuePairs.add(new BasicNameValuePair("nodelay", "1"));
        try {
            method.setEntity(new UrlEncodedFormEntity(nameValuePairs));
            HttpResponse response = httpclient.execute(targetHost, method, localcontext);
            checkIfHttpStatusOk(response);
            consumeEntity(response);
        } catch (IOException e) {
            throw new BuildServiceException(e.getMessage(), e);
        }
    }

    public boolean isProjectDisabled(final Project project) throws BuildServiceException, IOException {
        InputStream is = getProjectInputStream(project.getName());
        try {
            return ProjectParser.isDisabled(is);
        } finally {
            is.close();
        }
    }

    public String getProjectConfig(final String project) throws BuildServiceException {
        final HttpGet method = new HttpGet("/source/" + project + "/_config");
        try {
            return EntityUtils.toString(getEntity(method));
        } catch (IOException e) {
            throw new BuildServiceException(e.getMessage(), e);
        }
    }

    public ProjectReference[] getProjectList() throws BuildServiceException {
        final HttpGet method = new HttpGet("/source");
        int retries = MAX_RETRIES;
        ProjectReference[] results = null;
        do {
            BuildServiceException exception = null;
            retries--;
            try {
                InputStream is = getResponseStream(method);
                try {
                    results = ProjectParser.parseAll(is);
                } finally {
                    is.close();
                }
            } catch (final IOException e) {
                exception = new BuildServiceException(e.getMessage(), e);
            } catch (final BuildServiceException e) {
                exception = e;
            }
            if (null != exception && retries == 0) {
                throw exception;
            }
        } while (null == results);
        return results;
    }

    private InputStream getProjectInputStream(final String project) throws BuildServiceException {
        final HttpGet method = new HttpGet("/source/" + project + "/_meta");
        BuildServiceException exception = new BuildServiceException("unable to get project metadata for " + project);
        for (int i = 0; i < MAX_RETRIES; i++) {
            try {
                return getResponseStream(method);
            } catch (final IOException e) {
                exception = new BuildServiceException(e.getMessage(), e);
            } catch (final BuildServiceException e) {
                exception = e;
            }
        }
        throw exception;
    }

    public Project getRepositoryList(String projectName) throws BuildServiceException {
        int retries = MAX_RETRIES;
        Project project = null;
        do {
            BuildServiceException exception = null;
            retries--;
            final InputStream is = getProjectInputStream(projectName);
            try {
                project = ProjectParser.parse(is);
            } catch (BuildServiceException e) {
                exception = e;
            } finally {
                try {
                    is.close();
                } catch (IOException e) {
                    throw new BuildServiceException(e.getMessage(), e);
                }
            }
            if (null != exception && retries == 0) {
                throw exception;
            }
        } while (null == project);
        return project;
    }

    /**
     * Ensure that the http entity is consumed and associated resources are closed.
     * 
     * @param response response containing the http entity
     * @throws IOException
     */
    private static void consumeEntity(HttpResponse response) throws IOException {
        EntityUtils.consume(response.getEntity());
    }

    /**
     * Check if a http response is OK and throw BuildServiceException with detailed error message otherwise.
     * 
     * @param response the response to check
     * @throws BuildServiceException if the response isn't ok
     * @throws IOException if the response can't be analyzed
     */
    private static void checkIfHttpStatusOk(HttpResponse response) throws BuildServiceException, IOException {
        if (response.getStatusLine().getStatusCode() != HttpStatus.SC_OK) {
            String responseString = EntityUtils.toString(response.getEntity());
            consumeEntity(response);
            throw new BuildServiceException(response.getStatusLine().toString() + " : " + responseString);
        }
    }
}
