/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2012 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.usm.api.cache;

import com.openexchange.usm.api.session.assets.CompleteSessionID;

/**
 * {@link SyncStateCacheProvider}. A provider for SyncStateCaches that USM should use (to reduce number of database read operations for
 * stored SyncState data). If no SyncStateCacheProvider is available, USM will use an internal implementation that uses the Java heap to
 * store the SyncStates.
 * 
 * @author <a href="mailto:afe@microdoc.de">Alexander Feess</a>
 */
public interface SyncStateCacheProvider {

    /**
     * Returns a SyncStateCache that can be used for the given USM session referenced by the CompleteSessionID. The SyncStateCacheProvider
     * may return the same cache for all sessions or individual caches for each or some of the sessions. All requests on individual caches
     * will also use the CompleteSessionID.
     * 
     * @param session
     * @return SyncStateCache that should be used for the given USM session.
     */
    SyncStateCache getCache(CompleteSessionID session);

    /**
     * This method is called if the given USM session has been removed from memory (i.e. it is not active for some time, but might be
     * reactivated in the future). The SyncStateCacheProvider may execute any actions to reduce local resource usage, but is not required to
     * do so if enough resources are available.
     * 
     * @param session
     */
    void freeCache(CompleteSessionID session);

    /**
     * This method is called if the given USM session has been completely removed from the system (i.e. no future calls to this session are
     * to be expected, the client has terminated any synchronization). The SyncStateCacheProvider should remove any cached data for that
     * session. Note that this method may never get called, so the SyncStateCacheProvider should on its own remove any data that is very old
     * and has not been accessed in a very long time. (any cached data can be retrieved from the database as long as the session is stored
     * there)
     * 
     * @param session
     */
    void removeCacheCompletely(CompleteSessionID session);

    /**
     * @return the current number of SyncStates stored in all caches of all sessions. Used by USM to provide cache information in the JMX
     *         interface.
     */
    int getSyncStateCount();

    /**
     * Called to remove any cached data that is older than the configured limit for cached sync states. Any sync state which has not been
     * accessed since the given limit should be removed from the cache (special cache implementations may choose to ignore this if they have
     * enough resources available).
     * 
     * @param limit timestamp (in milliseconds since the epoch). All sync states that have not been accessed since that time should be
     *            removed
     * @return total number of sync states that have been removed from all sync state caches
     */
    int removeUnusedSyncStates(long limit);
}
