package com.openexchange.usm.clt;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import javax.management.AttributeNotFoundException;
import javax.management.InstanceNotFoundException;
import javax.management.MBeanException;
import javax.management.MBeanServerConnection;
import javax.management.ReflectionException;
import javax.management.remote.JMXConnector;
import javax.management.remote.JMXConnectorFactory;
import javax.management.remote.JMXServiceURL;
import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.CommandLineParser;
import org.apache.commons.cli.HelpFormatter;
import org.apache.commons.cli.Option;
import org.apache.commons.cli.OptionGroup;
import org.apache.commons.cli.Options;
import org.apache.commons.cli.ParseException;
import org.apache.commons.cli.PosixParser;

/**
 * {@link USMMonitoringTool} Tool which displays advanced administrative information about the USM system.
 * 
 * @author <a href="mailto:ldo@microdoc.de">Ljupka Grabsch</a>
 */
public class USMMonitoringTool {
    
    
    private static final Options toolOptions;
    
    private String _jmxLogin;

    private String _jmxPassword;

    private int _jmxPort = 9999;
    
    private MBeanServerConnection _connection;
    
    private Option[] _optionsToExecute;
    
    private static List<Option> _gettersList = new ArrayList<Option>();
    
    private boolean _quietMode = false;
    
    static { 
        _gettersList.add(new Option("asc", "ActiveSessionCount", false, "Number of currently active (i.e. stored in local memory) USM sessions"));
        _gettersList.add(new Option("wepd", "WaitForChangesEmailPullDelay", false, "Delay in ms. (>= 0) or percent (< 0) after which an internal pull is performed if a client waits on email folder changes"));
        _gettersList.add(new Option("wempd", "WaitForChangesEmailMinPullDelay", false, "Minimum time in ms. before an internal pull is performed if a client waits on email folder changes. A value < 0 disables internal email pulling completely"));
        _gettersList.add(new Option("msdb", "MaxSyncStatesInDB", false, "Maximum number of SyncStates kept in database for any folder of a session"));
        _gettersList.add(new Option("sct", "USMSesssionCacheInactivityTimeout", false, "Currently set timeout (in seconds), after which an USM session will be explicitly removed from the memory cache if it hasn't been accessed"));
        _gettersList.add(new Option("ssct", "SyncStateCacheInactivityTimeout", false, "Currently set timeout (in seconds), after which a SyncState will be removed from the memory cache if it hasn't been accessed"));
        _gettersList.add(new Option("tsm", "TotalUSMSessionsInMemory", false, "Number of USM sessions currently stored in memory"));
        _gettersList.add(new Option("tssm", "TotalSyncStatesInMemory", false, "Number of USM SyncStates (for all sessions) currently stored in memory"));
        _gettersList.add(new Option("nsm", "NewUSMSessionsInMemory", false, "Number of USM sessions that have been created or loaded from DB (since last counter reset)"));
        _gettersList.add(new Option("sri", "USMSessionsRemovedDueToInactivity", false, "Number of USM sessions that have been removed from the memory cache due to long inactivity (since the last counter reset)"));
        _gettersList.add(new Option("srjvm", "USMSessionsRemovedByJVM", false, "Number of USM sessions that have been removed from the memory cache by the Java VM because the memory was needed for other data (since the last counter reset)"));
        _gettersList.add(new Option("sssdb", "SyncStatesSavedToDatabase", false, "Number of new SyncStates that have been saved to the database (since the last counter reset)"));
        _gettersList.add(new Option("ssldb", "SyncStatesLoadedFromDatabase", false, "Number of SyncStates that have been restored from the database (since the last counter reset)"));
        _gettersList.add(new Option("ssri", "SyncStatesRemovedDueToInactivity", false, "Number of SyncStates that have been removed from the memory cache due to inactivity (since the last counter reset)"));
        _gettersList.add(new Option("mc", "MaxTotalConnections", false, "Configured max number of overall HTTP connections to access the OX"));
        _gettersList.add(new Option("cu", "ConnectionsInUse", false, "Currently used HTTP connections to access the OX"));
               
        final Options opts = new Options();
        opts.addOption("h", "help", false, "Prints a help text");
        opts.addOption("l", "login", true, "The optional JMX login (if JMX has authentication enabled)");
        opts.addOption("p", "port", true, "The optional JMX port (default:9999)");
        opts.addOption("s", "password", true, "The optional JMX password (if JMX has authentication enabled)");
        opts.addOption("a", "all", false, "Prints all available system information");
        
        opts.addOption("q", "quiet", false, "Quiet Mode: only prints the requested value without readable explanation");
        
        OptionGroup getters = new OptionGroup();
        for (Option option : _gettersList) {
            getters.addOption(option);
        }
        getters.setRequired(false);
        opts.addOptionGroup(getters);
        toolOptions = opts;
    }

    /**
     * @param args
     */
    public static void main(String[] args) {
       new USMMonitoringTool(args).execute();
    }

    public USMMonitoringTool(String[] args) {
        CommandLineParser parser = new PosixParser();
        CommandLine cmd;
        try {
            cmd = parser.parse(toolOptions, args);
            if (cmd.hasOption('h')) {
                printHelpAndExit();
            }
            if (cmd.hasOption('l'))
                _jmxLogin = CLToolsUtility.getOptionStringValue(cmd, 'l');
            if (cmd.hasOption('p'))
                _jmxPort = CLToolsUtility.getOptionIntValue(cmd, 'p', 1, 65535);
            if (cmd.hasOption('s'))
                _jmxPassword = cmd.getOptionValue('s');
            if (cmd.hasOption('q'))
                _quietMode = true;
            if (cmd.hasOption('a')) {
                _optionsToExecute = _gettersList.toArray(new Option[_gettersList.size()]);
            } else {
                _optionsToExecute = cmd.getOptions();
                if(_optionsToExecute == null || _optionsToExecute.length == 0) {
                    System.err.println("No arguments specified");
                    printHelpAndExit();
                }
            }
        } catch (ParseException e) {
            System.err.println("Unable to parse command line: " + e.getMessage());
            printHelpAndExit();
        }
    }
    
    
    private static void printHelpAndExit() {
        HelpFormatter helpFormatter = new HelpFormatter();
        helpFormatter.printHelp(
            "usmmonitoring",
            "",
            toolOptions, 
            "");
        System.exit(0);
    }
    
    
    private void execute() {
        try {
            final JMXServiceURL url = new JMXServiceURL(
                new StringBuilder("service:jmx:rmi:///jndi/rmi://localhost:").append(_jmxPort).append("/server").toString());
            final JMXConnector jmxConnector = JMXConnectorFactory.connect(url, CLToolsUtility.createJMXEnvironment(_jmxLogin, _jmxPassword));
            try {
                _connection = jmxConnector.getMBeanServerConnection();
                for (Option option : _optionsToExecute) {
                    if (_gettersList.contains(option)) {
                        executeGetterMethod(option.getLongOpt(), option.getDescription());
                    }
                }
            } finally {
                jmxConnector.close();
            }
        } catch (Exception e) {
            System.err.println("An error occurred while executing your request:");
            e.printStackTrace();
            printHelpAndExit();
        }
    }
    
    private void executeGetterMethod(String methodName, String optionDescription) throws  AttributeNotFoundException, InstanceNotFoundException, MBeanException, ReflectionException, IOException {
        Object result = _connection.getAttribute(JMXObjectNames.USM_SESSION_INFO_MBEAN, methodName);
        if (result == null) {
            System.err.println("Warning: Empty result received");
        } else if(_quietMode) {
            System.out.println(result);
        } else {
            System.out.println(optionDescription + ": " + result);
        }
    }
}
