/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2014 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.smtp;

import static com.openexchange.mail.MailExceptionCode.getSize;
import static com.openexchange.mail.MailServletInterface.mailInterfaceMonitor;
import static com.openexchange.mail.mime.utils.MimeMessageUtility.parseAddressList;
import static com.openexchange.mail.text.TextProcessing.performLineFolding;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.net.InetAddress;
import java.net.InetSocketAddress;
import java.net.UnknownHostException;
import java.nio.charset.UnsupportedCharsetException;
import java.security.PrivilegedActionException;
import java.security.PrivilegedExceptionAction;
import java.text.DateFormat;
import java.util.Arrays;
import java.util.Date;
import java.util.Locale;
import java.util.Map;
import java.util.Properties;
import java.util.Queue;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.regex.Matcher;
import javax.activation.DataHandler;
import javax.mail.Address;
import javax.mail.Message.RecipientType;
import javax.mail.MessagingException;
import javax.mail.Multipart;
import javax.mail.NoSuchProviderException;
import javax.mail.Part;
import javax.mail.Provider;
import javax.mail.Transport;
import javax.mail.event.TransportEvent;
import javax.mail.event.TransportListener;
import javax.mail.internet.InternetAddress;
import javax.mail.internet.MailDateFormat;
import javax.mail.internet.MimeBodyPart;
import javax.mail.internet.MimeMessage;
import javax.mail.internet.MimeMultipart;
import javax.mail.internet.idn.IDNA;
import javax.security.auth.Subject;
import com.openexchange.config.ConfigurationService;
import com.openexchange.config.Filter;
import com.openexchange.config.cascade.ConfigProperty;
import com.openexchange.config.cascade.ConfigView;
import com.openexchange.config.cascade.ConfigViewFactory;
import com.openexchange.exception.OXException;
import com.openexchange.groupware.contexts.Context;
import com.openexchange.groupware.contexts.impl.ContextStorage;
import com.openexchange.groupware.i18n.MailStrings;
import com.openexchange.groupware.ldap.UserStorage;
import com.openexchange.groupware.notify.hostname.HostnameService;
import com.openexchange.i18n.tools.StringHelper;
import com.openexchange.java.Charsets;
import com.openexchange.java.Java7ConcurrentLinkedQueue;
import com.openexchange.java.Strings;
import com.openexchange.java.util.MsisdnCheck;
import com.openexchange.log.LogProperties;
import com.openexchange.mail.MailExceptionCode;
import com.openexchange.mail.MailPath;
import com.openexchange.mail.api.MailAccess;
import com.openexchange.mail.config.MailProperties;
import com.openexchange.mail.dataobjects.MailMessage;
import com.openexchange.mail.dataobjects.compose.ComposeType;
import com.openexchange.mail.dataobjects.compose.ComposedMailMessage;
import com.openexchange.mail.dataobjects.compose.ContentAware;
import com.openexchange.mail.mime.ContentType;
import com.openexchange.mail.mime.MessageHeaders;
import com.openexchange.mail.mime.MimeHeaderNameChecker;
import com.openexchange.mail.mime.MimeMailException;
import com.openexchange.mail.mime.MimeMailExceptionCode;
import com.openexchange.mail.mime.QuotedInternetAddress;
import com.openexchange.mail.mime.converters.MimeMessageConverter;
import com.openexchange.mail.mime.datasource.MimeMessageDataSource;
import com.openexchange.mail.mime.filler.MimeMessageFiller;
import com.openexchange.mail.mime.utils.MimeMessageUtility;
import com.openexchange.mail.transport.MailTransport;
import com.openexchange.mail.transport.MimeSupport;
import com.openexchange.mail.transport.MtaStatusInfo;
import com.openexchange.mail.transport.config.ITransportProperties;
import com.openexchange.mail.transport.config.TransportConfig;
import com.openexchange.mail.transport.config.TransportProperties;
import com.openexchange.mail.usersetting.UserSettingMail;
import com.openexchange.mail.usersetting.UserSettingMailStorage;
import com.openexchange.mail.utils.MessageUtility;
import com.openexchange.mailaccount.MailAccount;
import com.openexchange.mailaccount.MailAccountStorageService;
import com.openexchange.session.Session;
import com.openexchange.smtp.config.ISMTPProperties;
import com.openexchange.smtp.config.MailAccountSMTPProperties;
import com.openexchange.smtp.config.SMTPConfig;
import com.openexchange.smtp.config.SMTPSessionProperties;
import com.openexchange.smtp.filler.SMTPMessageFiller;
import com.openexchange.smtp.services.Services;
import com.openexchange.tools.ssl.TrustAllSSLSocketFactory;
import com.sun.mail.smtp.JavaSMTPTransport;
import com.sun.mail.smtp.SMTPMessage;
import com.sun.mail.smtp.SMTPSendFailedException;

/**
 * {@link SMTPTransport} - The SMTP mail transport.
 *
 * @author <a href="mailto:thorben.betten@open-xchange.com">Thorben Betten</a>
 */
public final class SMTPTransport extends MailTransport implements MimeSupport {

    private static final org.slf4j.Logger LOG = org.slf4j.LoggerFactory.getLogger(SMTPTransport.class);

    /**
     * The SMTP protocol name.
     */
    private static final String SMTP = SMTPProvider.PROTOCOL_SMTP.getName();

    private static final class SaslSmtpLoginAction implements PrivilegedExceptionAction<Object> {

        private final Transport transport;
        private final String server;
        private final int port;
        private final String login;
        private final String pw;

        protected SaslSmtpLoginAction(final Transport transport, final String server, final int port, final String login, final String pw) {
            super();
            this.transport = transport;
            this.server = server;
            this.port = port;
            this.login = login;
            this.pw = pw;
        }

        @Override
        public Object run() throws MessagingException {
            transport.connect(server, port, login, pw);
            return null;
        }

    }

    // private static final String CHARENC_ISO_8859_1 = "ISO-8859-1";

    private static final String KERBEROS_SESSION_SUBJECT = "kerberosSubject";

    private static volatile String staticHostName;

    private static volatile UnknownHostException warnSpam;

    static {
        try {
            staticHostName = InetAddress.getLocalHost().getCanonicalHostName();
        } catch (final UnknownHostException e) {
            staticHostName = "localhost";
            warnSpam = e;
        }
    }

    private final Queue<Runnable> pendingInvocations;

    private volatile javax.mail.Session smtpSession;

    private final int accountId;

    private final Session session;

    private transient Subject kerberosSubject;

    private final Context ctx;

    private final UserSettingMail usm;

    private volatile SMTPConfig cachedSmtpConfig;

    protected SMTPTransport() {
        super();
        accountId = MailAccount.DEFAULT_ID;
        smtpSession = null;
        session = null;
        ctx = null;
        usm = null;
        pendingInvocations = new Java7ConcurrentLinkedQueue<Runnable>();
    }

    /**
     * Constructor
     *
     * @param session The session
     * @throws OXException If initialization fails
     */
    public SMTPTransport(final Session session) throws OXException {
        this(session, MailAccount.DEFAULT_ID);
    }

    /**
     * Constructor
     *
     * @param session The session
     * @param accountId The account ID
     * @throws OXException If initialization fails
     */
    public SMTPTransport(final Session session, final int accountId) throws OXException {
        super();
        pendingInvocations = new ConcurrentLinkedQueue<Runnable>();
        this.session = session;
        this.accountId = accountId;
        if (session == null) {
            /*
             * Dummy instance
             */
            ctx = null;
            usm = null;
        } else {
            try {
                ctx = ContextStorage.getStorageContext(session.getContextId());
            } catch (final OXException e) {
                throw e;
            }
            usm = UserSettingMailStorage.getInstance().getUserSettingMail(session.getUserId(), ctx);
        }
    }

    private void clearUp() {
        doInvocations();
    }

    @Override
    public void close() {
        clearUp();
    }

    /**
     * Executes all tasks queued for execution
     */
    private void doInvocations() {
        for (Runnable task = pendingInvocations.poll(); task != null; task = pendingInvocations.poll()) {
            task.run();
        }
    }

    /**
     * Executes the given task. This method returns as soon as the task is scheduled, without waiting for it to be executed.
     *
     * @param task The task to be executed.
     */
    private void invokeLater(final Runnable task) {
        pendingInvocations.offer(task);
    }

    /**
     * Checks if Kerberos authentication is supposed to be performed.
     *
     * @return <code>true</code> for Kerberos authentication; otherwise <code>false</code>
     */
    private boolean isKerberosAuth() {
        return MailAccount.DEFAULT_ID == accountId && null != kerberosSubject;
    }

    private static void handlePrivilegedActionException(final PrivilegedActionException e) throws MessagingException, OXException {
        if (null == e) {
            return;
        }
        final Exception cause = e.getException();
        if (null == cause) {
            throw MailExceptionCode.UNEXPECTED_ERROR.create(e.getCause(), e.getMessage());
        }
        if (cause instanceof MessagingException) {
            throw (MessagingException) cause;
        }
        if (cause instanceof OXException) {
            throw (OXException) cause;
        }
        throw MailExceptionCode.UNEXPECTED_ERROR.create(cause, cause.getMessage());
    }

    private long getMaxMailSize() throws OXException {
        final ConfigViewFactory factory = Services.getService(ConfigViewFactory.class);

        if (factory != null) {
            final ConfigView view = factory.getView(session.getUserId(), session.getContextId());
            final ConfigProperty<Long> property = view.property("com.openexchange.mail.maxMailSize", Long.class);

            if (property.isDefined()) {
                final Long l = property.get();
                final long maxMailSize = null == l ? -1 : l.longValue();
                if (maxMailSize > 0) {
                    return maxMailSize;
                }
            }
        }

        return -1;
    }

    private javax.mail.Session getSMTPSession() throws OXException {
        SMTPConfig smtpConfig = getTransportConfig0();
        return getSMTPSession(smtpConfig, accountId > 0 && (smtpConfig.isRequireTls() || MailProperties.getInstance().isEnforceSecureConnection()));
    }

    private javax.mail.Session getSMTPSession(SMTPConfig smtpConfig, boolean forceSecure) throws OXException {
        if (null == smtpSession) {
            synchronized (this) {
                if (null == smtpSession) {
                    final Properties smtpProps = SMTPSessionProperties.getDefaultSessionProperties();
                    smtpProps.put("mail.smtp.class", JavaSMTPTransport.class.getName());
                    smtpProps.put("com.openexchange.mail.maxMailSize", Long.toString(getMaxMailSize()));


                    /*
                     * Set properties
                     */
                    final ISMTPProperties smtpProperties = smtpConfig.getSMTPProperties();
                    /*
                     * Check for Kerberos subject
                     */
                    this.kerberosSubject = (Subject) session.getParameter(KERBEROS_SESSION_SUBJECT);
                    final boolean kerberosAuth = isKerberosAuth();
                    if (kerberosAuth) {
                        smtpProps.put("mail.smtp.auth", "true");
                        smtpProps.put("mail.smtp.sasl.enable", "true");
                        smtpProps.put("mail.smtp.sasl.authorizationid", smtpConfig.getLogin());
                        smtpProps.put("mail.smtp.sasl.mechanisms", (kerberosAuth ? "GSSAPI" : "PLAIN"));
                    } else {
                        smtpProps.put("mail.smtp.auth", smtpProperties.isSmtpAuth() ? "true" : "false");
                    }
                    /*
                     * Localhost, & timeouts
                     */
                    final String smtpLocalhost = smtpProperties.getSmtpLocalhost();
                    if (smtpLocalhost != null) {
                        smtpProps.put("mail.smtp.localhost", smtpLocalhost);
                    }
                    if (smtpProperties.getSmtpTimeout() > 0) {
                        smtpProps.put("mail.smtp.timeout", Integer.toString(smtpProperties.getSmtpTimeout()));
                    }
                    if (smtpProperties.getSmtpConnectionTimeout() > 0) {
                        smtpProps.put("mail.smtp.connectiontimeout", Integer.toString(smtpProperties.getSmtpConnectionTimeout()));
                    }
                    /*
                     * Check if a secure SMTP connection should be established
                     */
                    final String sPort = String.valueOf(smtpConfig.getPort());
                    final String socketFactoryClass = TrustAllSSLSocketFactory.class.getName();
                    if (smtpConfig.isSecure()) {
                        /*
                         * Enables the use of the STARTTLS command (if supported by the server) to switch the connection to a TLS-protected
                         * connection before issuing any login commands.
                         */
                        // smtpProps.put("mail.smtp.starttls.enable", "true");
                        /*
                         * Force use of SSL through specifying the name of the javax.net.SocketFactory interface. This class will be used to
                         * create SMTP sockets.
                         */
                        smtpProps.put("mail.smtp.socketFactory.class", socketFactoryClass);
                        smtpProps.put("mail.smtp.socketFactory.port", sPort);
                        smtpProps.put("mail.smtp.socketFactory.fallback", "false");
                        /*
                         * Specify SSL protocols
                         */
                        smtpProps.put("mail.smtp.ssl.protocols", smtpConfig.getSMTPProperties().getSSLProtocols());
                        /*
                         * Specify SSL cipher suites
                         */
                        final String cipherSuites = smtpConfig.getSMTPProperties().getSSLCipherSuites();
                        if (false == Strings.isEmpty(cipherSuites)) {
                            smtpProps.put("mail.smtp.ssl.ciphersuites", cipherSuites);
                        }
                        // smtpProps.put("mail.smtp.ssl", "true");
                        /*
                         * Needed for JavaMail >= 1.4
                         */
                        // Security.setProperty("ssl.SocketFactory.provider", socketFactoryClass);
                    } else {
                        /*
                         * Enables the use of the STARTTLS command (if supported by the server) to switch the connection to a TLS-protected
                         * connection before issuing any login commands.
                         */
                        String hostName = smtpLocalhost;
                        if (null == hostName) {
                            final HostnameService hostnameService = Services.getService(HostnameService.class);
                            if (null == hostnameService) {
                                hostName = getHostName();
                            } else {
                                hostName = hostnameService.getHostname(session.getUserId(), session.getContextId());
                            }
                            if (null == hostName) {
                                hostName = getHostName();
                            }
                        }
                        try {
                            final InetSocketAddress address = new InetSocketAddress(IDNA.toASCII(smtpConfig.getServer()), smtpConfig.getPort());
                            final Map<String, String> capabilities = SMTPCapabilityCache.getCapabilities(address, smtpConfig.isSecure(), smtpProperties, hostName);
                            if (capabilities.containsKey("STARTTLS")) {
                                smtpProps.put("mail.smtp.starttls.enable", "true");
                            } else if (forceSecure) {
                                // No SSL demanded and SMTP server seems not to support TLS
                                throw MailExceptionCode.NON_SECURE_DENIED.create(smtpConfig.getServer());
                            }
                        } catch (final IOException e) {
                            smtpProps.put("mail.smtp.starttls.enable", "true");
                        }
                        /*
                         * Specify the javax.net.ssl.SSLSocketFactory class, this class will be used to create SMTP SSL sockets if TLS
                         * handshake says so.
                         */
                        smtpProps.put("mail.smtp.socketFactory.port", sPort);
                        smtpProps.put("mail.smtp.ssl.socketFactory.class", socketFactoryClass);
                        smtpProps.put("mail.smtp.ssl.socketFactory.port", sPort);
                        smtpProps.put("mail.smtp.socketFactory.fallback", "false");
                        /*
                         * Specify SSL protocols
                         */
                        smtpProps.put("mail.smtp.ssl.protocols", smtpConfig.getSMTPProperties().getSSLProtocols());
                        /*
                         * Specify SSL cipher suites
                         */
                        final String cipherSuites = smtpConfig.getSMTPProperties().getSSLCipherSuites();
                        if (false == Strings.isEmpty(cipherSuites)) {
                            smtpProps.put("mail.smtp.ssl.ciphersuites", cipherSuites);
                        }
                        // smtpProps.put("mail.smtp.ssl", "true");
                        /*
                         * Needed for JavaMail >= 1.4
                         */
                        // Security.setProperty("ssl.SocketFactory.provider", socketFactoryClass);
                    }
                    /*
                     * Apply host & port to SMTP session
                     */
                    // smtpProps.put(MIMESessionPropertyNames.PROP_SMTPHOST, smtpConfig.getServer());
                    // smtpProps.put(MIMESessionPropertyNames.PROP_SMTPPORT, sPort);
                    smtpSession = javax.mail.Session.getInstance(smtpProps, null);
                    smtpSession.addProvider(new Provider(
                        Provider.Type.TRANSPORT,
                        "smtp",
                        JavaSMTPTransport.class.getName(),
                        "Open-Xchange, Inc.",
                        "7.2.2"));
                }
            }
        }
        return smtpSession;
    }

    @Override
    public SMTPConfig getTransportConfig() throws OXException {
        return getTransportConfig0();
    }

    private SMTPConfig getTransportConfig0() throws OXException {
        SMTPConfig tmp = cachedSmtpConfig;
        if (tmp == null) {
            synchronized (this) {
                tmp = cachedSmtpConfig;
                if (tmp == null) {
                    tmp = TransportConfig.getTransportConfig(new SMTPConfig(), session, accountId);
                    tmp.setTransportProperties(createNewMailProperties());
                    cachedSmtpConfig = tmp;
                }
            }
        }
        return tmp;
    }

    private static final String ACK_TEXT =
        "Reporting-UA: OPEN-XCHANGE - WebMail\r\nFinal-Recipient: rfc822; #FROM#\r\n" + "Original-Message-ID: #MSG ID#\r\nDisposition: manual-action/MDN-sent-manually; displayed\r\n";

    private static final String CT_TEXT_PLAIN = "text/plain; charset=#CS#";

    private static final String CT_READ_ACK = "message/disposition-notification; name=MDNPart1.txt; charset=UTF-8";

    private static final String CD_READ_ACK = "attachment; filename=MDNPart1.txt";

    private static final String MULTI_SUBTYPE_REPORT = "report; report-type=disposition-notification";

    @Override
    public void sendReceiptAck(final MailMessage srcMail, final String fromAddr) throws OXException {
        if (null == srcMail) {
            return;
        }
        SMTPConfig smtpConfig = null;
        // There is an issue in the OSGi framework preventing the MailCap
        // from loading correctly. When getting the session here,
        // temporarily set the ClassLoader to the loader inside the bundle
        // that houses javax.mail. Reset at the end.
        // ClassLoader tcl = Thread.currentThread().getContextClassLoader();
        try {
            // Set the ClassLoader to the javax.mail bundle loader.
            // Thread.currentThread().setContextClassLoader(MailMessage.class.getClassLoader());
            // Go ahead...
            final InternetAddress dispNotification = srcMail.getDispositionNotification();
            if (dispNotification == null) {
                throw SMTPExceptionCode.MISSING_NOTIFICATION_HEADER.create(MessageHeaders.HDR_DISP_TO, Long.valueOf(srcMail.getMailId()));
            }
            final SMTPMessage smtpMessage = new SMTPMessage(getSMTPSession());
            final String userMail = UserStorage.getInstance().getUser(session.getUserId(), ctx).getMail();
            /*
             * Set from
             */
            final String from;
            if (fromAddr == null) {
                if ((usm.getSendAddr() == null) && (userMail == null)) {
                    throw SMTPExceptionCode.NO_SEND_ADDRESS_FOUND.create();
                }
                from = usm.getSendAddr() == null ? userMail : usm.getSendAddr();
            } else {
                from = fromAddr;
            }
            smtpMessage.addFrom(parseAddressList(from, false));
            /*
             * Set to
             */
            final Address[] recipients = new Address[] { dispNotification };
            processAddressHeader(smtpMessage);
            checkRecipients(recipients);
            smtpMessage.addRecipients(RecipientType.TO, recipients);
            /*
             * Set header
             */
            smtpMessage.setHeader(MessageHeaders.HDR_X_PRIORITY, "3 (normal)");
            smtpMessage.setHeader(MessageHeaders.HDR_IMPORTANCE, "Medium");
            /*
             * Subject
             */
            final Locale locale = UserStorage.getInstance().getUser(session.getUserId(), ctx).getLocale();
            final StringHelper strHelper = StringHelper.valueOf(locale);
            smtpMessage.setSubject(strHelper.getString(MailStrings.ACK_SUBJECT));
            /*
             * Sent date in UTC time
             */
            {
                final MailDateFormat mdf = MimeMessageUtility.getMailDateFormat(session);
                synchronized (mdf) {
                    smtpMessage.setHeader("Date", mdf.format(new Date()));
                }
            }
            /*
             * Set common headers
             */
            smtpConfig = getTransportConfig0();
            new SMTPMessageFiller(smtpConfig.getSMTPProperties(), session, ctx, usm).setAccountId(accountId).setCommonHeaders(smtpMessage);
            /*
             * Compose body
             */
            final String defaultMimeCS = MailProperties.getInstance().getDefaultMimeCharset();
            final ContentType ct = new ContentType(CT_TEXT_PLAIN.replaceFirst("#CS#", defaultMimeCS));
            final Multipart mixedMultipart = new MimeMultipart(MULTI_SUBTYPE_REPORT);
            /*
             * Define text content
             */
            final Date sentDate = srcMail.getSentDate();
            {
                final MimeBodyPart text = new MimeBodyPart();
                final String txt = performLineFolding(
                    strHelper.getString(MailStrings.ACK_NOTIFICATION_TEXT)
                    .replaceFirst("#DATE#", sentDate == null ? "" : quoteReplacement(DateFormat.getDateInstance(DateFormat.LONG, locale).format(sentDate)))
                    .replaceFirst("#RECIPIENT#", quoteReplacement(from)).replaceFirst("#SUBJECT#", quoteReplacement(srcMail.getSubject())), usm.getAutoLinebreak());
                MessageUtility.setText(txt, defaultMimeCS, text);
                // text.setText(txt,defaultMimeCS);
                text.setHeader(MessageHeaders.HDR_MIME_VERSION, "1.0");
                text.setHeader(MessageHeaders.HDR_CONTENT_TYPE, MimeMessageUtility.foldContentType(ct.toString()));
                mixedMultipart.addBodyPart(text);
            }
            /*
             * Define ack
             */
            ct.setContentType(CT_READ_ACK);
            {
                final MimeBodyPart ack = new MimeBodyPart();
                final String msgId = srcMail.getFirstHeader(MessageHeaders.HDR_MESSAGE_ID);
                final String txt = strHelper.getString(ACK_TEXT).replaceFirst("#FROM#", quoteReplacement(from)).replaceFirst(
                    "#MSG ID#",
                    quoteReplacement(msgId));
                MessageUtility.setText(txt, defaultMimeCS, ack);
                // ack.setText(txt,defaultMimeCS);
                ack.setHeader(MessageHeaders.HDR_MIME_VERSION, "1.0");
                ack.setHeader(MessageHeaders.HDR_CONTENT_TYPE, MimeMessageUtility.foldContentType(ct.toString()));
                ack.setHeader(MessageHeaders.HDR_CONTENT_DISPOSITION, CD_READ_ACK);
                mixedMultipart.addBodyPart(ack);
            }
            /*
             * Set message content
             */
            MessageUtility.setContent(mixedMultipart, smtpMessage);
            // smtpMessage.setContent(mixedMultipart);
            /*
             * Transport message
             */
            final long start = System.currentTimeMillis();
            final Transport transport = getSMTPSession().getTransport(SMTP);
            try {
                connectTransport(transport, smtpConfig);
                saveChangesSafe(smtpMessage);
                transport(smtpMessage, smtpMessage.getAllRecipients(), transport, smtpConfig);
                mailInterfaceMonitor.addUseTime(System.currentTimeMillis() - start);
            } catch (final javax.mail.AuthenticationFailedException e) {
                throw MimeMailExceptionCode.TRANSPORT_INVALID_CREDENTIALS.create(e, smtpConfig.getServer(), e.getMessage());
            } finally {
                transport.close();
            }
        } catch (final MessagingException e) {
            throw MimeMailException.handleMessagingException(e, smtpConfig, session);
        } finally {
            // Restore the ClassLoader
            // Thread.currentThread().setContextClassLoader(tcl);
        }
    }

    @Override
    public MailMessage sendRawMessage(final byte[] asciiBytes, final Address[] allRecipients) throws OXException {
        final SMTPConfig smtpConfig = getTransportConfig0();
        // There is an issue in the OSGi framework preventing the MailCap
        // from loading correctly. When getting the session here,
        // temporarily set the ClassLoader to the loader inside the bundle
        // that houses javax.mail. Reset at the end.
        // ClassLoader tcl = Thread.currentThread().getContextClassLoader();
        try {
            // Set the ClassLoader to the javax.mail bundle loader.
            // Thread.currentThread().setContextClassLoader(Address.class.getClassLoader());
            // Go ahead...
            final SMTPMessage smtpMessage = new SMTPMessage(getSMTPSession(), MimeHeaderNameChecker.sanitizeHeaderNames(asciiBytes));
            smtpMessage.removeHeader("x-original-headers");
            /*
             * Check recipients
             */
            final Address[] recipients = allRecipients == null ? smtpMessage.getAllRecipients() : allRecipients;
            processAddressHeader(smtpMessage);
            final boolean poisoned = checkRecipients(recipients);
            if (poisoned) {
                saveChangesSafe(smtpMessage);
            } else {
                try {
                    final long start = System.currentTimeMillis();
                    final Transport transport = getSMTPSession().getTransport(SMTP);
                    try {
                        connectTransport(transport, smtpConfig);
                        saveChangesSafe(smtpMessage);
                        transport(smtpMessage, recipients, transport, smtpConfig);
                        mailInterfaceMonitor.addUseTime(System.currentTimeMillis() - start);
                    } catch (final javax.mail.AuthenticationFailedException e) {
                        throw MimeMailExceptionCode.TRANSPORT_INVALID_CREDENTIALS.create(e, smtpConfig.getServer(), e.getMessage());
                    } finally {
                        transport.close();
                    }
                } catch (final MessagingException e) {
                    throw MimeMailException.handleMessagingException(e, smtpConfig, session);
                }
            }
            return MimeMessageConverter.convertMessage(smtpMessage);
        } catch (final MessagingException e) {
            throw MimeMailException.handleMessagingException(e, smtpConfig, session);
        } finally {
            // Restore the ClassLoader
            // Thread.currentThread().setContextClassLoader(tcl);
        }
    }

    @Override
    public void sendMimeMessage(MimeMessage mimeMessage, Address[] allRecipients) throws OXException {
        final SMTPConfig smtpConfig = getTransportConfig0();
        try {
            /*
             * Check recipients
             */
            final Address[] recipients = allRecipients == null ? mimeMessage.getAllRecipients() : allRecipients;
            processAddressHeader(mimeMessage);
            final boolean poisoned = checkRecipients(recipients);
            if (poisoned) {
                saveChangesSafe(mimeMessage);
            } else {
                try {
                    final long start = System.currentTimeMillis();
                    final Transport transport = getSMTPSession().getTransport(SMTP);
                    try {
                        connectTransport(transport, smtpConfig);
                        saveChangesSafe(mimeMessage);
                        transport(mimeMessage, recipients, transport, smtpConfig);
                        mailInterfaceMonitor.addUseTime(System.currentTimeMillis() - start);
                    } catch (final javax.mail.AuthenticationFailedException e) {
                        throw MimeMailExceptionCode.TRANSPORT_INVALID_CREDENTIALS.create(e, smtpConfig.getServer(), e.getMessage());
                    } finally {
                        transport.close();
                    }
                } catch (final MessagingException e) {
                    throw MimeMailException.handleMessagingException(e, smtpConfig, session);
                }
            }
        } catch (final MessagingException e) {
            throw MimeMailException.handleMessagingException(e, smtpConfig, session);
        }
    }

    /**
     * Gets the connected SMTP transport.
     *
     * @return The connected SMTP transport
     * @throws OXException If an error occurs
     * @throws MessagingException If a messaging error occurs
     */
    public com.sun.mail.smtp.SMTPTransport getSmtpTransport() throws OXException, MessagingException {
        final com.sun.mail.smtp.SMTPTransport transport = (com.sun.mail.smtp.SMTPTransport) getSMTPSession().getTransport(SMTP);
        connectTransport(transport, getTransportConfig0());
        return transport;
    }

    @Override
    public MailMessage sendMailMessage(final ComposedMailMessage composedMail, final ComposeType sendType, final Address[] allRecipients) throws OXException {
        return sendMailMessage(composedMail, sendType, allRecipients, null);
    }

    @Override
    public MailMessage sendMailMessage(final ComposedMailMessage composedMail, final ComposeType sendType, final Address[] allRecipients, final MtaStatusInfo mtaStatusInfo) throws OXException {
        final SMTPConfig smtpConfig = getTransportConfig0();
        // There is an issue in the OSGi framework preventing the MailCap
        // from loading correctly. When getting the session here,
        // temporarily set the ClassLoader to the loader inside the bundle
        // that houses javax.mail. Reset at the end.
        // ClassLoader tcl = Thread.currentThread().getContextClassLoader();
        try {
            // Set the ClassLoader to the javax.mail bundle loader.
            // Thread.currentThread().setContextClassLoader(ComposedMailMessage.class.getClassLoader());
            // Go ahead...
            /*
             * Message content available?
             */
            MimeMessage mimeMessage = null;
            if (composedMail instanceof ContentAware) {
                try {
                    final Object content = composedMail.getContent();
                    if (content instanceof MimeMessage) {
                        mimeMessage = (MimeMessage) content;
                    }
                } catch (final Exception e) {
                    // Ignore
                }
            }
            /*
             * Proceed
             */
            if (null != mimeMessage) {
                mimeMessage.removeHeader("x-original-headers");
                /*
                 * Check for reply
                 */
                {
                    final MailPath msgref;
                    if (ComposeType.REPLY.equals(sendType) && ((msgref = composedMail.getMsgref()) != null)) {
                        MailAccess<?, ?> access = null;
                        try {
                            access = MailAccess.getInstance(session, msgref.getAccountId());
                            access.connect();
                            MimeMessageFiller.setReplyHeaders(access.getMessageStorage().getMessage(msgref.getFolder(), msgref.getMailID(), false), mimeMessage);
                        } finally {
                            if (null != access) {
                                access.close(true);
                            }
                        }
                    }
                }
                /*
                 * Set common headers
                 */
                final SMTPMessageFiller smtpFiller = new SMTPMessageFiller(smtpConfig.getSMTPProperties(), session, ctx, usm);
                smtpFiller.setAccountId(accountId);
                smtpFiller.setCommonHeaders(mimeMessage);
                /*
                 * Check recipients
                 */
                final Address[] recipients = allRecipients == null ? mimeMessage.getAllRecipients() : allRecipients;
                processAddressHeader(mimeMessage);
                final boolean poisoned = checkRecipients(recipients);
                if (poisoned) {
                    saveChangesSafe(mimeMessage);
                } else {
                    try {
                        final long start = System.currentTimeMillis();
                        final Transport transport = getSMTPSession().getTransport(SMTP);
                        try {
                            connectTransport(transport, smtpConfig);
                            saveChangesSafe(mimeMessage);
                            transport(mimeMessage, recipients, transport, smtpConfig, mtaStatusInfo);
                            mailInterfaceMonitor.addUseTime(System.currentTimeMillis() - start);
                        } catch (final javax.mail.AuthenticationFailedException e) {
                            throw MimeMailExceptionCode.TRANSPORT_INVALID_CREDENTIALS.create(e, smtpConfig.getServer(), e.getMessage());
                        } finally {
                            transport.close();
                        }
                    } catch (final MessagingException e) {
                        throw MimeMailException.handleMessagingException(e, smtpConfig, session);
                    }
                }
                return MimeMessageConverter.convertMessage(mimeMessage);
            }
            /*
             * Fill from scratch
             */
            final SMTPMessage smtpMessage = new SMTPMessage(getSMTPSession());
            /*
             * Fill message dependent on send type
             */
            final SMTPMessageFiller smtpFiller = new SMTPMessageFiller(smtpConfig.getSMTPProperties(), session, ctx, usm);
            smtpFiller.setAccountId(accountId);
            composedMail.setFiller(smtpFiller);
            try {
                smtpFiller.fillMail(composedMail, smtpMessage, sendType);
                /*
                 * Check recipients
                 */
                final Address[] recipients;
                if (allRecipients == null) {
                    if (composedMail.hasRecipients()) {
                        recipients = composedMail.getRecipients();
                    } else {
                        recipients = smtpMessage.getAllRecipients();
                    }
                } else {
                    recipients = allRecipients;
                }
                final boolean poisoned = checkRecipients(recipients);
                smtpFiller.setSendHeaders(composedMail, smtpMessage);
                processAddressHeader(smtpMessage);
                /*
                 * Drop special "x-original-headers" header
                 */
                smtpMessage.removeHeader("x-original-headers");
                if (poisoned) {
                    saveChangesSafe(smtpMessage);
                } else {
                    final long start = System.currentTimeMillis();
                    final Transport transport = getSMTPSession().getTransport(SMTP);
                    try {
                        connectTransport(transport, smtpConfig);
                        /*
                         * Save changes
                         */
                        saveChangesSafe(smtpMessage);
                        /*
                         * TODO: Do encryption here
                         */
                        transport(smtpMessage, recipients, transport, smtpConfig, mtaStatusInfo);
                        mailInterfaceMonitor.addUseTime(System.currentTimeMillis() - start);
                    } catch (final javax.mail.AuthenticationFailedException e) {
                        throw MimeMailExceptionCode.TRANSPORT_INVALID_CREDENTIALS.create(e, smtpConfig.getServer(), e.getMessage());
                    }  finally {
                        transport.close();
                    }
                }
            } finally {
                invokeLater(new MailCleanerTask(composedMail));
            }
            return MimeMessageConverter.convertMessage(smtpMessage);
        } catch (final MessagingException e) {
            throw MimeMailException.handleMessagingException(e, smtpConfig, session);
        } catch (final IOException e) {
            throw SMTPExceptionCode.IO_ERROR.create(e, e.getMessage());
        }
        //finally {
        // Restore the ClassLoader
        // Thread.currentThread().setContextClassLoader(tcl);
        //}
    }

    private void connectTransport(final Transport transport, final SMTPConfig smtpConfig) throws OXException, MessagingException {
        final String server = IDNA.toASCII(smtpConfig.getServer());
        final int port = smtpConfig.getPort();
        try {
            if (smtpConfig.getSMTPProperties().isSmtpAuth()) {
                final String encodedPassword = encodePassword(smtpConfig.getPassword());
                if (isKerberosAuth()) {
                    try {
                        Subject.doAs(kerberosSubject, new SaslSmtpLoginAction(transport, server, port, smtpConfig.getLogin(), encodedPassword));
                    } catch (final PrivilegedActionException e) {
                        handlePrivilegedActionException(e);
                    }
                } else {
                    final String login = smtpConfig.getLogin();
                    transport.connect(server, port, null == login ? "" : login, null == encodedPassword ? "" : encodedPassword);
                }
            } else {
                transport.connect(server, port, null, null);
            }
        } catch (final MessagingException e) {
            if (e.getNextException() instanceof javax.net.ssl.SSLHandshakeException) {
                throw SMTPExceptionCode.SECURE_CONNECTION_NOT_POSSIBLE.create(e.getNextException(), server, smtpConfig.getLogin());
            }
            throw e;
        }
    }

    private void logMessageTransport(final MimeMessage smtpMessage, final SMTPConfig smtpConfig) throws OXException, MessagingException {
        if (getTransportConfig0().getSMTPProperties().isLogTransport()) {
            LogProperties.putSessionProperties(session);
            LOG.info("Sent \"{}\" for login \"{}\" using SMTP server \"{}\" on port {}.", smtpMessage.getMessageID(), smtpConfig.getLogin(), smtpConfig.getServer(), Integer.valueOf(smtpConfig.getPort()));
        }
    }

    private void transport(final MimeMessage smtpMessage, final Address[] recipients, final Transport transport, final SMTPConfig smtpConfig) throws OXException {
        transport(smtpMessage, recipients, transport, smtpConfig, null);
    }

    private void transport(final MimeMessage smtpMessage, final Address[] recipients, final Transport transport, final SMTPConfig smtpConfig, final MtaStatusInfo mtaInfo) throws OXException {
        // Prepare addresses
        prepareAddresses(recipients);

        // Register transport listener to fill addresses to status info
        if (null != mtaInfo) {
            transport.addTransportListener(new AddressAddingTransportListener(mtaInfo));
        }

        // Try to send the message
        try {
            transport.sendMessage(smtpMessage, recipients);
            logMessageTransport(smtpMessage, smtpConfig);
        } catch (SMTPSendFailedException sendFailed) {
            OXException oxe = MimeMailException.handleMessagingException(sendFailed, smtpConfig, session);
            if (null != mtaInfo) {
                mtaInfo.setReturnCode(sendFailed.getReturnCode());
                oxe.setArgument("mta_info", mtaInfo);
            }
            throw oxe;
        } catch (final MessagingException e) {
            if (e.getNextException() instanceof javax.activation.UnsupportedDataTypeException) {
                // Check for "no object DCH for MIME type xxxxx/yyyy"
                final String message = e.getNextException().getMessage();
                if (toLowerCase(message).indexOf("no object dch") >= 0) {
                    // Not able to recover from JAF's "no object DCH for MIME type xxxxx/yyyy" error
                    // Perform the alternative transport with custom JAF DataHandler
                    LOG.warn(message.replaceFirst("[dD][cC][hH]", Matcher.quoteReplacement("javax.activation.DataContentHandler")));
                    transportAlt(smtpMessage, recipients, transport, smtpConfig);
                    return;
                }
            } else if (e.getNextException() instanceof IOException) {
                if (e.getNextException().getMessage().equals("Maximum message size is exceeded.")) {
                    throw MailExceptionCode.MAX_MESSAGE_SIZE_EXCEEDED.create(getSize(getMaxMailSize(), 2, false, true));
                }
            }
            throw MimeMailException.handleMessagingException(e, smtpConfig, session);
        }
    }

    private void transportAlt(final MimeMessage smtpMessage, final Address[] recipients, final Transport transport, final SMTPConfig smtpConfig) throws OXException {
        try {
            final MimeMessageDataSource dataSource = new MimeMessageDataSource(smtpMessage, smtpConfig, session);
            smtpMessage.setDataHandler(new DataHandler(dataSource));
            if (!transport.isConnected()) {
                connectTransport(transport, smtpConfig);
            }
            transport.sendMessage(smtpMessage, recipients);
            logMessageTransport(smtpMessage, smtpConfig);
            invokeLater(new Runnable() {

                @Override
                public void run() {
                    try {
                        dataSource.cleanUp();
                    } catch (final Exception e) {
                        // Ignore
                    }
                }
            });
        } catch (final MessagingException me) {
            throw MimeMailException.handleMessagingException(me, smtpConfig, session);
        }
    }

    private String encodePassword(final String password) throws OXException {
        String tmpPass = password;
        if (tmpPass != null) {
            try {
                tmpPass = new String(password.getBytes(Charsets.forName(getTransportConfig0().getSMTPProperties().getSmtpAuthEnc())), Charsets.ISO_8859_1);
            } catch (final UnsupportedCharsetException e) {
                LOG.error("Unsupported encoding in a message detected and monitored", e);
                mailInterfaceMonitor.addUnsupportedEncodingExceptions(e.getMessage());
            }
        }
        return tmpPass;
    }

    private void prepareAddresses(final Address[] addresses) {
        final int length = addresses.length;
        final StringBuilder tmp = new StringBuilder(32);
        for (int i = 0; i < length; i++) {
            final InternetAddress address = (InternetAddress) addresses[i];
            final String sAddress = address.getAddress();
            if (MsisdnCheck.checkMsisdn(sAddress)) {
                final int pos = sAddress.indexOf('/');
                if (pos < 0) {
                    tmp.setLength(0);
                    address.setAddress(tmp.append(sAddress).append("/TYPE=PLMN").toString());
                }
            }
        }
    }

    @Override
    protected void shutdown() {
        SMTPSessionProperties.resetDefaultSessionProperties();
        SMTPCapabilityCache.tearDown();
    }

    @Override
    protected void startup() {
        SMTPCapabilityCache.init();
    }

    private static void processAddressHeader(final MimeMessage mimeMessage) throws OXException, MessagingException {
        {
            final String str = mimeMessage.getHeader("From", null);
            if (!com.openexchange.java.Strings.isEmpty(str)) {
                final InternetAddress[] addresses = QuotedInternetAddress.parse(str, false);
                checkRecipients(addresses);
                mimeMessage.setFrom(addresses[0]);
            }
        }
        {
            final String str = mimeMessage.getHeader("Sender", null);
            if (!com.openexchange.java.Strings.isEmpty(str)) {
                final InternetAddress[] addresses = QuotedInternetAddress.parse(str, false);
                checkRecipients(addresses);
                mimeMessage.setSender(addresses[0]);
            }
        }
        {
            final String str = mimeMessage.getHeader("To", null);
            if (!com.openexchange.java.Strings.isEmpty(str)) {
                final InternetAddress[] addresses = QuotedInternetAddress.parse(str, false);
                checkRecipients(addresses);
                mimeMessage.setRecipients(RecipientType.TO, addresses);
            }
        }
        {
            final String str = mimeMessage.getHeader("Cc", null);
            if (!com.openexchange.java.Strings.isEmpty(str)) {
                final InternetAddress[] addresses = QuotedInternetAddress.parse(str, false);
                checkRecipients(addresses);
                mimeMessage.setRecipients(RecipientType.CC, addresses);
            }
        }
        {
            final String str = mimeMessage.getHeader("Bcc", null);
            if (!com.openexchange.java.Strings.isEmpty(str)) {
                final InternetAddress[] addresses = QuotedInternetAddress.parse(str, false);
                checkRecipients(addresses);
                mimeMessage.setRecipients(RecipientType.BCC, addresses);
            }
        }
        {
            final String str = mimeMessage.getHeader("Reply-To", null);
            if (!com.openexchange.java.Strings.isEmpty(str)) {
                final InternetAddress[] addresses = QuotedInternetAddress.parse(str, false);
                checkRecipients(addresses);
                mimeMessage.setReplyTo(addresses);
            }
        }
        {
            final String str = mimeMessage.getHeader("Disposition-Notification-To", null);
            if (!com.openexchange.java.Strings.isEmpty(str)) {
                final InternetAddress[] addresses = QuotedInternetAddress.parse(str, false);
                checkRecipients(addresses);
                mimeMessage.setHeader("Disposition-Notification-To", addresses[0].toString());
            }
        }
    }

    private static final boolean isPoisoned(final Address[] recipients) {
        if ((recipients == null) || (recipients.length == 0)) {
            return false;
        }
        for (final Address address : recipients) {
            if (MimeMessageUtility.POISON_ADDRESS == address) {
                return true;
            }
        }
        return false;
    }

    private static boolean checkRecipients(final Address[] recipients) throws OXException {
        if ((recipients == null) || (recipients.length == 0)) {
            throw SMTPExceptionCode.MISSING_RECIPIENTS.create();
        }
        Boolean poisoned = null;
        final ConfigurationService service = Services.getService(ConfigurationService.class);
        if (null != service) {
            final Filter filter = service.getFilterFromProperty("com.openexchange.mail.transport.redirectWhitelist");
            if (null != filter) {
                for (final Address address : recipients) {
                    if (MimeMessageUtility.POISON_ADDRESS == address) {
                        poisoned = Boolean.TRUE;
                    } else {
                        final InternetAddress internetAddress = (InternetAddress) address;
                        if (!filter.accepts(internetAddress.getAddress())) {
                            throw SMTPExceptionCode.RECIPIENT_NOT_ALLOWED.create(internetAddress.toUnicodeString());
                        }
                    }
                }
            }
        }
        if (MailProperties.getInstance().isSupportMsisdnAddresses()) {
            InternetAddress internetAddress;
            for (final Address address : recipients) {
                if (MimeMessageUtility.POISON_ADDRESS == address) {
                    poisoned = Boolean.TRUE;
                } else {
                    internetAddress = (InternetAddress) address;
                    final String sAddress = internetAddress.getAddress();
                    if (MsisdnCheck.checkMsisdn(sAddress)) {
                        if (sAddress.indexOf('/') < 0) {
                            // Detected a MSISDN address that misses "/TYPE=" appendix necessary for the MTA
                            internetAddress.setAddress(sAddress + "/TYPE=PLMN");
                        }
                        try {
                            internetAddress.setPersonal("", "US-ASCII");
                        } catch (final UnsupportedEncodingException e) {
                            // Ignore as personal is cleared
                        }
                    }
                }
            }
        }
        return null == poisoned ? isPoisoned(recipients) : poisoned.booleanValue();
    }

    private static final class MailCleanerTask implements Runnable {

        private final ComposedMailMessage composedMail;

        public MailCleanerTask(final ComposedMailMessage composedMail) {
            super();
            this.composedMail = composedMail;
        }

        @Override
        public void run() {
            composedMail.cleanUp();
        }

    }

    @Override
    public void ping() throws OXException {
        // Connect to SMTP server
        final Transport transport;
        try {
            SMTPConfig smtpConfig = getTransportConfig0();
            transport = getSMTPSession(smtpConfig, smtpConfig.isRequireTls() || MailProperties.getInstance().isEnforceSecureConnection()).getTransport(SMTP);
        } catch (final NoSuchProviderException e) {
            throw MimeMailException.handleMessagingException(e);
        }
        boolean close = false;
        final SMTPConfig config = getTransportConfig0();
        try {
            connectTransport(transport, config);
            close = true;
        } catch (final javax.mail.AuthenticationFailedException e) {
            throw MimeMailExceptionCode.TRANSPORT_INVALID_CREDENTIALS.create(e, config.getServer(), e.getMessage());
        } catch (final MessagingException e) {
            throw MimeMailException.handleMessagingException(e, config, session);
        } finally {
            if (close) {
                try {
                    transport.close();
                } catch (final MessagingException e) {
                    LOG.error("Closing SMTP transport failed.", e);
                }
            }
        }
    }

    @Override
    protected ITransportProperties createNewMailProperties() throws OXException {
        try {
            final MailAccountStorageService storageService = Services.getService(MailAccountStorageService.class);
            return new MailAccountSMTPProperties(storageService.getMailAccount(accountId, session.getUserId(), session.getContextId()));
        } catch (final OXException e) {
            throw e;
        }
    }

    private void saveChangesSafe(final MimeMessage mimeMessage) throws OXException {
        final HostnameService hostnameService = Services.getService(HostnameService.class);
        String hostName;
        if (null == hostnameService) {
            hostName = getHostName();
        } else {
            hostName = hostnameService.getHostname(session.getUserId(), session.getContextId());
        }
        if (null == hostName) {
            hostName = getHostName();
        }
        MimeMessageConverter.saveChanges(mimeMessage, hostName);
        // Check whether to remove MIME-Version headers from sub-parts
        if (TransportProperties.getInstance().isRemoveMimeVersionInSubParts()) {
            /*-
             *  Note that the MIME-Version header field is required at the top level
             *  of a message.  It is not required for each body part of a multipart
             *  entity.  It is required for the embedded headers of a body of type
             *  "message/rfc822" or "message/partial" if and only if the embedded
             *  message is itself claimed to be MIME-conformant.
             */
            try {
                checkMimeVersionHeader(mimeMessage);
            } catch (final Exception e) {
                LOG.warn("Could not check for proper usage of \"MIME-Version\" header according to RFC2045.", e);
            }
        }
    }

    private void checkMimeVersionHeader(final MimeMessage mimeMessage) throws MessagingException, IOException {
        final String header = mimeMessage.getHeader("Content-Type", null);
        if (null != header && toLowerCase(header).startsWith("multipart/")) {
            final Multipart multipart = (Multipart) mimeMessage.getContent();
            final int count = multipart.getCount();
            for (int i = 0; i < count; i++) {
                checkMimeVersionHeader(multipart.getBodyPart(i));
            }
        }
    }

    private void checkMimeVersionHeader(final Part part) throws MessagingException, IOException {
        final String[] header = part.getHeader("Content-Type");
        if (null != header && header.length > 0 && null != header[0]) {
            final String cts = toLowerCase(header[0]);
            if (cts.startsWith("multipart/")) {
                final Multipart multipart = (Multipart) part.getContent();
                final int count = multipart.getCount();
                for (int i = 0; i < count; i++) {
                    checkMimeVersionHeader(multipart.getBodyPart(i));
                }
            } else if (cts.startsWith("message/rfc822") || cts.startsWith("message/partial")) {
                part.setHeader("MIME-Version", "1.0");
                Object content;
                try {
                    content = part.getContent();
                } catch (Exception e) {
                    content = null;
                }
                if (content instanceof MimeMessage) {
                    checkMimeVersionHeader((MimeMessage) content);
                }
            } else {
                part.removeHeader("MIME-Version");
            }
        } else {
            part.removeHeader("MIME-Version");
        }
    }

    private static String getHostName() {
        final String serverName = LogProperties.getLogProperty(LogProperties.Name.AJP_SERVER_NAME);
        if (null == serverName) {
            return getStaticHostName();
        }
        return serverName;
    }

    private static String getStaticHostName() {
        final UnknownHostException warning = warnSpam;
        if (warning != null) {
            LOG.error("Can't resolve my own hostname, using 'localhost' instead, which is certainly not what you want!", warning);
        }
        return staticHostName;
    }

    private static String quoteReplacement(final String str) {
        return com.openexchange.java.Strings.isEmpty(str) ? "" : quoteReplacement0(str);
    }

    private static String quoteReplacement0(final String s) {
        if ((s.indexOf('\\') < 0) && (s.indexOf('$') < 0)) {
            return s;
        }
        final int length = s.length();
        final StringBuilder sb = new StringBuilder(length << 1);
        for (int i = 0; i < length; i++) {
            final char c = s.charAt(i);
            if (c == '\\') {
                sb.append('\\');
                sb.append('\\');
            } else if (c == '$') {
                sb.append('\\');
                sb.append('$');
            } else {
                sb.append(c);
            }
        }
        return sb.toString();
    }

    /** ASCII-wise to lower-case */
    private static String toLowerCase(final CharSequence chars) {
        if (null == chars) {
            return null;
        }
        final int length = chars.length();
        final StringBuilder builder = new StringBuilder(length);
        for (int i = 0; i < length; i++) {
            final char c = chars.charAt(i);
            builder.append((c >= 'A') && (c <= 'Z') ? (char) (c ^ 0x20) : c);
        }
        return builder.toString();
    }

    private static final class AddressAddingTransportListener implements TransportListener {

        private final MtaStatusInfo statusInfo;

        AddressAddingTransportListener(MtaStatusInfo statusInfo) {
            super();
            this.statusInfo = statusInfo;
        }

        @Override
        public void messagePartiallyDelivered(TransportEvent e) {
            fillAddressesFromEvent(e);
        }

        @Override
        public void messageNotDelivered(TransportEvent e) {
            fillAddressesFromEvent(e);

        }

        @Override
        public void messageDelivered(TransportEvent e) {
            fillAddressesFromEvent(e);

        }

        private void fillAddressesFromEvent(TransportEvent e) {
            javax.mail.Address[] arr = e.getInvalidAddresses();
            if (null != arr) {
                statusInfo.getInvalidAddresses().addAll(Arrays.asList(arr));
            }

            arr = e.getValidUnsentAddresses();
            if (null != arr) {
                statusInfo.getUnsentAddresses().addAll(Arrays.asList(arr));
            }

            arr = e.getValidSentAddresses();
            if (null != arr) {
                statusInfo.getSentAddresses().addAll(Arrays.asList(arr));
            }
        }
    }

}
